import os
import argparse
import ConfigParser
import sys


class EnvDefault(argparse.Action):
    def __init__(self, envvar, required=False, default=None, **kwargs):
        if not default and envvar:
            if envvar in os.environ:
                default = os.environ[envvar]
        if required and default:
            required = False
        super(EnvDefault, self).__init__(default=default, required=required, **kwargs)

    def __call__(self, parser, namespace, values, option_string=None):
        setattr(namespace, self.dest, values)


class AwsArgParser(argparse.ArgumentParser):
    def __init__(self, *args, **kwargs):
        argparse.ArgumentParser.__init__(self, *args, **kwargs)

        self._add_conf_argument_group()
        self._add_aws_argument_group()

    def _add_conf_default(self, conf_file, profile):
        if not profile:
            section = 'default'
        else:
            section = 'profile ' + profile

        config = ConfigParser.SafeConfigParser()
        config.read(conf_file)

        result = dict(config.items(section))
        result['conf_file'] = conf_file
        result['profile'] = profile

        self.set_defaults(**result)

    def _add_conf_argument_group(self):
        group = self.add_argument_group('configuration options')
        group.add_argument("-c", "--conf-file", action=EnvDefault, envvar='AWS_CONFIG_FILE',
                           help="specify config files", metavar="FILE")
        group.add_argument('-p', '--profile', action=EnvDefault, envvar='AWS_DEFAULT_PROFILE',
                           help='Default profile name')

    def _add_aws_argument_group(self):
        group = self.add_argument_group('aws options')
        group.add_argument('-r', '--region', action=EnvDefault, envvar='AWS_DEFAULT_REGION',
                           help='Default AWS Region')
        group.add_argument('--aws-access-key-id', action=EnvDefault, envvar='AWS_ACCESS_KEY_ID',
                           help='AWS Access Key', metavar='KEY')
        group.add_argument('--aws-secret-access-key', action=EnvDefault,
                           envvar='AWS_SECRET_ACCESS_KEY', help='AWS Secret Key', metavar='KEY')
        group.add_argument('--aws-security-token', action=EnvDefault, envvar='AWS_SECURITY_TOKEN',
                           help='AWS Token (temp credentials)', metavar='TOKEN')

    def parse_known_args(self, args=sys.argv[1:], namespace=None):
        options, argv = argparse.ArgumentParser.parse_known_args(self, args, namespace)
        if options.conf_file:
            self._add_conf_default(options.conf_file, options.profile)
            return argparse.ArgumentParser.parse_known_args(self, args, namespace)
        return options, argv


