import sys
import json
import time
import subprocess
import logging
import copy
from grandfatherson import dates_to_delete, MONDAY
from dateutil.parser import parse
from datetime import date, datetime
from opsaws.awscli import AwsCli

logger = logging.getLogger(__name__)


def tagdict2list(tag_dict):
    tags = list()
    for tag in tag_dict:
        tags.append({'Key': tag, 'Value': tag_dict[tag]})
    return tags


def taglist2dict(tags):
    tag_dict = dict()
    for tag in tags:
        tag_dict[tag['Key']] = tag['Value']
    return tag_dict


class EBSSnapshot(AwsCli):
    def describe_volumes(self, filters=None):
        command = 'aws ec2 describe-volumes'
        if filters:
            for f in filters:
                command += ' --filter ' + "'" + json.dumps(f) + "'"
        return self.run(command)['Volumes']

    def describe_snapshots(self, snapshot_ids=None, owner_id='self', filters=None):
        command = 'aws ec2 describe-snapshots'

        if owner_id:
            command += ' --owner-id ' + owner_id
        if filters:
            for f in filters:
                command += ' --filter ' + "'" + json.dumps(f) + "'"
        if snapshot_ids:
            command += ' --snapshot-ids'
            for snapshot_id in snapshot_ids:
                command += ' ' + snapshot_id

        return self.run(command)['Snapshots']

    def add_tag_to_snapshot(self, snapshot_id, tags, dry_run=False):
        logger.info('Add tag to snapshot: {0} tag: {1}'.format(snapshot_id, tags))
        command = 'aws ec2 create-tags --resources ' + snapshot_id
        if dry_run:
            command += ' --dry-run'

        command += ' --tags ' + "'" + json.dumps(tags) + "'"
        return self.run(command)

    def create_snapshot(self, volume_id, instance_id=None, description=None, dry_run=False):
        logger.info(
            'Create snapshot for volume: {0} instance_id: {1}'.format(volume_id, instance_id))
        command = 'aws ec2 create-snapshot --volume-id ' + volume_id
        if dry_run:
            command += ' --dry-run'
        if description:
            command += ' --description "' + description + '"'
        snapshot = self.run(command)

        # fake data for dry_run
        if dry_run:
            snapshot = {'SnapshotId': 'snap-12345678'}

        # snapshot tag dictionary
        snapshot_tag_dict = dict()
        snapshot_tag_dict['TakenFromVolumeID'] = volume_id
        snapshot_tag_dict['TakenAtUnixTime'] = str(time.time())
        snapshot_tag_dict['TakenDate'] = date.today().isoformat()

        # add instance_id information if available
        if instance_id:
            command = 'aws ec2 describe-instances --instance-ids ' + instance_id
            instance = self.run(command)['Reservations'][0]['Instances'][0]
            instance_tag_dict = taglist2dict(instance['Tags'])

            if 'Tenant' in instance_tag_dict:
                snapshot_tag_dict['Tenant'] = instance_tag_dict['Tenant']
            snapshot_tag_dict['TakenWhileAttachedToID'] = instance_id
            snapshot_tag_dict['TakenWhileAttachedToName'] = instance_tag_dict['Name']

        # add tag
        self.add_tag_to_snapshot(snapshot['SnapshotId'], tagdict2list(snapshot_tag_dict),
                                 dry_run=dry_run)

        return snapshot

    def delete_snapshot(self, snapshot_id, dry_run=False):
        logger.info('Delete snapshot: {0}'.format(snapshot_id))
        command = 'aws ec2 delete-snapshot --snapshot-id ' + snapshot_id
        if dry_run:
            command += ' --dry-run'
            self.run(command)
        else:
            result = self.run(command)
            if result['return'] != 'true':
                raise subprocess.CalledProcessError(-1, command)

    def create_snapshot_all_volumes(self, filters=None, dry_run=False, in_use_only=True):

        # clone it , we need to modify the array
        filters = copy.deepcopy(filters)

        if in_use_only:
            if not filters:
                filters = [[{"Values": ["in-use"], "Name": "status"}]]
            else:
                for f in filters:
                    f.append({"Values": ["in-use"], "Name": "status"})

        volumes = self.describe_volumes(filters)

        snapshots = list()
        for volume in volumes:
            description = volume['VolumeId'] + '-' + date.today().isoformat()
            if 'Attachments' not in volume:
                snapshot = self.create_snapshot(volume['VolumeId'], description=description,
                                                dry_run=dry_run)
            else:
                snapshot = self.create_snapshot(volume['VolumeId'],
                                                volume['Attachments'][0]['InstanceId'],
                                                description=description,
                                                dry_run=dry_run)
            logger.info('Snapshot_id = ' + snapshot['SnapshotId'])
            snapshots.append(snapshot)

        return snapshots

    def rotate_snapshot(self, retention, filters=None, dry_run=False):

        # parse the retention rule
        days, weeks, months = map(int, retention.split(':'))

        # get all snapshot information
        snapshots = self.describe_snapshots(filters=filters)

        # get all snapshot date and calculate the delete set
        all_dt = [parse(snapshot['StartTime']).date() for snapshot in snapshots]
        delete_set = dates_to_delete(all_dt, days=days, weeks=weeks, months=months, firstweekday=MONDAY)

        # delete the snapshot if it is in the delete set
        for snapshot in snapshots:
            if parse(snapshot['StartTime']).date() in delete_set:
                self.delete_snapshot(snapshot['SnapshotId'], dry_run=dry_run)

