import sys
import time
import logging
from opsaws.awsargparse import AwsArgParser
from opsaws.ec2.snapshot import EBSSnapshot

logger = logging.getLogger(__name__)


def add_backup_argument_group(parser):
    group = parser.add_argument_group('backup options')
    group.add_argument('--retention', default='7:4:1',
                       help="days:weeks:months worth of backups to keep")
    group.add_argument('--tenants', default=None, action="append")
    group.add_argument('--dry-run', action="store_true",
                       help="Don't execute snapshot run or delete run")
    group.add_argument('--time-limit', type=int, default=3600)


def parse_options():
    parser = AwsArgParser()
    add_backup_argument_group(parser)
    parser.add_argument('-d', '--debug', action='store_true', help='Turn on debug info',
                        default=False)

    options = parser.parse_args()

    if options.debug:
        logging.basicConfig(level=logging.DEBUG, format='%(asctime)s %(levelname)s %(message)s')
    else:
        logging.basicConfig(level=logging.INFO, format='%(asctime)s %(levelname)s %(message)s')

    logging.debug(options)

    return options


def validate_snapshots(snapshots, state='completed'):
    for snapshot in snapshots:
        if snapshot['State'] != state:
            return False
    return True


def main():
    options = parse_options()

    if options.tenants:
        filters = [[{"Name": "tag:Tenant", "Values": options.tenants}]]
    else:
        filters = None

    # setup ebs_backup AWS connection information
    ebs_backup = EBSSnapshot(options.conf_file, options.profile, options.region)

    # create all snapshot
    snapshots = ebs_backup.create_snapshot_all_volumes(filters = filters, dry_run=options.dry_run)

    # delete old retention
    if options.retention:
        ebs_backup.rotate_snapshot(options.retention, filters = filters, dry_run=options.dry_run)

    # check if all snapshot ready
    start_time = time.time()
    snapshot_ids = list()

    if not options.dry_run:
        while True:
            time.sleep(10)
            snapshot_ids = [s['SnapshotId'] for s in snapshots if s['State'] != 'completed']
            logger.info('Checking snapshot: {0}'.format(snapshot_ids))
            snapshots = ebs_backup.describe_snapshots(snapshot_ids)
            if validate_snapshots(snapshots):
                snapshots = None
                break
            elif time.time() - start_time > options.time_limit:
                snapshot_ids = [s['SnapshotId'] for s in snapshots if s['State'] != 'completed']
                break
    else:
        snapshots = None

    if not snapshots:
        msg = 'OK: All snapshot created'
        logger.info(msg)
        print msg
        return 0
    else:
        msg = 'WARNING: Some snapshot is not ready. {0}'.format(snapshot_ids)
        logger.info(msg)
        print >> sys.stderr, msg
        return 1

if __name__ == "__main__":
    sys.exit(main())