#!/usr/bin/env python
"""org archive to html table converter"""

import os
import datetime
import itertools

from .utils.date import minutestr, total_minutes


def rootname_from_archive_olpath(node):
    """
    Find rootname from ARCHIVE_OLPATH property.
    Return None if not found.
    """
    olpath = node.get_property('ARCHIVE_OLPATH')
    if olpath:
        olpathlist = olpath.split('/', 1)
        if len(olpathlist) > 1:
            (rootname, dummy) = olpathlist
        else:
            rootname = olpath
        return rootname
    return None


def find_rootname(node):
    """
    Find rootname given node
    """
    rootname = rootname_from_archive_olpath(node)
    if not rootname:
        n = node
        p = node.parent
        while not p.is_root():
            n = p
            p = p.parent
        # n is root node
        rootname = rootname_from_archive_olpath(n) or n.heading
    return rootname


def key_row_from_node(node):
    """
    Return three tuple (key, row) whose elemens are
    key object for sorting table and dictionary which has following
    keywords: heading, closed, scheduled, effort, clocksum, rootname.
    """
    heading = node.heading
    # find rootname
    rootname = find_rootname(node)
    if heading == rootname:
        rootname = ""
    # calc clocksum if CLOCK exists
    clocksum = ''
    clocklist = node.clock
    if clocklist:
        clocksum = sum([total_minutes(c.duration) for c in clocklist])
    closed = node.closed
    scheduled = node.scheduled
    effort = node.get_property('Effort')
    row = dict(
        heading=heading,
        closed=closed and closed.start.strftime('%a %d %b %H:%M'),
        scheduled=scheduled and scheduled.start.strftime('%a %d %b %H:%M'),
        effort=effort and minutestr(effort),
        clocksum=clocksum and minutestr(clocksum),
        rootname=rootname,
        )
    return (closed.start if closed else None, row)


def unique_name_from_paths(pathlist):
    namelist = []
    for path in pathlist:
        name = os.path.basename(path)
        if name in namelist:
            name_orig = name
            i = 1
            while name not in namelist:
                name = "%s <%d>" % (name_orig, i)
                i += 1
        namelist.append(name)
    return namelist


def sameday(dt1, dt2):
    return (isinstance(dt1, datetime.date) and
            isinstance(dt2, datetime.date) and
            dt1.year == dt2.year and
            dt1.month == dt2.month and
            dt1.day == dt2.day)


def table_add_oddday(key_table):
    """
    Add oddday key in each rows of key_table *IN PLACE*.
    Note that key should be a ``datetime.date`` object.
    """
    previous = None
    odd = True
    for (key, row) in key_table:
        this = key
        if not sameday(this, previous):
            odd = not odd
        row['oddday'] = odd
        previous = this


def get_data(orgnodes_list, orgpath_list, done, num=100):
    """
    Get data for rendering jinja2 template. Data is dictionary like this:

    table: list of `row`
        list of row generated by ``row_from_node``
    orgpathname_list: list of `orgpathname`
        orgpathname: dict
            contains `orgpath` and `orgname`.
            `orgname` is short and unique name for `orgpath`.
    title: str
        a title

    """
    key_table = []
    orgname_list = unique_name_from_paths(orgpath_list)
    for (nodelist, orgname) in zip(orgnodes_list, orgname_list):
        for node in nodelist:
            if node.todo == done:
                (key, row) = key_row_from_node(node)
                if key:
                    row['orgname'] = orgname
                    key_table.append((key, row))
    orgpathname_list = [
        dict(orgpath=orgpath, orgname=orgname)
        for (orgpath, orgname) in zip(orgpath_list, orgname_list)]
    key_table.sort(reverse=True)
    table_add_oddday(key_table)
    table = list(itertools.islice((row for (key, row) in key_table), num))
    return dict(table=table, orgpathname_list=orgpathname_list,
                title='Recently archived tasks')
