import logging
import os

from pyramid.config import Configurator
from pyramid.exceptions import ConfigurationError
from pyramid_who.whov2 import WhoV2AuthenticationPolicy
from pyramid.authorization import ACLAuthorizationPolicy

log = logging.getLogger(__name__)


def default_setup(config):
    from pyramid.session import UnencryptedCookieSessionFactoryConfig

    log.info('Using an unencrypted cookie-based session. This can be '
             'changed by pointing the "velruse.setup" setting at a different '
             'function for configuring the session factory.')

    settings = config.registry.settings
    secret = settings.get('osiris.session.secret')
    cookie_name = settings.get('osiris.session.cookie_name',
                               'osiris.session')
    if secret is None:
        log.warn('Configuring unencrypted cookie-based session with a '
                 'random secret which will invalidate old cookies when '
                 'restarting the app.')
        secret = ''.join('%02x' % ord(x) for x in os.urandom(16))
        log.info('autogenerated session secret: %s', secret)
    factory = UnencryptedCookieSessionFactoryConfig(
        secret, cookie_name=cookie_name)

    config.set_session_factory(factory)

    whoconfig = settings['osiris.whoconfig']
    identifier_id = 'auth_tkt'
    authn_policy = WhoV2AuthenticationPolicy(whoconfig, identifier_id)
    authz_policy = ACLAuthorizationPolicy()

    config.set_authentication_policy(authn_policy)
    config.set_authorization_policy(authz_policy)


def includeme(config):
    """Configuration function to make a pyramid app a osiris enabled one."""
    settings = config.registry.settings

    # setup application
    setup = settings.get('velruse.setup', default_setup)
    if setup:
        config.include(setup)

    # setup backing storage
    store = settings.get('osiris.store')
    if store is None:
        raise ConfigurationError(
            'invalid setting osiris.store: {0}'.format(store))
    config.include(store)

    # add the error views
    config.scan(__name__)


def make_app(**settings):
    config = Configurator(settings=settings)
    config.include(includeme)
    return config.make_wsgi_app()


def make_osiris_app(global_conf, **settings):
    """Construct a complete WSGI app ready to serve by Paste

    Example INI file:

    .. code-block:: ini

        [server:main]
        use = egg:Paste#http
        host = 0.0.0.0
        port = 80

        [composite:main]
        use = egg:Paste#urlmap
        / = YOURAPP
        /oauth2 = osiris

        [app:osiris]
        use = egg:osiris

        [app:YOURAPP]
        use = egg:YOURAPP
        full_stack = true
        static_files = true

    """
    return make_app(**settings)
