#!python3
import os
import getpass
import hashlib
import distutils.util
import shutil

import click

import ownrepo
import ownrepo.authentication
from ownrepo.utils import read_config, write_config, check_storage


@click.group()
@click.option('--storage', '-s', help='The storage directory', default='.')
@click.pass_context
def cli(ctx, storage):
    """ Manage your OwnRepo instance """
    # Detect the configuration path
    if storage:
        storage_path = storage
    elif 'OWNREPO_STORAGE' in os.environ:
        storage_path = os.environ['OWNREPO_STORAGE']
    else:
        click.echo("Error: unable to locate the storage directory", err=True)
        exit(1)

    # Don't check storage validity for these commands
    not_check_storage = ['init']

    if ctx.invoked_subcommand not in not_check_storage:
        if not check_storage(storage_path):
            click.echo("Error: invalid storage directory: {}"
                       .format(os.path.realpath(storage_path)), err=True)
            exit(1)

        ctx.obj = {}
        ctx.obj['storage'] = os.path.realpath(storage_path)

    else:
        ctx.obj = {}
        ctx.obj['storage'] = None


@cli.command()
@click.option('-b', '--bind', help='The IP to bind the port',
              default='0.0.0.0')
@click.option('-p', '--port', help='The port to listen on', default=80)
@click.option('-w', '--workers', help='How much workers run', default=1)
@click.option('-d', '--debug', help='Run in debug mode', is_flag=True)
@click.pass_context
def run(ctx, bind, port, debug, workers):
    """ Run the webserver """
    # If the application is in debug mode run it with the flask's own server
    if debug:
        app = ownrepo.create_app(ctx.obj['storage'])
        app.run(host=bind, port=port, debug=debug)
    # Else run the application with gunicorn
    else:
        options = {
            'bind': bind,
            'port': port,
            'workers': workers,
            'accesslog': '-',  # Standard output
            'errorlog': '-',  # Standard error
        }
        app = ownrepo.create_gunicorn_process(ctx.obj['storage'], options)

        try:
            app.run()  # Run gunicorn, not flask!
        except SystemExit:
            pass


@cli.command('add-user')
@click.argument('name')
@click.option('--password', '-p', help='The user\'s password', default='')
@click.pass_context
def add_user(ctx, name, password):
    """ Add an user """
    users = read_config(ctx.obj['storage'], 'users')

    # Check if the user already exists
    # If a TypeError is raised means that no user currently exists
    try:
        if name in users['users']:
            click.echo('Error: the user {} already exists'.format(name),
                       err=True)
            exit(1)
    except (TypeError, KeyError):
        users['users'] = {}

    # Ask the password only if it wasn't provided before
    if password == '':
        password = getpass.getpass()

    # Hash the password (salted sha512) and add the user
    hashed = ownrepo.authentication.hash_password(users, name, password)
    users['users'][name] = {'password': hashed}

    # Save the configuration
    write_config(ctx.obj['storage'], 'users', users)


@cli.command('remove-user')
@click.argument('name')
@click.option('-y', '--yes', help='Skip confirmation', is_flag=True)
@click.pass_context
def remove_user(ctx, name, yes):
    """ Remove an user """
    users = read_config(ctx.obj['storage'], 'users')
    acl = read_config(ctx.obj['storage'], 'acl')

    # Check if the user doesn't exist
    try:
        if name not in users['users']:
            raise TypeError  # Skip to the exception block
    except (TypeError, KeyError):
        click.echo('Error: the user {} doesn\'t exist'.format(name),
                   err=True)
        exit(1)

    if not yes:
        click.echo('Are you sure you want to delete user {}? [y/N] '
                   .format(name), nl=False)
        response = click.getchar()
        click.echo()
        try:
            if not distutils.util.strtobool(response):
                raise ValueError  # Skip to the exception block
        except ValueError:
            exit(0)

    del users['users'][name]

    # Remove the user from acl
    for repository in acl['repositories'].values():
        if name in repository['read']:
            repository['read'].remove(name)
        if name in repository['write']:
            repository['write'].remove(name)

    # Save the configuration
    write_config(ctx.obj['storage'], 'users', users)
    write_config(ctx.obj['storage'], 'acl', acl)


@cli.command('change-password')
@click.argument('name')
@click.option('--password', '-p', help='The new password', default='')
@click.pass_context
def change_password(ctx, name, password):
    """ Change an user's password """
    users = read_config(ctx.obj['storage'], 'users')

    # Check if the user doesn't exist
    try:
        if name not in users['users']:
            raise TypeError  # Skip to the exception block
    except (TypeError, KeyError):
        click.echo('Error: the user {} doesn\'t exists'.format(name))
        exit(1)

    # Ask for the new password only if it wasn't provided before
    if password == '':
        password = getpass.getpass()

    # Hash the password (salted sha512) and add the user
    hashed = ownrepo.authentication.hash_password(users, name, password)
    users['users'][name]['password'] = hashed

    write_config(ctx.obj['storage'], 'users', users)


@cli.command('create-repo')
@click.argument('name')
@click.option('--public', help='Mark the repo as public', is_flag=True)
@click.option('--allow-read', help='Allow this user to view the repo',
              multiple=True)
@click.option('--allow-write', help='Allow this user to upload to the repo',
              multiple=True)
@click.pass_context
def create_repo(ctx, name, public, allow_read, allow_write):
    """ Create a new repository """
    users = read_config(ctx.obj['storage'], 'users')
    acl = read_config(ctx.obj['storage'], 'acl')

    repo_path = ctx.obj['storage']+'/'+name

    try:
        if name in acl['repositories']:
            raise TypeError  # Skip to the exception block
    except (TypeError, KeyError):
        click.echo('Error: repository {} already exists'.format(name),
                   err=True)
        exit(1)

    users_filter = lambda user: user in users['users']

    acl['repositories'][name] = {}
    acl['repositories'][name]['read'] = list(filter(users_filter, allow_read))
    acl['repositories'][name]['write'] = list(filter(users_filter,
                                                     allow_write))

    # If the repository is public allow all to read it
    if public:
        acl['repositories'][name]['read'].append(':all')

    write_config(ctx.obj['storage'], 'acl', acl)


@cli.command('remove-repo')
@click.argument('name')
@click.option('-y', '--yes', help='Skip confirmation', is_flag=True)
@click.pass_context
def remove_repo(ctx, name, yes):
    """ Remove a repository """
    acl = read_config(ctx.obj['storage'], 'acl')
    packages = read_config(ctx.obj['storage'], 'packages')

    # Check if the repository doesn't exist
    try:
        if name not in acl['repositories']:
            raise TypeError  # Skip to the exception block
    except (TypeError, KeyError):
        click.echo('Error: the repo {} doesn\'t exist'.format(name), err=True)
        exit(1)

    if not yes:
        click.echo('Are you sure you want to delete repo {}? [y/N] '
                   .format(name), nl=False)
        response = click.getchar()
        click.echo()
        try:
            if not distutils.util.strtobool(response):
                raise ValueError  # Skip to the exception block
        except ValueError:
            exit(0)

    del acl['repositories'][name]
    shutil.rmtree(ctx.obj['storage']+'/'+name)

    del packages['repositories'][name]

    # Save the configuration
    write_config(ctx.obj['storage'], 'acl', acl)
    write_config(ctx.obj['storage'], 'packages', packages)


@cli.command('acl')
@click.argument('repo')
@click.option('--allow-read', help='Allow this user to view the repo',
              multiple=True)
@click.option('--allow-write', help='Allow this user to upload to the repo',
              multiple=True)
@click.option('--deny-read', help='Deny this user to view the repo',
              multiple=True)
@click.option('--deny-write', help='Deny this user to upload to the repo',
              multiple=True)
@click.option('--public', help='Convert the repo to public', is_flag=True)
@click.option('--private', help='Convert the repo to private', is_flag=True)
@click.pass_context
def acl(ctx, repo, allow_read, allow_write, deny_read, deny_write, public,
        private):
    """ Update a repo's acl """
    users = read_config(ctx.obj['storage'], 'users')
    acl = read_config(ctx.obj['storage'], 'acl')

    try:
        if repo not in acl['repositories']:
            raise TypeError  # Skip to the exception block
    except (TypeError, KeyError):
        click.echo('Error: repository {} doesn\'t exist'.format(repo),
                   err=True)
        exit(1)

    users_filter = lambda user: user in users['users']
    repo_acl = acl['repositories'][repo]

    # Toggle public and private
    if public and ':all' not in repo_acl['read']:
        repo_acl['read'].append(':all')
    if private and ':all' in repo_acl['read']:
            repo_acl['read'].remove(':all')

    # Apply new acls

    def _allow(perm, who):
        to_allow = list(filter(users_filter, who))
        for user in to_allow:
            if user in repo_acl[perm]:
                continue
            repo_acl[perm].append(user)
    _allow('read', allow_read)
    _allow('write', allow_write)

    def _deny(perm, who):
        to_deny = list(filter(users_filter, who))
        for user in to_deny:
            if user not in repo_acl[perm]:
                continue
            repo_acl[perm].remove(user)
    _deny('read', deny_read)
    _deny('write', deny_write)

    # Save the configuration
    write_config(ctx.obj['storage'], 'acl', acl)


@cli.command('rename-repo')
@click.argument('origin')
@click.argument('destination')
@click.option('-y', '--yes', help='Skip confirmation', is_flag=True)
@click.pass_context
def rename_repo(ctx, origin, destination, yes):
    """ Rename a repository """
    acl = read_config(ctx.obj['storage'], 'acl')
    packages = read_config(ctx.obj['storage'], 'packages')

    # Check if the origin repository exists
    try:
        if origin not in acl['repositories']:
            raise TypeError  # Skip to the exception block
    except (TypeError, KeyError):
        click.echo('Error: the repo {} doesn\'t exist'.format(origin),
                   err=True)
        exit(1)

    # Check if the destination repository doesn't exist
    try:
        if destination in acl['repositories']:
            click.echo('Error: the repo {} already exists'
                       .format(destination), err=True)
            exit(1)
    except (KeyError, TypeError):
        pass

    if not yes:
        click.echo('Are you sure you want to rename the repo {} to {}? [y/N]'
                   .format(origin, destination), nl=False)
        response = click.getchar()
        click.echo()
        try:
            if not distutils.util.strtobool(response):
                raise ValueError  # Skip to the exception block
        except ValueError:
            exit(0)

    acl['repositories'][destination] = acl['repositories'][origin]
    del acl['repositories'][origin]

    os.rename(ctx.obj['storage']+'/'+origin,
              ctx.obj['storage']+'/'+destination)

    packages['repositories'][destination] = packages['repositories'][origin]
    del packages['repositories'][origin]

    write_config(ctx.obj['storage'], 'acl', acl)
    write_config(ctx.obj['storage'], 'packages', packages)


@cli.command()
@click.argument('directory', default='.', required=False)
@click.option('--sample', help='Include sample data', is_flag=True)
@click.pass_context
def init(ctx, directory, sample):
    """ Initialize a storage directory """
    directory = os.path.realpath(directory)  # Convert to the full path

    # If the directory doesn't exist, create it
    if not os.path.exists(directory):
        os.makedirs(directory)

    # If the directory is not empty
    if os.listdir(directory):
        click.echo('Error: directory {} is not empty'.format(directory),
                   err=True)
        exit(1)

    # Create the basic structure
    base = os.path.join(directory, '.ownrepo')
    os.makedirs(base)

    # If --sample is provided include some sample data
    if sample:
        users = {
            'password_salt': ownrepo.authentication.random_salt(),
            'users': {},
        }
        password = ownrepo.authentication.hash_password(users, 'admin',
                                                        'admin')
        users['users'] = {'admin': {'password': password}}

        acl = {
            'repositories': {
                'public': {
                    'read': {':all'},
                    'write': {'admin'},
                    },
                'private': {
                    'read': {'admin'},
                    'write': {'admin'},
                },
            },
        }

        packages = {
            'repositories': {
                'public': {},
                'private': {},
            },
        }
    # Else included only the needed
    else:
        users = {
            'password_salt': ownrepo.authentication.random_salt(),
            'users': {},
        }
        acl = {
            'repositories': {},
        }
        packages = {
            'repositories': {},
        }

    # Write configuration files
    write_config(directory, 'users', users)
    write_config(directory, 'acl', acl)
    write_config(directory, 'packages', packages)


if __name__ == '__main__':
    cli()
