#!python3
import os

import hashlib
import yaml
import pkg_resources
import flask
import werkzeug

from ownrepo.utils import read_config, write_config


class RepositoriesManager:
    """ Manage all repositories """

    def __init__(self, app):
        self._app = app
        self._repos = {}

        acls = read_config(app.ownrepo_storage, 'acl')
        for name, acl in acls['repositories'].items():
            self._init_repo(name, acl)

    def _init_repo(self, name, acl):
        """ Initialize a repository """
        if name in self._repos:
            raise NameError('Repository {} already exists'.format(name))
        self._repos[name] = Repository(self, name, acl)

    def list(self):
        """ Return a list of accessible repositories """
        return dict(filter(lambda i: i[1].readable, self._repos.items()))

    def get(self, name):
        """ Return an accessible repository """
        if name in self._repos:
            if self._repos[name].readable:
                return self._repos[name]
            raise ValueError('Not enough privileges for reading {}'
                             .format(name))
        else:
            raise ValueError('Invalid repository: {}'.format(name))

    def exists(self, name):
        """ Check if a repository exists """
        return name in self._repos


class Repository:
    """ Representation of a repository """

    def __init__(self, manager, name, acl):
        self._manager = manager
        self._acl = acl
        self._directory = os.path.join(manager._app.ownrepo_storage, name)
        self.name = name
        self.packages = {}

        self.private = ':all' not in self._acl['read']

        if not os.path.exists(self._directory):
            self._init_new()
        self._load_packages()

    def __str__(self):
        return self.name

    def _init_new(self):
        """ Initialize the repository directory """
        os.makedirs(self._directory)

        # Update packages configuration file
        packages = read_config(self._manager._app.ownrepo_storage, 'packages')
        packages['repositories'][self.name] = {}
        write_config(self._manager._app.ownrepo_storage, 'packages', packages)

    def _load_packages(self):
        """ Load all packages informations """
        # Load packages data
        packages = read_config(self._manager._app.ownrepo_storage, 'packages')
        packages = packages['repositories'][self.name]

        for name, details in packages.items():
            self._load_package(name, details)

    def _load_package(self, name, details):
        """ Initialize a package """
        if name in self.packages:
            raise NameError('Package already registered')

        self.packages[name] = Package(self, name, details)

    @property
    def url(self):
        return flask.url_for('web.simple_repo', repo=self.name,
                             _external=True)

    @property
    def readable(self):
        # If all can read the repository return True
        if ':all' in self._acl['read']:
            return True

        # If the user isn't authenticated return False
        if getattr(flask.g, 'authenticated_as', None) is None:
            return False

        # If the user isn't allowed return False
        if flask.g.authenticated_as not in self._acl['read']:
            return False

        return True

    @property
    def writable(self):
        # If all can edit the repository return True
        if ':all' in self._acl['write']:
            return True

        # If the user isn't authenticated return False
        if getattr(flask.g, 'authenticated_as', None) is None:
            return False

        # If the user isn't allowed return False
        if flask.g.authenticated_as not in self._acl['write']:
            return False

        return True

    def create_package(self, name):
        """ Create a new package on the repository """
        if name in self.packages:
            raise NameError('Package already exists')

        # Add the new package to the "database"
        content = read_config(self._manager._app.ownrepo_storage, 'packages')
        content['repositories'][self.name][name] = {'releases': {}}
        write_config(self._manager._app.ownrepo_storage, 'packages', content)

        # Load the new package
        self._load_package(name, content['repositories'][self.name][name])


class Package:
    """ Representation of a package """

    def __init__(self, repo, name, details):
        self.repo = repo
        self.name = name
        self._calculate_releases(details)
        self._details = details

    def __str__(self):
        return self.name

    def _calculate_releases(self, details):
        """ Calculate available releases """
        sort_releases = lambda r: pkg_resources.parse_version(r)

        self.releases = details["releases"]
        self.releases_sorted = list(reversed(sorted(self.releases.keys(),
                                                    key=sort_releases)))

    def files(self, release=None):
        """ Get all available files """
        # Calculate md5 of a file
        md5 = lambda f: hashlib.md5(open(self.repo._directory+'/'+f, 'rb')
                                    .read()).hexdigest()

        # Capture all files of this package or release if it's passed
        releases = self.releases
        if release:
            releases = {release: self.releases[release]}
        files = {}
        for name, release in releases.items():
            try:
                files[release['source']] = md5(release['source'])
            except KeyError:
                pass
            try:
                for one in release['binary']:
                    files[one] = md5(one)
            except KeyError:
                pass
            try:
                for one in release['wheels']:
                    files[one] = md5(one)
            except KeyError:
                pass
            try:
                for one in release['eggs']:
                    files[one] = md5(one)
            except KeyError:
                pass
        return files

    def add_release(self, version, archive_type, archive, md5=None):
        """ Add a new release to the package """
        # Save the new archive
        file_name = werkzeug.secure_filename(archive.filename)
        archive.save(self.repo._directory+'/'+file_name)

        content = read_config(self.repo._manager._app.ownrepo_storage,
                              'packages')
        package = content['repositories'][self.repo.name][self.name]
        if version not in package['releases']:
            package['releases'][version] = {}

        release = package['releases'][version]
        if archive_type == 'sdist':
            release['source'] = file_name
        elif archive_type == 'bdist':
            if 'binary' not in release:
                release['binary'] = []
            release['binary'].append(file_name)
        elif archive_type == 'bdist_egg':
            if 'eggs' not in release:
                release['eggs'] = []
            release['eggs'].append(file_name)
        elif archive_type == 'bdist_wheel':
            if 'wheels' not in release:
                release['wheels'] = []
            release['wheels'].append(file_name)
        else:
            # Remove saved file
            os.unlink(self.repo._directory+'/'+file_name)
            raise ValueError('Invalid archive type')

        write_config(self.repo._manager._app.ownrepo_storage, 'packages',
                     content)

        # Update cache
        self._details = content['repositories'][self.repo.name][self.name]
        self._calculate_releases(self._details)

    def install_command(self, release=None):
        """ Generate the command for installing this package """
        parts = ['pip install '+self.name]
        if release:
            parts.append('=='+release)
        parts.append(' -i '+self.repo.url)
        return "".join(parts)
