#!python3
import os

import yaml
import click


def read_config(storage, file_name):
    """ Read a configuration file ad return the output """
    path = os.path.join(storage, '.ownrepo', file_name+'.yml')

    try:
        with open(path, 'r') as f:
            config = yaml.load(f.read())
    except OSError:
        click.echo("Error: Unable to open the configuration file: {}"
                   .format(path), err=True)
        exit(1)
    except yaml.YAMLError as e:
        click.echo("Error: Syntax error in the configuration file:", err=True)
        click.echo(e, err=True)
        exit(1)
    else:
        return config


def write_config(storage, file_name, config):
    """ Write a dict in a configuration file """
    path = os.path.join(storage, '.ownrepo', file_name+'.yml')

    try:
        with open(path, 'w') as f:
            f.write(yaml.dump(config, default_flow_style=False))
    except OSError:
        click.echo("Error: Unable to write the configuration file to {}"
                   .format(path), err=True)
        exit(1)


def check_storage(path):
    """ Return if a storage path is an ownrepo storage directory """
    # The directory must exists
    if not os.path.exists(path):
        return False

    # It must have a .ownrepo subdirectory
    subdir = os.path.join(path, '.ownrepo')
    if not os.path.exists(subdir) or not os.path.isdir(subdir):
        return False

    # It must have some files in the .ownrepo subdirectory
    needed_files = ('acl.yml', 'users.yml', 'packages.yml')
    for one in needed_files:
        if not os.path.exists(os.path.join(subdir, one)):
            return False

    return True
