# -*- coding: utf-8 -*-
"""
A thin wrapper around the docker management script ``fig`` developed by
Orchard to work around some of the quirks.
"""
from __future__ import unicode_literals, absolute_import
import os
import sys
import logging
import subprocess

from inspect import getdoc
from StringIO import StringIO
from ConfigParser import ParsingError, SafeConfigParser

from fig.project import NoSuchService, ConfigurationError
from fig.service import BuildError
from fig.cli.main import TopLevelCommand, parse_doc_section
from fig.packages.docker.errors import APIError
from fig.cli.errors import UserError
from fig.cli.docopt_command import NoSuchCommand
from fig.cli.utils import cached_property

FNULL = open(os.devnull, 'w')

log = logging.getLogger(__name__)


def get_docker_ip():
    try:
        return subprocess.check_output(['boot2docker', 'ip'], stderr=FNULL)
    except subprocess.CalledProcessError:
        return None


def get_docker_port():
    try:
        raw_config = subprocess.check_output(
            ['boot2docker', 'config'], stderr=FNULL)
    except subprocess.CalledProcessError:
        return None

    raw_config = "[boot2docker]\n{}".format(raw_config)

    cfg = SafeConfigParser()

    try:
        cfg.readfp(StringIO(raw_config))
    except ParsingError:
        return None

    return cfg.get('boot2docker', 'DockerPort')


class SpeckCommand(TopLevelCommand):
    __doc__ = TopLevelCommand.__doc__

    def dispatch(self, *args, **kwargs):
        if sys.platform == 'darwin' and 'DOCKER_HOST' not in os.environ:
            os.environ['DOCKER_HOST'] = "tcp://{}:{}".format(
                get_docker_ip(), get_docker_port())
        super(SpeckCommand, self).dispatch(*args, **kwargs)

    def debug(self, options):
        docker_host = os.getenv('DOCKER_HOST')
        if docker_host:
            print 'Using specified docker host: {}'.format(docker_host)
        else:
            print 'DOCKER_HOST not set, using defaults'

    def find_path(self, filename='fig.yml'):
        fig_path = None
        segments = os.path.abspath(self.base_dir).split('/')
        segments[0] = '/'

        for idx in xrange(1, len(segments)):
            base_path = os.path.join(*segments[:-idx])
            path = os.path.join(base_path, filename)
            if os.path.exists(path):
                fig_path = path
        if fig_path:
            return fig_path
        return os.path.join(self.base_dir, filename)

    @cached_property
    def yaml_path(self):
        if self._yaml_path is not None and os.path.exists(self._yaml_path):
            return self._yaml_path
        return self.find_path()

    @yaml_path.setter
    def yaml_path(self, value):
        self._yaml_path = value


def cli():
    console_handler = logging.StreamHandler(stream=sys.stderr)
    console_handler.setFormatter(logging.Formatter())
    console_handler.setLevel(logging.INFO)
    root_logger = logging.getLogger()
    root_logger.addHandler(console_handler)
    root_logger.setLevel(logging.DEBUG)

    # Disable requests logging
    logging.getLogger("requests").propagate = False

    try:
        command = SpeckCommand()
        command.sys_dispatch()
    except KeyboardInterrupt:
        log.error("\nAborting.")
        sys.exit(1)
    except (UserError, NoSuchService, ConfigurationError) as e:
        log.error(e.msg)
        sys.exit(1)
    except NoSuchCommand as e:
        log.error("No such command: %s", e.command)
        log.error("")
        log.error(
            "\n".join(parse_doc_section("commands:", getdoc(e.supercommand))))
        sys.exit(1)
    except APIError as e:
        log.error(e.explanation)
        sys.exit(1)
    except BuildError as e:
        log.error(
            "Service '%s' failed to build: %s" % (e.service.name, e.reason))
        sys.exit(1)
