# -*- coding: utf-8 -*-

import bunch


class MultiParser(object):
    """Manages multiple parsers with support for chaining.

    The use-case of chaining in this situation is, until this day,
    unknown.

    It was only 5 lines of code and it adds a lot of flexibility.

    Chaining would work like this::

        from parsers import MultiParser, parser

        # This is a very special case where the default xml parser does
        # not work. Thankfully, we can override this default parser
        # without dropping support for other parsers and without
        # interfering with the default xml parser.
        import superspecialxml
        myspecialparser = MultiParser(parser)
        myspecialparser.register_parser('xml', superspecialssxml.parse)

        # my_special_parser can still parse json and stuff without
        # re-registering those parsers.

    Sometimes, we have to prepare ourselves for the worst situation.
    """

    def __init__(self, parent=None, bunchify=False):
        self.parent = parent
        self.default_parser = None
        self.registered_parsers = {}
        self.bunchify = bunchify

    def register_parser(self, format, function):
        """Registers a new parsing method.

        You can also use this method to overwrite existing parsers by
        providing the same ``format`` argument.

        :param format: The unique name of the parser.
        :param function: The method which should be executed when
            parsing something.
        """
        self.registered_parsers[format] = function

        # the first registered parser will be set as the default one
        if self.default_parser is None:
            self.default_parser = format

    def registered_parser(self, format):
        """Registers the decorated parsing method as a new parser.

        :param format: The unique name of the parser.
        """
        def decorator(f):
            self.registered_parsers[format] = f
            return f
        return decorator

    def parse(self, text, format=None, bunchify=None):
        """Parses the given text with the given formatting.

        When no formatting is provided, the default parser will be
        used. If anything fails, it will try to pass everything to the
        parent's :meth:`parse` method. The parent option is completely
        optional.

        :param text: The text to parse.
        :param format: The method of parsing to use.
        :param bunchify: (optional) Overwrite the default bunchify
            parameter. This will allow dictionaries to be accessible
            via dots.
        :return: Entirely dependent on the parsing method. This is
            usually a dictionary.
        """
        if bunchify is None:
            bunchify = self.bunchify
        if format is None:
            if self.default_parser is None:
                if self.parent:
                    return self.parent.parse(text, format, bunchify)
                return text
            else:
                format = self.default_parser
        try:
            # the actual parsing happens here
            parsed_text = self.registered_parsers[format](text)
            if bunchify:
                return bunch.bunchify(parsed_text)
            return parsed_text
        except KeyError:
            if self.parent:
                return self.parent.parse(text, format, bunchify)
            raise UnknownParserError('No registered parser '
                                     'with name "{name}"'.format(name=format))


class UnknownParserError(Exception):
    """This error is raised whenever an invalid format is provided to
    the parsing method."""


parser = MultiParser()

# json is set as the default parser
import json
parser.register_parser('json', json.loads)

try:
    # xml support::
    #     pip install xmltodict
    import xmltodict
    parser.register_parser('xml', xmltodict.parse)
except ImportError:
    pass

try:
    # yaml support::
    #     pip install PyYAML
    import yaml
    parser.register_parser('yaml', yaml.load)
except ImportError:
    pass
