# PayPal REST SDK 

Continuous integration status:

[![Build Status](https://travis-ci.org/paypal/rest-api-sdk-python.png?branch=master)](https://travis-ci.org/paypal/rest-api-sdk-python) [![Coverage Status](https://coveralls.io/repos/paypal/rest-api-sdk-python/badge.png?branch=master)](https://coveralls.io/r/paypal/rest-api-sdk-python?branch=master) 

PyPI status:

[![PyPi version](https://pypip.in/v/paypalrestsdk/badge.png)](https://crate.io/packages/paypalrestsdk/)
[![PyPi downloads](https://pypip.in/d/paypalrestsdk/badge.png)](https://crate.io/packages/paypalrestsdk/)

The PayPal REST SDK provides Python APIs to create, process and manage payment. The [Paypal REST APIs](https://developer.paypal.com/webapps/developer/docs/api/) are fully supported by the sdk.

> **Before starting to use the sdk, please be aware of the [existing issues and currently available or upcoming features](https://github.com/paypal/rest-api-sdk-python/wiki/Existing-Issues-and-Unavailable%5CUpcoming-features) for the REST APIs. (which the sdks are based on)** 

## Installation

Install using `easy_install`:

```sh
easy_install paypalrestsdk
```

Install using `pip`:

```sh
pip install paypalrestsdk
```

## Configuration

 Register for a developer account and get your client_id and secret at [PayPal Developer Portal](https://developer.paypal.com).

```python
import paypalrestsdk
paypalrestsdk.configure({
  "mode": "sandbox", # sandbox or live
  "client_id": "EBWKjlELKMYqRNQ6sYvFo64FtaRLRR5BdHEESmha49TM",
  "client_secret": "EO422dn3gQLgDbuwqTjzrFgFtaRLRR5BdHEESmha49TM" })
```

Configure through environment variables:

```sh
export PAYPAL_MODE=sandbox   # sandbox or live
export PAYPAL_CLIENT_ID=EBWKjlELKMYqRNQ6sYvFo64FtaRLRR5BdHEESmha49TM
export PAYPAL_CLIENT_SECRET=EO422dn3gQLgDbuwqTjzrFgFtaRLRR5BdHEESmha49TM
```

Configure through a non-global API object
```python
import paypalrestsdk
my_api = paypalrestsdk.Api({
  'mode': 'sandbox',
  'client_id': '...',
  'client_secret': '...'})

payment = paypalrestsdk.Payment({...}, api=my_api)

```

### Create Payment

```python
import paypalrestsdk
import logging

# Include Headers and Content by setting logging level to DEBUG, particularly for
# Paypal-Debug-Id if requesting PayPal Merchant Technical Services for support
logging.basicConfig(level=logging.INFO)

paypalrestsdk.configure({
  "mode": "sandbox", # sandbox or live
  "client_id": "EBWKjlELKMYqRNQ6sYvFo64FtaRLRR5BdHEESmha49TM",
  "client_secret": "EO422dn3gQLgDbuwqTjzrFgFtaRLRR5BdHEESmha49TM" })

payment = paypalrestsdk.Payment({
  "intent": "sale",
  "payer": {
    "payment_method": "credit_card",
    "funding_instruments": [{
      "credit_card": {
        "type": "visa",
        "number": "4417119669820331",
        "expire_month": "11",
        "expire_year": "2018",
        "cvv2": "874",
        "first_name": "Joe",
        "last_name": "Shopper" }}]},
  "transactions": [{
    "item_list": {
      "items": [{
        "name": "item",
        "sku": "item",
        "price": "1.00",
        "currency": "USD",
        "quantity": 1 }]},
    "amount": {
      "total": "1.00",
      "currency": "USD" },
    "description": "This is the payment transaction description." }]})

if payment.create():
  print("Payment created successfully")
else:
  print(payment.error)
```

### Get Payment details

```python
# Fetch Payment
payment = paypalrestsdk.Payment.find("PAY-57363176S1057143SKE2HO3A")

# Get List of Payments
payment_history = paypalrestsdk.Payment.all({"count": 10})
payment_history.payments
```

### Execute Payment

Only for [Payment](https://github.com/paypal/rest-api-sdk-python/blob/master/samples/payment/create_with_paypal.py) with `payment_method` as `"paypal"`

```python
payment = paypalrestsdk.Payment.find("PAY-57363176S1057143SKE2HO3A")

if payment.execute({"payer_id": "DUFRQ8GWYMJXC"}):
  print("Payment execute successfully")
else:
  print(payment.error) # Error Hash
```

### OpenID Connect

```python
import paypalrestsdk
from paypalrestsdk.openid_connect import Tokeninfo, Userinfo

paypalrestsdk.configure({
  "mode": "sandbox",
  "client_id": "CLIENT_ID",
  "client_secret": "CLIENT_SECRET",
  "openid_redirect_uri": "http://example.com" })

# Generate login url
login_url = Tokeninfo.authorize_url({ "scope": "openid profile"})

# Create tokeninfo with Authorize code
tokeninfo = Tokeninfo.create("Replace with Authorize code")

# Refresh tokeninfo
tokeninfo = tokeninfo.refresh()

# Create tokeninfo with refresh_token
tokeninfo = Tokeninfo.create_with_refresh_token("Replace with refresh_token")

# Get userinfo
userinfo  = tokeninfo.userinfo()

# Get userinfo with access_token
userinfo  = Userinfo.get("Replace with access_token")

# Generate logout url
logout_url = tokeninfo.logout_url()
```

### Future Payments

Check out this [sample](/samples/payment/create_future_payment.py) for executing [future payments](https://developer.paypal.com/docs/integration/mobile/make-future-payment/) for a customer who has granted consent on a mobile device.

### Subscription

Create [subscription payments](https://developer.paypal.com/docs/integration/direct/create-billing-plan/) i.e. planned sets of future recurring payments at periodic intervals. Billing plans serve as the template for a subscription while billing agreements can be used to have customers subscribe to the plan.

### Create a billing plan

```python
from paypalrestsdk import BillingPlan

billing_plan = BillingPlan({
    "name": "Fast Speed Plan",
    "description": "Create Plan for Regular",
    "merchant_preferences": {
        "auto_bill_amount": "yes",
        "cancel_url": "http://www.paypal.com/cancel",
        "initial_fail_amount_action": "continue",
        "max_fail_attempts": "1",
        "return_url": "http://www.paypal.com/execute",
        "setup_fee": {
            "currency": "USD",
            "value": "25"
        }
    },
    "payment_definitions": [
        {
            "amount": {
                "currency": "USD",
                "value": "100"
            },
            "charge_models": [
                {
                    "amount": {
                        "currency": "USD",
                        "value": "10.60"
                    },
                    "type": "SHIPPING"
                },
                {
                    "amount": {
                        "currency": "USD",
                        "value": "20"
                    },
                    "type": "TAX"
                }
            ],
            "cycles": "0",
            "frequency": "MONTH",
            "frequency_interval": "1",
            "name": "Regular 1",
            "type": "REGULAR"
        }
    ],
    "type": "INFINITE"
})

response = billing_plan.create()
print response
```

Check out [more samples](/samples/subscription/). The [Subscription REST APIs](https://developer.paypal.com/webapps/developer/docs/api/#subscriptions) are fully supported by the sdk.

### Invoicing

Create, send and manage [invoices](https://developer.paypal.com/docs/integration/direct/invoicing/).

### Create an invoice

```python
from paypalrestsdk import Invoice

invoice = Invoice({
  'merchant_info': {
    "email": "default@merchant.com",
  },
  "billing_info": [{
    "email": "example@example.com"
  }],
  "items": [{
      "name": "Widgets",
      "quantity": 20,
      "unit_price": {
        "currency": "USD",
        "value": 2
      }
    }],
})

response = invoice.create()
print response
```

Check out [more samples](/samples/invoice/). The [Invoicing REST APIs](https://developer.paypal.com/webapps/developer/docs/api/#invoicing) are fully supported by the sdk.
