#!/usr/bin/env python

""" Python implementation of Rijndael encryption algorithm.

This code is in the public domain.

This code is based on a public domain C implementation
by Philip J. Erdelsky:
  http://www.efgh.com/software/rijndael.htm

"""

import sys
from struct import pack, unpack

def KEYLENGTH(keybits): return (keybits)/8
def RKLENGTH(keybits): return (keybits)/8+28
def NROUNDS(keybits): return (keybits)/32+6

Te0 = [
  0xc66363a5L, 0xf87c7c84L, 0xee777799L, 0xf67b7b8dL,
  0xfff2f20dL, 0xd66b6bbdL, 0xde6f6fb1L, 0x91c5c554L,
  0x60303050L, 0x02010103L, 0xce6767a9L, 0x562b2b7dL,
  0xe7fefe19L, 0xb5d7d762L, 0x4dababe6L, 0xec76769aL,
  0x8fcaca45L, 0x1f82829dL, 0x89c9c940L, 0xfa7d7d87L,
  0xeffafa15L, 0xb25959ebL, 0x8e4747c9L, 0xfbf0f00bL,
  0x41adadecL, 0xb3d4d467L, 0x5fa2a2fdL, 0x45afafeaL,
  0x239c9cbfL, 0x53a4a4f7L, 0xe4727296L, 0x9bc0c05bL,
  0x75b7b7c2L, 0xe1fdfd1cL, 0x3d9393aeL, 0x4c26266aL,
  0x6c36365aL, 0x7e3f3f41L, 0xf5f7f702L, 0x83cccc4fL,
  0x6834345cL, 0x51a5a5f4L, 0xd1e5e534L, 0xf9f1f108L,
  0xe2717193L, 0xabd8d873L, 0x62313153L, 0x2a15153fL,
  0x0804040cL, 0x95c7c752L, 0x46232365L, 0x9dc3c35eL,
  0x30181828L, 0x379696a1L, 0x0a05050fL, 0x2f9a9ab5L,
  0x0e070709L, 0x24121236L, 0x1b80809bL, 0xdfe2e23dL,
  0xcdebeb26L, 0x4e272769L, 0x7fb2b2cdL, 0xea75759fL,
  0x1209091bL, 0x1d83839eL, 0x582c2c74L, 0x341a1a2eL,
  0x361b1b2dL, 0xdc6e6eb2L, 0xb45a5aeeL, 0x5ba0a0fbL,
  0xa45252f6L, 0x763b3b4dL, 0xb7d6d661L, 0x7db3b3ceL,
  0x5229297bL, 0xdde3e33eL, 0x5e2f2f71L, 0x13848497L,
  0xa65353f5L, 0xb9d1d168L, 0x00000000L, 0xc1eded2cL,
  0x40202060L, 0xe3fcfc1fL, 0x79b1b1c8L, 0xb65b5bedL,
  0xd46a6abeL, 0x8dcbcb46L, 0x67bebed9L, 0x7239394bL,
  0x944a4adeL, 0x984c4cd4L, 0xb05858e8L, 0x85cfcf4aL,
  0xbbd0d06bL, 0xc5efef2aL, 0x4faaaae5L, 0xedfbfb16L,
  0x864343c5L, 0x9a4d4dd7L, 0x66333355L, 0x11858594L,
  0x8a4545cfL, 0xe9f9f910L, 0x04020206L, 0xfe7f7f81L,
  0xa05050f0L, 0x783c3c44L, 0x259f9fbaL, 0x4ba8a8e3L,
  0xa25151f3L, 0x5da3a3feL, 0x804040c0L, 0x058f8f8aL,
  0x3f9292adL, 0x219d9dbcL, 0x70383848L, 0xf1f5f504L,
  0x63bcbcdfL, 0x77b6b6c1L, 0xafdada75L, 0x42212163L,
  0x20101030L, 0xe5ffff1aL, 0xfdf3f30eL, 0xbfd2d26dL,
  0x81cdcd4cL, 0x180c0c14L, 0x26131335L, 0xc3ecec2fL,
  0xbe5f5fe1L, 0x359797a2L, 0x884444ccL, 0x2e171739L,
  0x93c4c457L, 0x55a7a7f2L, 0xfc7e7e82L, 0x7a3d3d47L,
  0xc86464acL, 0xba5d5de7L, 0x3219192bL, 0xe6737395L,
  0xc06060a0L, 0x19818198L, 0x9e4f4fd1L, 0xa3dcdc7fL,
  0x44222266L, 0x542a2a7eL, 0x3b9090abL, 0x0b888883L,
  0x8c4646caL, 0xc7eeee29L, 0x6bb8b8d3L, 0x2814143cL,
  0xa7dede79L, 0xbc5e5ee2L, 0x160b0b1dL, 0xaddbdb76L,
  0xdbe0e03bL, 0x64323256L, 0x743a3a4eL, 0x140a0a1eL,
  0x924949dbL, 0x0c06060aL, 0x4824246cL, 0xb85c5ce4L,
  0x9fc2c25dL, 0xbdd3d36eL, 0x43acacefL, 0xc46262a6L,
  0x399191a8L, 0x319595a4L, 0xd3e4e437L, 0xf279798bL,
  0xd5e7e732L, 0x8bc8c843L, 0x6e373759L, 0xda6d6db7L,
  0x018d8d8cL, 0xb1d5d564L, 0x9c4e4ed2L, 0x49a9a9e0L,
  0xd86c6cb4L, 0xac5656faL, 0xf3f4f407L, 0xcfeaea25L,
  0xca6565afL, 0xf47a7a8eL, 0x47aeaee9L, 0x10080818L,
  0x6fbabad5L, 0xf0787888L, 0x4a25256fL, 0x5c2e2e72L,
  0x381c1c24L, 0x57a6a6f1L, 0x73b4b4c7L, 0x97c6c651L,
  0xcbe8e823L, 0xa1dddd7cL, 0xe874749cL, 0x3e1f1f21L,
  0x964b4bddL, 0x61bdbddcL, 0x0d8b8b86L, 0x0f8a8a85L,
  0xe0707090L, 0x7c3e3e42L, 0x71b5b5c4L, 0xcc6666aaL,
  0x904848d8L, 0x06030305L, 0xf7f6f601L, 0x1c0e0e12L,
  0xc26161a3L, 0x6a35355fL, 0xae5757f9L, 0x69b9b9d0L,
  0x17868691L, 0x99c1c158L, 0x3a1d1d27L, 0x279e9eb9L,
  0xd9e1e138L, 0xebf8f813L, 0x2b9898b3L, 0x22111133L,
  0xd26969bbL, 0xa9d9d970L, 0x078e8e89L, 0x339494a7L,
  0x2d9b9bb6L, 0x3c1e1e22L, 0x15878792L, 0xc9e9e920L,
  0x87cece49L, 0xaa5555ffL, 0x50282878L, 0xa5dfdf7aL,
  0x038c8c8fL, 0x59a1a1f8L, 0x09898980L, 0x1a0d0d17L,
  0x65bfbfdaL, 0xd7e6e631L, 0x844242c6L, 0xd06868b8L,
  0x824141c3L, 0x299999b0L, 0x5a2d2d77L, 0x1e0f0f11L,
  0x7bb0b0cbL, 0xa85454fcL, 0x6dbbbbd6L, 0x2c16163aL,
  ]

Te1 = [
  0xa5c66363L, 0x84f87c7cL, 0x99ee7777L, 0x8df67b7bL,
  0x0dfff2f2L, 0xbdd66b6bL, 0xb1de6f6fL, 0x5491c5c5L,
  0x50603030L, 0x03020101L, 0xa9ce6767L, 0x7d562b2bL,
  0x19e7fefeL, 0x62b5d7d7L, 0xe64dababL, 0x9aec7676L,
  0x458fcacaL, 0x9d1f8282L, 0x4089c9c9L, 0x87fa7d7dL,
  0x15effafaL, 0xebb25959L, 0xc98e4747L, 0x0bfbf0f0L,
  0xec41adadL, 0x67b3d4d4L, 0xfd5fa2a2L, 0xea45afafL,
  0xbf239c9cL, 0xf753a4a4L, 0x96e47272L, 0x5b9bc0c0L,
  0xc275b7b7L, 0x1ce1fdfdL, 0xae3d9393L, 0x6a4c2626L,
  0x5a6c3636L, 0x417e3f3fL, 0x02f5f7f7L, 0x4f83ccccL,
  0x5c683434L, 0xf451a5a5L, 0x34d1e5e5L, 0x08f9f1f1L,
  0x93e27171L, 0x73abd8d8L, 0x53623131L, 0x3f2a1515L,
  0x0c080404L, 0x5295c7c7L, 0x65462323L, 0x5e9dc3c3L,
  0x28301818L, 0xa1379696L, 0x0f0a0505L, 0xb52f9a9aL,
  0x090e0707L, 0x36241212L, 0x9b1b8080L, 0x3ddfe2e2L,
  0x26cdebebL, 0x694e2727L, 0xcd7fb2b2L, 0x9fea7575L,
  0x1b120909L, 0x9e1d8383L, 0x74582c2cL, 0x2e341a1aL,
  0x2d361b1bL, 0xb2dc6e6eL, 0xeeb45a5aL, 0xfb5ba0a0L,
  0xf6a45252L, 0x4d763b3bL, 0x61b7d6d6L, 0xce7db3b3L,
  0x7b522929L, 0x3edde3e3L, 0x715e2f2fL, 0x97138484L,
  0xf5a65353L, 0x68b9d1d1L, 0x00000000L, 0x2cc1ededL,
  0x60402020L, 0x1fe3fcfcL, 0xc879b1b1L, 0xedb65b5bL,
  0xbed46a6aL, 0x468dcbcbL, 0xd967bebeL, 0x4b723939L,
  0xde944a4aL, 0xd4984c4cL, 0xe8b05858L, 0x4a85cfcfL,
  0x6bbbd0d0L, 0x2ac5efefL, 0xe54faaaaL, 0x16edfbfbL,
  0xc5864343L, 0xd79a4d4dL, 0x55663333L, 0x94118585L,
  0xcf8a4545L, 0x10e9f9f9L, 0x06040202L, 0x81fe7f7fL,
  0xf0a05050L, 0x44783c3cL, 0xba259f9fL, 0xe34ba8a8L,
  0xf3a25151L, 0xfe5da3a3L, 0xc0804040L, 0x8a058f8fL,
  0xad3f9292L, 0xbc219d9dL, 0x48703838L, 0x04f1f5f5L,
  0xdf63bcbcL, 0xc177b6b6L, 0x75afdadaL, 0x63422121L,
  0x30201010L, 0x1ae5ffffL, 0x0efdf3f3L, 0x6dbfd2d2L,
  0x4c81cdcdL, 0x14180c0cL, 0x35261313L, 0x2fc3ececL,
  0xe1be5f5fL, 0xa2359797L, 0xcc884444L, 0x392e1717L,
  0x5793c4c4L, 0xf255a7a7L, 0x82fc7e7eL, 0x477a3d3dL,
  0xacc86464L, 0xe7ba5d5dL, 0x2b321919L, 0x95e67373L,
  0xa0c06060L, 0x98198181L, 0xd19e4f4fL, 0x7fa3dcdcL,
  0x66442222L, 0x7e542a2aL, 0xab3b9090L, 0x830b8888L,
  0xca8c4646L, 0x29c7eeeeL, 0xd36bb8b8L, 0x3c281414L,
  0x79a7dedeL, 0xe2bc5e5eL, 0x1d160b0bL, 0x76addbdbL,
  0x3bdbe0e0L, 0x56643232L, 0x4e743a3aL, 0x1e140a0aL,
  0xdb924949L, 0x0a0c0606L, 0x6c482424L, 0xe4b85c5cL,
  0x5d9fc2c2L, 0x6ebdd3d3L, 0xef43acacL, 0xa6c46262L,
  0xa8399191L, 0xa4319595L, 0x37d3e4e4L, 0x8bf27979L,
  0x32d5e7e7L, 0x438bc8c8L, 0x596e3737L, 0xb7da6d6dL,
  0x8c018d8dL, 0x64b1d5d5L, 0xd29c4e4eL, 0xe049a9a9L,
  0xb4d86c6cL, 0xfaac5656L, 0x07f3f4f4L, 0x25cfeaeaL,
  0xafca6565L, 0x8ef47a7aL, 0xe947aeaeL, 0x18100808L,
  0xd56fbabaL, 0x88f07878L, 0x6f4a2525L, 0x725c2e2eL,
  0x24381c1cL, 0xf157a6a6L, 0xc773b4b4L, 0x5197c6c6L,
  0x23cbe8e8L, 0x7ca1ddddL, 0x9ce87474L, 0x213e1f1fL,
  0xdd964b4bL, 0xdc61bdbdL, 0x860d8b8bL, 0x850f8a8aL,
  0x90e07070L, 0x427c3e3eL, 0xc471b5b5L, 0xaacc6666L,
  0xd8904848L, 0x05060303L, 0x01f7f6f6L, 0x121c0e0eL,
  0xa3c26161L, 0x5f6a3535L, 0xf9ae5757L, 0xd069b9b9L,
  0x91178686L, 0x5899c1c1L, 0x273a1d1dL, 0xb9279e9eL,
  0x38d9e1e1L, 0x13ebf8f8L, 0xb32b9898L, 0x33221111L,
  0xbbd26969L, 0x70a9d9d9L, 0x89078e8eL, 0xa7339494L,
  0xb62d9b9bL, 0x223c1e1eL, 0x92158787L, 0x20c9e9e9L,
  0x4987ceceL, 0xffaa5555L, 0x78502828L, 0x7aa5dfdfL,
  0x8f038c8cL, 0xf859a1a1L, 0x80098989L, 0x171a0d0dL,
  0xda65bfbfL, 0x31d7e6e6L, 0xc6844242L, 0xb8d06868L,
  0xc3824141L, 0xb0299999L, 0x775a2d2dL, 0x111e0f0fL,
  0xcb7bb0b0L, 0xfca85454L, 0xd66dbbbbL, 0x3a2c1616L,
  ]

Te2 = [
  0x63a5c663L, 0x7c84f87cL, 0x7799ee77L, 0x7b8df67bL,
  0xf20dfff2L, 0x6bbdd66bL, 0x6fb1de6fL, 0xc55491c5L,
  0x30506030L, 0x01030201L, 0x67a9ce67L, 0x2b7d562bL,
  0xfe19e7feL, 0xd762b5d7L, 0xabe64dabL, 0x769aec76L,
  0xca458fcaL, 0x829d1f82L, 0xc94089c9L, 0x7d87fa7dL,
  0xfa15effaL, 0x59ebb259L, 0x47c98e47L, 0xf00bfbf0L,
  0xadec41adL, 0xd467b3d4L, 0xa2fd5fa2L, 0xafea45afL,
  0x9cbf239cL, 0xa4f753a4L, 0x7296e472L, 0xc05b9bc0L,
  0xb7c275b7L, 0xfd1ce1fdL, 0x93ae3d93L, 0x266a4c26L,
  0x365a6c36L, 0x3f417e3fL, 0xf702f5f7L, 0xcc4f83ccL,
  0x345c6834L, 0xa5f451a5L, 0xe534d1e5L, 0xf108f9f1L,
  0x7193e271L, 0xd873abd8L, 0x31536231L, 0x153f2a15L,
  0x040c0804L, 0xc75295c7L, 0x23654623L, 0xc35e9dc3L,
  0x18283018L, 0x96a13796L, 0x050f0a05L, 0x9ab52f9aL,
  0x07090e07L, 0x12362412L, 0x809b1b80L, 0xe23ddfe2L,
  0xeb26cdebL, 0x27694e27L, 0xb2cd7fb2L, 0x759fea75L,
  0x091b1209L, 0x839e1d83L, 0x2c74582cL, 0x1a2e341aL,
  0x1b2d361bL, 0x6eb2dc6eL, 0x5aeeb45aL, 0xa0fb5ba0L,
  0x52f6a452L, 0x3b4d763bL, 0xd661b7d6L, 0xb3ce7db3L,
  0x297b5229L, 0xe33edde3L, 0x2f715e2fL, 0x84971384L,
  0x53f5a653L, 0xd168b9d1L, 0x00000000L, 0xed2cc1edL,
  0x20604020L, 0xfc1fe3fcL, 0xb1c879b1L, 0x5bedb65bL,
  0x6abed46aL, 0xcb468dcbL, 0xbed967beL, 0x394b7239L,
  0x4ade944aL, 0x4cd4984cL, 0x58e8b058L, 0xcf4a85cfL,
  0xd06bbbd0L, 0xef2ac5efL, 0xaae54faaL, 0xfb16edfbL,
  0x43c58643L, 0x4dd79a4dL, 0x33556633L, 0x85941185L,
  0x45cf8a45L, 0xf910e9f9L, 0x02060402L, 0x7f81fe7fL,
  0x50f0a050L, 0x3c44783cL, 0x9fba259fL, 0xa8e34ba8L,
  0x51f3a251L, 0xa3fe5da3L, 0x40c08040L, 0x8f8a058fL,
  0x92ad3f92L, 0x9dbc219dL, 0x38487038L, 0xf504f1f5L,
  0xbcdf63bcL, 0xb6c177b6L, 0xda75afdaL, 0x21634221L,
  0x10302010L, 0xff1ae5ffL, 0xf30efdf3L, 0xd26dbfd2L,
  0xcd4c81cdL, 0x0c14180cL, 0x13352613L, 0xec2fc3ecL,
  0x5fe1be5fL, 0x97a23597L, 0x44cc8844L, 0x17392e17L,
  0xc45793c4L, 0xa7f255a7L, 0x7e82fc7eL, 0x3d477a3dL,
  0x64acc864L, 0x5de7ba5dL, 0x192b3219L, 0x7395e673L,
  0x60a0c060L, 0x81981981L, 0x4fd19e4fL, 0xdc7fa3dcL,
  0x22664422L, 0x2a7e542aL, 0x90ab3b90L, 0x88830b88L,
  0x46ca8c46L, 0xee29c7eeL, 0xb8d36bb8L, 0x143c2814L,
  0xde79a7deL, 0x5ee2bc5eL, 0x0b1d160bL, 0xdb76addbL,
  0xe03bdbe0L, 0x32566432L, 0x3a4e743aL, 0x0a1e140aL,
  0x49db9249L, 0x060a0c06L, 0x246c4824L, 0x5ce4b85cL,
  0xc25d9fc2L, 0xd36ebdd3L, 0xacef43acL, 0x62a6c462L,
  0x91a83991L, 0x95a43195L, 0xe437d3e4L, 0x798bf279L,
  0xe732d5e7L, 0xc8438bc8L, 0x37596e37L, 0x6db7da6dL,
  0x8d8c018dL, 0xd564b1d5L, 0x4ed29c4eL, 0xa9e049a9L,
  0x6cb4d86cL, 0x56faac56L, 0xf407f3f4L, 0xea25cfeaL,
  0x65afca65L, 0x7a8ef47aL, 0xaee947aeL, 0x08181008L,
  0xbad56fbaL, 0x7888f078L, 0x256f4a25L, 0x2e725c2eL,
  0x1c24381cL, 0xa6f157a6L, 0xb4c773b4L, 0xc65197c6L,
  0xe823cbe8L, 0xdd7ca1ddL, 0x749ce874L, 0x1f213e1fL,
  0x4bdd964bL, 0xbddc61bdL, 0x8b860d8bL, 0x8a850f8aL,
  0x7090e070L, 0x3e427c3eL, 0xb5c471b5L, 0x66aacc66L,
  0x48d89048L, 0x03050603L, 0xf601f7f6L, 0x0e121c0eL,
  0x61a3c261L, 0x355f6a35L, 0x57f9ae57L, 0xb9d069b9L,
  0x86911786L, 0xc15899c1L, 0x1d273a1dL, 0x9eb9279eL,
  0xe138d9e1L, 0xf813ebf8L, 0x98b32b98L, 0x11332211L,
  0x69bbd269L, 0xd970a9d9L, 0x8e89078eL, 0x94a73394L,
  0x9bb62d9bL, 0x1e223c1eL, 0x87921587L, 0xe920c9e9L,
  0xce4987ceL, 0x55ffaa55L, 0x28785028L, 0xdf7aa5dfL,
  0x8c8f038cL, 0xa1f859a1L, 0x89800989L, 0x0d171a0dL,
  0xbfda65bfL, 0xe631d7e6L, 0x42c68442L, 0x68b8d068L,
  0x41c38241L, 0x99b02999L, 0x2d775a2dL, 0x0f111e0fL,
  0xb0cb7bb0L, 0x54fca854L, 0xbbd66dbbL, 0x163a2c16L,
  ]

Te3 = [
  0x6363a5c6L, 0x7c7c84f8L, 0x777799eeL, 0x7b7b8df6L,
  0xf2f20dffL, 0x6b6bbdd6L, 0x6f6fb1deL, 0xc5c55491L,
  0x30305060L, 0x01010302L, 0x6767a9ceL, 0x2b2b7d56L,
  0xfefe19e7L, 0xd7d762b5L, 0xababe64dL, 0x76769aecL,
  0xcaca458fL, 0x82829d1fL, 0xc9c94089L, 0x7d7d87faL,
  0xfafa15efL, 0x5959ebb2L, 0x4747c98eL, 0xf0f00bfbL,
  0xadadec41L, 0xd4d467b3L, 0xa2a2fd5fL, 0xafafea45L,
  0x9c9cbf23L, 0xa4a4f753L, 0x727296e4L, 0xc0c05b9bL,
  0xb7b7c275L, 0xfdfd1ce1L, 0x9393ae3dL, 0x26266a4cL,
  0x36365a6cL, 0x3f3f417eL, 0xf7f702f5L, 0xcccc4f83L,
  0x34345c68L, 0xa5a5f451L, 0xe5e534d1L, 0xf1f108f9L,
  0x717193e2L, 0xd8d873abL, 0x31315362L, 0x15153f2aL,
  0x04040c08L, 0xc7c75295L, 0x23236546L, 0xc3c35e9dL,
  0x18182830L, 0x9696a137L, 0x05050f0aL, 0x9a9ab52fL,
  0x0707090eL, 0x12123624L, 0x80809b1bL, 0xe2e23ddfL,
  0xebeb26cdL, 0x2727694eL, 0xb2b2cd7fL, 0x75759feaL,
  0x09091b12L, 0x83839e1dL, 0x2c2c7458L, 0x1a1a2e34L,
  0x1b1b2d36L, 0x6e6eb2dcL, 0x5a5aeeb4L, 0xa0a0fb5bL,
  0x5252f6a4L, 0x3b3b4d76L, 0xd6d661b7L, 0xb3b3ce7dL,
  0x29297b52L, 0xe3e33eddL, 0x2f2f715eL, 0x84849713L,
  0x5353f5a6L, 0xd1d168b9L, 0x00000000L, 0xeded2cc1L,
  0x20206040L, 0xfcfc1fe3L, 0xb1b1c879L, 0x5b5bedb6L,
  0x6a6abed4L, 0xcbcb468dL, 0xbebed967L, 0x39394b72L,
  0x4a4ade94L, 0x4c4cd498L, 0x5858e8b0L, 0xcfcf4a85L,
  0xd0d06bbbL, 0xefef2ac5L, 0xaaaae54fL, 0xfbfb16edL,
  0x4343c586L, 0x4d4dd79aL, 0x33335566L, 0x85859411L,
  0x4545cf8aL, 0xf9f910e9L, 0x02020604L, 0x7f7f81feL,
  0x5050f0a0L, 0x3c3c4478L, 0x9f9fba25L, 0xa8a8e34bL,
  0x5151f3a2L, 0xa3a3fe5dL, 0x4040c080L, 0x8f8f8a05L,
  0x9292ad3fL, 0x9d9dbc21L, 0x38384870L, 0xf5f504f1L,
  0xbcbcdf63L, 0xb6b6c177L, 0xdada75afL, 0x21216342L,
  0x10103020L, 0xffff1ae5L, 0xf3f30efdL, 0xd2d26dbfL,
  0xcdcd4c81L, 0x0c0c1418L, 0x13133526L, 0xecec2fc3L,
  0x5f5fe1beL, 0x9797a235L, 0x4444cc88L, 0x1717392eL,
  0xc4c45793L, 0xa7a7f255L, 0x7e7e82fcL, 0x3d3d477aL,
  0x6464acc8L, 0x5d5de7baL, 0x19192b32L, 0x737395e6L,
  0x6060a0c0L, 0x81819819L, 0x4f4fd19eL, 0xdcdc7fa3L,
  0x22226644L, 0x2a2a7e54L, 0x9090ab3bL, 0x8888830bL,
  0x4646ca8cL, 0xeeee29c7L, 0xb8b8d36bL, 0x14143c28L,
  0xdede79a7L, 0x5e5ee2bcL, 0x0b0b1d16L, 0xdbdb76adL,
  0xe0e03bdbL, 0x32325664L, 0x3a3a4e74L, 0x0a0a1e14L,
  0x4949db92L, 0x06060a0cL, 0x24246c48L, 0x5c5ce4b8L,
  0xc2c25d9fL, 0xd3d36ebdL, 0xacacef43L, 0x6262a6c4L,
  0x9191a839L, 0x9595a431L, 0xe4e437d3L, 0x79798bf2L,
  0xe7e732d5L, 0xc8c8438bL, 0x3737596eL, 0x6d6db7daL,
  0x8d8d8c01L, 0xd5d564b1L, 0x4e4ed29cL, 0xa9a9e049L,
  0x6c6cb4d8L, 0x5656faacL, 0xf4f407f3L, 0xeaea25cfL,
  0x6565afcaL, 0x7a7a8ef4L, 0xaeaee947L, 0x08081810L,
  0xbabad56fL, 0x787888f0L, 0x25256f4aL, 0x2e2e725cL,
  0x1c1c2438L, 0xa6a6f157L, 0xb4b4c773L, 0xc6c65197L,
  0xe8e823cbL, 0xdddd7ca1L, 0x74749ce8L, 0x1f1f213eL,
  0x4b4bdd96L, 0xbdbddc61L, 0x8b8b860dL, 0x8a8a850fL,
  0x707090e0L, 0x3e3e427cL, 0xb5b5c471L, 0x6666aaccL,
  0x4848d890L, 0x03030506L, 0xf6f601f7L, 0x0e0e121cL,
  0x6161a3c2L, 0x35355f6aL, 0x5757f9aeL, 0xb9b9d069L,
  0x86869117L, 0xc1c15899L, 0x1d1d273aL, 0x9e9eb927L,
  0xe1e138d9L, 0xf8f813ebL, 0x9898b32bL, 0x11113322L,
  0x6969bbd2L, 0xd9d970a9L, 0x8e8e8907L, 0x9494a733L,
  0x9b9bb62dL, 0x1e1e223cL, 0x87879215L, 0xe9e920c9L,
  0xcece4987L, 0x5555ffaaL, 0x28287850L, 0xdfdf7aa5L,
  0x8c8c8f03L, 0xa1a1f859L, 0x89898009L, 0x0d0d171aL,
  0xbfbfda65L, 0xe6e631d7L, 0x4242c684L, 0x6868b8d0L,
  0x4141c382L, 0x9999b029L, 0x2d2d775aL, 0x0f0f111eL,
  0xb0b0cb7bL, 0x5454fca8L, 0xbbbbd66dL, 0x16163a2cL,
  ]

Te4 = [
  0x63636363L, 0x7c7c7c7cL, 0x77777777L, 0x7b7b7b7bL,
  0xf2f2f2f2L, 0x6b6b6b6bL, 0x6f6f6f6fL, 0xc5c5c5c5L,
  0x30303030L, 0x01010101L, 0x67676767L, 0x2b2b2b2bL,
  0xfefefefeL, 0xd7d7d7d7L, 0xababababL, 0x76767676L,
  0xcacacacaL, 0x82828282L, 0xc9c9c9c9L, 0x7d7d7d7dL,
  0xfafafafaL, 0x59595959L, 0x47474747L, 0xf0f0f0f0L,
  0xadadadadL, 0xd4d4d4d4L, 0xa2a2a2a2L, 0xafafafafL,
  0x9c9c9c9cL, 0xa4a4a4a4L, 0x72727272L, 0xc0c0c0c0L,
  0xb7b7b7b7L, 0xfdfdfdfdL, 0x93939393L, 0x26262626L,
  0x36363636L, 0x3f3f3f3fL, 0xf7f7f7f7L, 0xccccccccL,
  0x34343434L, 0xa5a5a5a5L, 0xe5e5e5e5L, 0xf1f1f1f1L,
  0x71717171L, 0xd8d8d8d8L, 0x31313131L, 0x15151515L,
  0x04040404L, 0xc7c7c7c7L, 0x23232323L, 0xc3c3c3c3L,
  0x18181818L, 0x96969696L, 0x05050505L, 0x9a9a9a9aL,
  0x07070707L, 0x12121212L, 0x80808080L, 0xe2e2e2e2L,
  0xebebebebL, 0x27272727L, 0xb2b2b2b2L, 0x75757575L,
  0x09090909L, 0x83838383L, 0x2c2c2c2cL, 0x1a1a1a1aL,
  0x1b1b1b1bL, 0x6e6e6e6eL, 0x5a5a5a5aL, 0xa0a0a0a0L,
  0x52525252L, 0x3b3b3b3bL, 0xd6d6d6d6L, 0xb3b3b3b3L,
  0x29292929L, 0xe3e3e3e3L, 0x2f2f2f2fL, 0x84848484L,
  0x53535353L, 0xd1d1d1d1L, 0x00000000L, 0xededededL,
  0x20202020L, 0xfcfcfcfcL, 0xb1b1b1b1L, 0x5b5b5b5bL,
  0x6a6a6a6aL, 0xcbcbcbcbL, 0xbebebebeL, 0x39393939L,
  0x4a4a4a4aL, 0x4c4c4c4cL, 0x58585858L, 0xcfcfcfcfL,
  0xd0d0d0d0L, 0xefefefefL, 0xaaaaaaaaL, 0xfbfbfbfbL,
  0x43434343L, 0x4d4d4d4dL, 0x33333333L, 0x85858585L,
  0x45454545L, 0xf9f9f9f9L, 0x02020202L, 0x7f7f7f7fL,
  0x50505050L, 0x3c3c3c3cL, 0x9f9f9f9fL, 0xa8a8a8a8L,
  0x51515151L, 0xa3a3a3a3L, 0x40404040L, 0x8f8f8f8fL,
  0x92929292L, 0x9d9d9d9dL, 0x38383838L, 0xf5f5f5f5L,
  0xbcbcbcbcL, 0xb6b6b6b6L, 0xdadadadaL, 0x21212121L,
  0x10101010L, 0xffffffffL, 0xf3f3f3f3L, 0xd2d2d2d2L,
  0xcdcdcdcdL, 0x0c0c0c0cL, 0x13131313L, 0xececececL,
  0x5f5f5f5fL, 0x97979797L, 0x44444444L, 0x17171717L,
  0xc4c4c4c4L, 0xa7a7a7a7L, 0x7e7e7e7eL, 0x3d3d3d3dL,
  0x64646464L, 0x5d5d5d5dL, 0x19191919L, 0x73737373L,
  0x60606060L, 0x81818181L, 0x4f4f4f4fL, 0xdcdcdcdcL,
  0x22222222L, 0x2a2a2a2aL, 0x90909090L, 0x88888888L,
  0x46464646L, 0xeeeeeeeeL, 0xb8b8b8b8L, 0x14141414L,
  0xdedededeL, 0x5e5e5e5eL, 0x0b0b0b0bL, 0xdbdbdbdbL,
  0xe0e0e0e0L, 0x32323232L, 0x3a3a3a3aL, 0x0a0a0a0aL,
  0x49494949L, 0x06060606L, 0x24242424L, 0x5c5c5c5cL,
  0xc2c2c2c2L, 0xd3d3d3d3L, 0xacacacacL, 0x62626262L,
  0x91919191L, 0x95959595L, 0xe4e4e4e4L, 0x79797979L,
  0xe7e7e7e7L, 0xc8c8c8c8L, 0x37373737L, 0x6d6d6d6dL,
  0x8d8d8d8dL, 0xd5d5d5d5L, 0x4e4e4e4eL, 0xa9a9a9a9L,
  0x6c6c6c6cL, 0x56565656L, 0xf4f4f4f4L, 0xeaeaeaeaL,
  0x65656565L, 0x7a7a7a7aL, 0xaeaeaeaeL, 0x08080808L,
  0xbabababaL, 0x78787878L, 0x25252525L, 0x2e2e2e2eL,
  0x1c1c1c1cL, 0xa6a6a6a6L, 0xb4b4b4b4L, 0xc6c6c6c6L,
  0xe8e8e8e8L, 0xddddddddL, 0x74747474L, 0x1f1f1f1fL,
  0x4b4b4b4bL, 0xbdbdbdbdL, 0x8b8b8b8bL, 0x8a8a8a8aL,
  0x70707070L, 0x3e3e3e3eL, 0xb5b5b5b5L, 0x66666666L,
  0x48484848L, 0x03030303L, 0xf6f6f6f6L, 0x0e0e0e0eL,
  0x61616161L, 0x35353535L, 0x57575757L, 0xb9b9b9b9L,
  0x86868686L, 0xc1c1c1c1L, 0x1d1d1d1dL, 0x9e9e9e9eL,
  0xe1e1e1e1L, 0xf8f8f8f8L, 0x98989898L, 0x11111111L,
  0x69696969L, 0xd9d9d9d9L, 0x8e8e8e8eL, 0x94949494L,
  0x9b9b9b9bL, 0x1e1e1e1eL, 0x87878787L, 0xe9e9e9e9L,
  0xcecececeL, 0x55555555L, 0x28282828L, 0xdfdfdfdfL,
  0x8c8c8c8cL, 0xa1a1a1a1L, 0x89898989L, 0x0d0d0d0dL,
  0xbfbfbfbfL, 0xe6e6e6e6L, 0x42424242L, 0x68686868L,
  0x41414141L, 0x99999999L, 0x2d2d2d2dL, 0x0f0f0f0fL,
  0xb0b0b0b0L, 0x54545454L, 0xbbbbbbbbL, 0x16161616L,
  ]

Td0 = [
  0x51f4a750L, 0x7e416553L, 0x1a17a4c3L, 0x3a275e96L,
  0x3bab6bcbL, 0x1f9d45f1L, 0xacfa58abL, 0x4be30393L,
  0x2030fa55L, 0xad766df6L, 0x88cc7691L, 0xf5024c25L,
  0x4fe5d7fcL, 0xc52acbd7L, 0x26354480L, 0xb562a38fL,
  0xdeb15a49L, 0x25ba1b67L, 0x45ea0e98L, 0x5dfec0e1L,
  0xc32f7502L, 0x814cf012L, 0x8d4697a3L, 0x6bd3f9c6L,
  0x038f5fe7L, 0x15929c95L, 0xbf6d7aebL, 0x955259daL,
  0xd4be832dL, 0x587421d3L, 0x49e06929L, 0x8ec9c844L,
  0x75c2896aL, 0xf48e7978L, 0x99583e6bL, 0x27b971ddL,
  0xbee14fb6L, 0xf088ad17L, 0xc920ac66L, 0x7dce3ab4L,
  0x63df4a18L, 0xe51a3182L, 0x97513360L, 0x62537f45L,
  0xb16477e0L, 0xbb6bae84L, 0xfe81a01cL, 0xf9082b94L,
  0x70486858L, 0x8f45fd19L, 0x94de6c87L, 0x527bf8b7L,
  0xab73d323L, 0x724b02e2L, 0xe31f8f57L, 0x6655ab2aL,
  0xb2eb2807L, 0x2fb5c203L, 0x86c57b9aL, 0xd33708a5L,
  0x302887f2L, 0x23bfa5b2L, 0x02036abaL, 0xed16825cL,
  0x8acf1c2bL, 0xa779b492L, 0xf307f2f0L, 0x4e69e2a1L,
  0x65daf4cdL, 0x0605bed5L, 0xd134621fL, 0xc4a6fe8aL,
  0x342e539dL, 0xa2f355a0L, 0x058ae132L, 0xa4f6eb75L,
  0x0b83ec39L, 0x4060efaaL, 0x5e719f06L, 0xbd6e1051L,
  0x3e218af9L, 0x96dd063dL, 0xdd3e05aeL, 0x4de6bd46L,
  0x91548db5L, 0x71c45d05L, 0x0406d46fL, 0x605015ffL,
  0x1998fb24L, 0xd6bde997L, 0x894043ccL, 0x67d99e77L,
  0xb0e842bdL, 0x07898b88L, 0xe7195b38L, 0x79c8eedbL,
  0xa17c0a47L, 0x7c420fe9L, 0xf8841ec9L, 0x00000000L,
  0x09808683L, 0x322bed48L, 0x1e1170acL, 0x6c5a724eL,
  0xfd0efffbL, 0x0f853856L, 0x3daed51eL, 0x362d3927L,
  0x0a0fd964L, 0x685ca621L, 0x9b5b54d1L, 0x24362e3aL,
  0x0c0a67b1L, 0x9357e70fL, 0xb4ee96d2L, 0x1b9b919eL,
  0x80c0c54fL, 0x61dc20a2L, 0x5a774b69L, 0x1c121a16L,
  0xe293ba0aL, 0xc0a02ae5L, 0x3c22e043L, 0x121b171dL,
  0x0e090d0bL, 0xf28bc7adL, 0x2db6a8b9L, 0x141ea9c8L,
  0x57f11985L, 0xaf75074cL, 0xee99ddbbL, 0xa37f60fdL,
  0xf701269fL, 0x5c72f5bcL, 0x44663bc5L, 0x5bfb7e34L,
  0x8b432976L, 0xcb23c6dcL, 0xb6edfc68L, 0xb8e4f163L,
  0xd731dccaL, 0x42638510L, 0x13972240L, 0x84c61120L,
  0x854a247dL, 0xd2bb3df8L, 0xaef93211L, 0xc729a16dL,
  0x1d9e2f4bL, 0xdcb230f3L, 0x0d8652ecL, 0x77c1e3d0L,
  0x2bb3166cL, 0xa970b999L, 0x119448faL, 0x47e96422L,
  0xa8fc8cc4L, 0xa0f03f1aL, 0x567d2cd8L, 0x223390efL,
  0x87494ec7L, 0xd938d1c1L, 0x8ccaa2feL, 0x98d40b36L,
  0xa6f581cfL, 0xa57ade28L, 0xdab78e26L, 0x3fadbfa4L,
  0x2c3a9de4L, 0x5078920dL, 0x6a5fcc9bL, 0x547e4662L,
  0xf68d13c2L, 0x90d8b8e8L, 0x2e39f75eL, 0x82c3aff5L,
  0x9f5d80beL, 0x69d0937cL, 0x6fd52da9L, 0xcf2512b3L,
  0xc8ac993bL, 0x10187da7L, 0xe89c636eL, 0xdb3bbb7bL,
  0xcd267809L, 0x6e5918f4L, 0xec9ab701L, 0x834f9aa8L,
  0xe6956e65L, 0xaaffe67eL, 0x21bccf08L, 0xef15e8e6L,
  0xbae79bd9L, 0x4a6f36ceL, 0xea9f09d4L, 0x29b07cd6L,
  0x31a4b2afL, 0x2a3f2331L, 0xc6a59430L, 0x35a266c0L,
  0x744ebc37L, 0xfc82caa6L, 0xe090d0b0L, 0x33a7d815L,
  0xf104984aL, 0x41ecdaf7L, 0x7fcd500eL, 0x1791f62fL,
  0x764dd68dL, 0x43efb04dL, 0xccaa4d54L, 0xe49604dfL,
  0x9ed1b5e3L, 0x4c6a881bL, 0xc12c1fb8L, 0x4665517fL,
  0x9d5eea04L, 0x018c355dL, 0xfa877473L, 0xfb0b412eL,
  0xb3671d5aL, 0x92dbd252L, 0xe9105633L, 0x6dd64713L,
  0x9ad7618cL, 0x37a10c7aL, 0x59f8148eL, 0xeb133c89L,
  0xcea927eeL, 0xb761c935L, 0xe11ce5edL, 0x7a47b13cL,
  0x9cd2df59L, 0x55f2733fL, 0x1814ce79L, 0x73c737bfL,
  0x53f7cdeaL, 0x5ffdaa5bL, 0xdf3d6f14L, 0x7844db86L,
  0xcaaff381L, 0xb968c43eL, 0x3824342cL, 0xc2a3405fL,
  0x161dc372L, 0xbce2250cL, 0x283c498bL, 0xff0d9541L,
  0x39a80171L, 0x080cb3deL, 0xd8b4e49cL, 0x6456c190L,
  0x7bcb8461L, 0xd532b670L, 0x486c5c74L, 0xd0b85742L,
  ]

Td1 = [
  0x5051f4a7L, 0x537e4165L, 0xc31a17a4L, 0x963a275eL,
  0xcb3bab6bL, 0xf11f9d45L, 0xabacfa58L, 0x934be303L,
  0x552030faL, 0xf6ad766dL, 0x9188cc76L, 0x25f5024cL,
  0xfc4fe5d7L, 0xd7c52acbL, 0x80263544L, 0x8fb562a3L,
  0x49deb15aL, 0x6725ba1bL, 0x9845ea0eL, 0xe15dfec0L,
  0x02c32f75L, 0x12814cf0L, 0xa38d4697L, 0xc66bd3f9L,
  0xe7038f5fL, 0x9515929cL, 0xebbf6d7aL, 0xda955259L,
  0x2dd4be83L, 0xd3587421L, 0x2949e069L, 0x448ec9c8L,
  0x6a75c289L, 0x78f48e79L, 0x6b99583eL, 0xdd27b971L,
  0xb6bee14fL, 0x17f088adL, 0x66c920acL, 0xb47dce3aL,
  0x1863df4aL, 0x82e51a31L, 0x60975133L, 0x4562537fL,
  0xe0b16477L, 0x84bb6baeL, 0x1cfe81a0L, 0x94f9082bL,
  0x58704868L, 0x198f45fdL, 0x8794de6cL, 0xb7527bf8L,
  0x23ab73d3L, 0xe2724b02L, 0x57e31f8fL, 0x2a6655abL,
  0x07b2eb28L, 0x032fb5c2L, 0x9a86c57bL, 0xa5d33708L,
  0xf2302887L, 0xb223bfa5L, 0xba02036aL, 0x5ced1682L,
  0x2b8acf1cL, 0x92a779b4L, 0xf0f307f2L, 0xa14e69e2L,
  0xcd65daf4L, 0xd50605beL, 0x1fd13462L, 0x8ac4a6feL,
  0x9d342e53L, 0xa0a2f355L, 0x32058ae1L, 0x75a4f6ebL,
  0x390b83ecL, 0xaa4060efL, 0x065e719fL, 0x51bd6e10L,
  0xf93e218aL, 0x3d96dd06L, 0xaedd3e05L, 0x464de6bdL,
  0xb591548dL, 0x0571c45dL, 0x6f0406d4L, 0xff605015L,
  0x241998fbL, 0x97d6bde9L, 0xcc894043L, 0x7767d99eL,
  0xbdb0e842L, 0x8807898bL, 0x38e7195bL, 0xdb79c8eeL,
  0x47a17c0aL, 0xe97c420fL, 0xc9f8841eL, 0x00000000L,
  0x83098086L, 0x48322bedL, 0xac1e1170L, 0x4e6c5a72L,
  0xfbfd0effL, 0x560f8538L, 0x1e3daed5L, 0x27362d39L,
  0x640a0fd9L, 0x21685ca6L, 0xd19b5b54L, 0x3a24362eL,
  0xb10c0a67L, 0x0f9357e7L, 0xd2b4ee96L, 0x9e1b9b91L,
  0x4f80c0c5L, 0xa261dc20L, 0x695a774bL, 0x161c121aL,
  0x0ae293baL, 0xe5c0a02aL, 0x433c22e0L, 0x1d121b17L,
  0x0b0e090dL, 0xadf28bc7L, 0xb92db6a8L, 0xc8141ea9L,
  0x8557f119L, 0x4caf7507L, 0xbbee99ddL, 0xfda37f60L,
  0x9ff70126L, 0xbc5c72f5L, 0xc544663bL, 0x345bfb7eL,
  0x768b4329L, 0xdccb23c6L, 0x68b6edfcL, 0x63b8e4f1L,
  0xcad731dcL, 0x10426385L, 0x40139722L, 0x2084c611L,
  0x7d854a24L, 0xf8d2bb3dL, 0x11aef932L, 0x6dc729a1L,
  0x4b1d9e2fL, 0xf3dcb230L, 0xec0d8652L, 0xd077c1e3L,
  0x6c2bb316L, 0x99a970b9L, 0xfa119448L, 0x2247e964L,
  0xc4a8fc8cL, 0x1aa0f03fL, 0xd8567d2cL, 0xef223390L,
  0xc787494eL, 0xc1d938d1L, 0xfe8ccaa2L, 0x3698d40bL,
  0xcfa6f581L, 0x28a57adeL, 0x26dab78eL, 0xa43fadbfL,
  0xe42c3a9dL, 0x0d507892L, 0x9b6a5fccL, 0x62547e46L,
  0xc2f68d13L, 0xe890d8b8L, 0x5e2e39f7L, 0xf582c3afL,
  0xbe9f5d80L, 0x7c69d093L, 0xa96fd52dL, 0xb3cf2512L,
  0x3bc8ac99L, 0xa710187dL, 0x6ee89c63L, 0x7bdb3bbbL,
  0x09cd2678L, 0xf46e5918L, 0x01ec9ab7L, 0xa8834f9aL,
  0x65e6956eL, 0x7eaaffe6L, 0x0821bccfL, 0xe6ef15e8L,
  0xd9bae79bL, 0xce4a6f36L, 0xd4ea9f09L, 0xd629b07cL,
  0xaf31a4b2L, 0x312a3f23L, 0x30c6a594L, 0xc035a266L,
  0x37744ebcL, 0xa6fc82caL, 0xb0e090d0L, 0x1533a7d8L,
  0x4af10498L, 0xf741ecdaL, 0x0e7fcd50L, 0x2f1791f6L,
  0x8d764dd6L, 0x4d43efb0L, 0x54ccaa4dL, 0xdfe49604L,
  0xe39ed1b5L, 0x1b4c6a88L, 0xb8c12c1fL, 0x7f466551L,
  0x049d5eeaL, 0x5d018c35L, 0x73fa8774L, 0x2efb0b41L,
  0x5ab3671dL, 0x5292dbd2L, 0x33e91056L, 0x136dd647L,
  0x8c9ad761L, 0x7a37a10cL, 0x8e59f814L, 0x89eb133cL,
  0xeecea927L, 0x35b761c9L, 0xede11ce5L, 0x3c7a47b1L,
  0x599cd2dfL, 0x3f55f273L, 0x791814ceL, 0xbf73c737L,
  0xea53f7cdL, 0x5b5ffdaaL, 0x14df3d6fL, 0x867844dbL,
  0x81caaff3L, 0x3eb968c4L, 0x2c382434L, 0x5fc2a340L,
  0x72161dc3L, 0x0cbce225L, 0x8b283c49L, 0x41ff0d95L,
  0x7139a801L, 0xde080cb3L, 0x9cd8b4e4L, 0x906456c1L,
  0x617bcb84L, 0x70d532b6L, 0x74486c5cL, 0x42d0b857L,
  ]

Td2 = [
  0xa75051f4L, 0x65537e41L, 0xa4c31a17L, 0x5e963a27L,
  0x6bcb3babL, 0x45f11f9dL, 0x58abacfaL, 0x03934be3L,
  0xfa552030L, 0x6df6ad76L, 0x769188ccL, 0x4c25f502L,
  0xd7fc4fe5L, 0xcbd7c52aL, 0x44802635L, 0xa38fb562L,
  0x5a49deb1L, 0x1b6725baL, 0x0e9845eaL, 0xc0e15dfeL,
  0x7502c32fL, 0xf012814cL, 0x97a38d46L, 0xf9c66bd3L,
  0x5fe7038fL, 0x9c951592L, 0x7aebbf6dL, 0x59da9552L,
  0x832dd4beL, 0x21d35874L, 0x692949e0L, 0xc8448ec9L,
  0x896a75c2L, 0x7978f48eL, 0x3e6b9958L, 0x71dd27b9L,
  0x4fb6bee1L, 0xad17f088L, 0xac66c920L, 0x3ab47dceL,
  0x4a1863dfL, 0x3182e51aL, 0x33609751L, 0x7f456253L,
  0x77e0b164L, 0xae84bb6bL, 0xa01cfe81L, 0x2b94f908L,
  0x68587048L, 0xfd198f45L, 0x6c8794deL, 0xf8b7527bL,
  0xd323ab73L, 0x02e2724bL, 0x8f57e31fL, 0xab2a6655L,
  0x2807b2ebL, 0xc2032fb5L, 0x7b9a86c5L, 0x08a5d337L,
  0x87f23028L, 0xa5b223bfL, 0x6aba0203L, 0x825ced16L,
  0x1c2b8acfL, 0xb492a779L, 0xf2f0f307L, 0xe2a14e69L,
  0xf4cd65daL, 0xbed50605L, 0x621fd134L, 0xfe8ac4a6L,
  0x539d342eL, 0x55a0a2f3L, 0xe132058aL, 0xeb75a4f6L,
  0xec390b83L, 0xefaa4060L, 0x9f065e71L, 0x1051bd6eL,
  0x8af93e21L, 0x063d96ddL, 0x05aedd3eL, 0xbd464de6L,
  0x8db59154L, 0x5d0571c4L, 0xd46f0406L, 0x15ff6050L,
  0xfb241998L, 0xe997d6bdL, 0x43cc8940L, 0x9e7767d9L,
  0x42bdb0e8L, 0x8b880789L, 0x5b38e719L, 0xeedb79c8L,
  0x0a47a17cL, 0x0fe97c42L, 0x1ec9f884L, 0x00000000L,
  0x86830980L, 0xed48322bL, 0x70ac1e11L, 0x724e6c5aL,
  0xfffbfd0eL, 0x38560f85L, 0xd51e3daeL, 0x3927362dL,
  0xd9640a0fL, 0xa621685cL, 0x54d19b5bL, 0x2e3a2436L,
  0x67b10c0aL, 0xe70f9357L, 0x96d2b4eeL, 0x919e1b9bL,
  0xc54f80c0L, 0x20a261dcL, 0x4b695a77L, 0x1a161c12L,
  0xba0ae293L, 0x2ae5c0a0L, 0xe0433c22L, 0x171d121bL,
  0x0d0b0e09L, 0xc7adf28bL, 0xa8b92db6L, 0xa9c8141eL,
  0x198557f1L, 0x074caf75L, 0xddbbee99L, 0x60fda37fL,
  0x269ff701L, 0xf5bc5c72L, 0x3bc54466L, 0x7e345bfbL,
  0x29768b43L, 0xc6dccb23L, 0xfc68b6edL, 0xf163b8e4L,
  0xdccad731L, 0x85104263L, 0x22401397L, 0x112084c6L,
  0x247d854aL, 0x3df8d2bbL, 0x3211aef9L, 0xa16dc729L,
  0x2f4b1d9eL, 0x30f3dcb2L, 0x52ec0d86L, 0xe3d077c1L,
  0x166c2bb3L, 0xb999a970L, 0x48fa1194L, 0x642247e9L,
  0x8cc4a8fcL, 0x3f1aa0f0L, 0x2cd8567dL, 0x90ef2233L,
  0x4ec78749L, 0xd1c1d938L, 0xa2fe8ccaL, 0x0b3698d4L,
  0x81cfa6f5L, 0xde28a57aL, 0x8e26dab7L, 0xbfa43fadL,
  0x9de42c3aL, 0x920d5078L, 0xcc9b6a5fL, 0x4662547eL,
  0x13c2f68dL, 0xb8e890d8L, 0xf75e2e39L, 0xaff582c3L,
  0x80be9f5dL, 0x937c69d0L, 0x2da96fd5L, 0x12b3cf25L,
  0x993bc8acL, 0x7da71018L, 0x636ee89cL, 0xbb7bdb3bL,
  0x7809cd26L, 0x18f46e59L, 0xb701ec9aL, 0x9aa8834fL,
  0x6e65e695L, 0xe67eaaffL, 0xcf0821bcL, 0xe8e6ef15L,
  0x9bd9bae7L, 0x36ce4a6fL, 0x09d4ea9fL, 0x7cd629b0L,
  0xb2af31a4L, 0x23312a3fL, 0x9430c6a5L, 0x66c035a2L,
  0xbc37744eL, 0xcaa6fc82L, 0xd0b0e090L, 0xd81533a7L,
  0x984af104L, 0xdaf741ecL, 0x500e7fcdL, 0xf62f1791L,
  0xd68d764dL, 0xb04d43efL, 0x4d54ccaaL, 0x04dfe496L,
  0xb5e39ed1L, 0x881b4c6aL, 0x1fb8c12cL, 0x517f4665L,
  0xea049d5eL, 0x355d018cL, 0x7473fa87L, 0x412efb0bL,
  0x1d5ab367L, 0xd25292dbL, 0x5633e910L, 0x47136dd6L,
  0x618c9ad7L, 0x0c7a37a1L, 0x148e59f8L, 0x3c89eb13L,
  0x27eecea9L, 0xc935b761L, 0xe5ede11cL, 0xb13c7a47L,
  0xdf599cd2L, 0x733f55f2L, 0xce791814L, 0x37bf73c7L,
  0xcdea53f7L, 0xaa5b5ffdL, 0x6f14df3dL, 0xdb867844L,
  0xf381caafL, 0xc43eb968L, 0x342c3824L, 0x405fc2a3L,
  0xc372161dL, 0x250cbce2L, 0x498b283cL, 0x9541ff0dL,
  0x017139a8L, 0xb3de080cL, 0xe49cd8b4L, 0xc1906456L,
  0x84617bcbL, 0xb670d532L, 0x5c74486cL, 0x5742d0b8L,
  ]

Td3 = [
  0xf4a75051L, 0x4165537eL, 0x17a4c31aL, 0x275e963aL,
  0xab6bcb3bL, 0x9d45f11fL, 0xfa58abacL, 0xe303934bL,
  0x30fa5520L, 0x766df6adL, 0xcc769188L, 0x024c25f5L,
  0xe5d7fc4fL, 0x2acbd7c5L, 0x35448026L, 0x62a38fb5L,
  0xb15a49deL, 0xba1b6725L, 0xea0e9845L, 0xfec0e15dL,
  0x2f7502c3L, 0x4cf01281L, 0x4697a38dL, 0xd3f9c66bL,
  0x8f5fe703L, 0x929c9515L, 0x6d7aebbfL, 0x5259da95L,
  0xbe832dd4L, 0x7421d358L, 0xe0692949L, 0xc9c8448eL,
  0xc2896a75L, 0x8e7978f4L, 0x583e6b99L, 0xb971dd27L,
  0xe14fb6beL, 0x88ad17f0L, 0x20ac66c9L, 0xce3ab47dL,
  0xdf4a1863L, 0x1a3182e5L, 0x51336097L, 0x537f4562L,
  0x6477e0b1L, 0x6bae84bbL, 0x81a01cfeL, 0x082b94f9L,
  0x48685870L, 0x45fd198fL, 0xde6c8794L, 0x7bf8b752L,
  0x73d323abL, 0x4b02e272L, 0x1f8f57e3L, 0x55ab2a66L,
  0xeb2807b2L, 0xb5c2032fL, 0xc57b9a86L, 0x3708a5d3L,
  0x2887f230L, 0xbfa5b223L, 0x036aba02L, 0x16825cedL,
  0xcf1c2b8aL, 0x79b492a7L, 0x07f2f0f3L, 0x69e2a14eL,
  0xdaf4cd65L, 0x05bed506L, 0x34621fd1L, 0xa6fe8ac4L,
  0x2e539d34L, 0xf355a0a2L, 0x8ae13205L, 0xf6eb75a4L,
  0x83ec390bL, 0x60efaa40L, 0x719f065eL, 0x6e1051bdL,
  0x218af93eL, 0xdd063d96L, 0x3e05aeddL, 0xe6bd464dL,
  0x548db591L, 0xc45d0571L, 0x06d46f04L, 0x5015ff60L,
  0x98fb2419L, 0xbde997d6L, 0x4043cc89L, 0xd99e7767L,
  0xe842bdb0L, 0x898b8807L, 0x195b38e7L, 0xc8eedb79L,
  0x7c0a47a1L, 0x420fe97cL, 0x841ec9f8L, 0x00000000L,
  0x80868309L, 0x2bed4832L, 0x1170ac1eL, 0x5a724e6cL,
  0x0efffbfdL, 0x8538560fL, 0xaed51e3dL, 0x2d392736L,
  0x0fd9640aL, 0x5ca62168L, 0x5b54d19bL, 0x362e3a24L,
  0x0a67b10cL, 0x57e70f93L, 0xee96d2b4L, 0x9b919e1bL,
  0xc0c54f80L, 0xdc20a261L, 0x774b695aL, 0x121a161cL,
  0x93ba0ae2L, 0xa02ae5c0L, 0x22e0433cL, 0x1b171d12L,
  0x090d0b0eL, 0x8bc7adf2L, 0xb6a8b92dL, 0x1ea9c814L,
  0xf1198557L, 0x75074cafL, 0x99ddbbeeL, 0x7f60fda3L,
  0x01269ff7L, 0x72f5bc5cL, 0x663bc544L, 0xfb7e345bL,
  0x4329768bL, 0x23c6dccbL, 0xedfc68b6L, 0xe4f163b8L,
  0x31dccad7L, 0x63851042L, 0x97224013L, 0xc6112084L,
  0x4a247d85L, 0xbb3df8d2L, 0xf93211aeL, 0x29a16dc7L,
  0x9e2f4b1dL, 0xb230f3dcL, 0x8652ec0dL, 0xc1e3d077L,
  0xb3166c2bL, 0x70b999a9L, 0x9448fa11L, 0xe9642247L,
  0xfc8cc4a8L, 0xf03f1aa0L, 0x7d2cd856L, 0x3390ef22L,
  0x494ec787L, 0x38d1c1d9L, 0xcaa2fe8cL, 0xd40b3698L,
  0xf581cfa6L, 0x7ade28a5L, 0xb78e26daL, 0xadbfa43fL,
  0x3a9de42cL, 0x78920d50L, 0x5fcc9b6aL, 0x7e466254L,
  0x8d13c2f6L, 0xd8b8e890L, 0x39f75e2eL, 0xc3aff582L,
  0x5d80be9fL, 0xd0937c69L, 0xd52da96fL, 0x2512b3cfL,
  0xac993bc8L, 0x187da710L, 0x9c636ee8L, 0x3bbb7bdbL,
  0x267809cdL, 0x5918f46eL, 0x9ab701ecL, 0x4f9aa883L,
  0x956e65e6L, 0xffe67eaaL, 0xbccf0821L, 0x15e8e6efL,
  0xe79bd9baL, 0x6f36ce4aL, 0x9f09d4eaL, 0xb07cd629L,
  0xa4b2af31L, 0x3f23312aL, 0xa59430c6L, 0xa266c035L,
  0x4ebc3774L, 0x82caa6fcL, 0x90d0b0e0L, 0xa7d81533L,
  0x04984af1L, 0xecdaf741L, 0xcd500e7fL, 0x91f62f17L,
  0x4dd68d76L, 0xefb04d43L, 0xaa4d54ccL, 0x9604dfe4L,
  0xd1b5e39eL, 0x6a881b4cL, 0x2c1fb8c1L, 0x65517f46L,
  0x5eea049dL, 0x8c355d01L, 0x877473faL, 0x0b412efbL,
  0x671d5ab3L, 0xdbd25292L, 0x105633e9L, 0xd647136dL,
  0xd7618c9aL, 0xa10c7a37L, 0xf8148e59L, 0x133c89ebL,
  0xa927eeceL, 0x61c935b7L, 0x1ce5ede1L, 0x47b13c7aL,
  0xd2df599cL, 0xf2733f55L, 0x14ce7918L, 0xc737bf73L,
  0xf7cdea53L, 0xfdaa5b5fL, 0x3d6f14dfL, 0x44db8678L,
  0xaff381caL, 0x68c43eb9L, 0x24342c38L, 0xa3405fc2L,
  0x1dc37216L, 0xe2250cbcL, 0x3c498b28L, 0x0d9541ffL,
  0xa8017139L, 0x0cb3de08L, 0xb4e49cd8L, 0x56c19064L,
  0xcb84617bL, 0x32b670d5L, 0x6c5c7448L, 0xb85742d0L,
  ]

Td4 = [
  0x52525252L, 0x09090909L, 0x6a6a6a6aL, 0xd5d5d5d5L,
  0x30303030L, 0x36363636L, 0xa5a5a5a5L, 0x38383838L,
  0xbfbfbfbfL, 0x40404040L, 0xa3a3a3a3L, 0x9e9e9e9eL,
  0x81818181L, 0xf3f3f3f3L, 0xd7d7d7d7L, 0xfbfbfbfbL,
  0x7c7c7c7cL, 0xe3e3e3e3L, 0x39393939L, 0x82828282L,
  0x9b9b9b9bL, 0x2f2f2f2fL, 0xffffffffL, 0x87878787L,
  0x34343434L, 0x8e8e8e8eL, 0x43434343L, 0x44444444L,
  0xc4c4c4c4L, 0xdedededeL, 0xe9e9e9e9L, 0xcbcbcbcbL,
  0x54545454L, 0x7b7b7b7bL, 0x94949494L, 0x32323232L,
  0xa6a6a6a6L, 0xc2c2c2c2L, 0x23232323L, 0x3d3d3d3dL,
  0xeeeeeeeeL, 0x4c4c4c4cL, 0x95959595L, 0x0b0b0b0bL,
  0x42424242L, 0xfafafafaL, 0xc3c3c3c3L, 0x4e4e4e4eL,
  0x08080808L, 0x2e2e2e2eL, 0xa1a1a1a1L, 0x66666666L,
  0x28282828L, 0xd9d9d9d9L, 0x24242424L, 0xb2b2b2b2L,
  0x76767676L, 0x5b5b5b5bL, 0xa2a2a2a2L, 0x49494949L,
  0x6d6d6d6dL, 0x8b8b8b8bL, 0xd1d1d1d1L, 0x25252525L,
  0x72727272L, 0xf8f8f8f8L, 0xf6f6f6f6L, 0x64646464L,
  0x86868686L, 0x68686868L, 0x98989898L, 0x16161616L,
  0xd4d4d4d4L, 0xa4a4a4a4L, 0x5c5c5c5cL, 0xccccccccL,
  0x5d5d5d5dL, 0x65656565L, 0xb6b6b6b6L, 0x92929292L,
  0x6c6c6c6cL, 0x70707070L, 0x48484848L, 0x50505050L,
  0xfdfdfdfdL, 0xededededL, 0xb9b9b9b9L, 0xdadadadaL,
  0x5e5e5e5eL, 0x15151515L, 0x46464646L, 0x57575757L,
  0xa7a7a7a7L, 0x8d8d8d8dL, 0x9d9d9d9dL, 0x84848484L,
  0x90909090L, 0xd8d8d8d8L, 0xababababL, 0x00000000L,
  0x8c8c8c8cL, 0xbcbcbcbcL, 0xd3d3d3d3L, 0x0a0a0a0aL,
  0xf7f7f7f7L, 0xe4e4e4e4L, 0x58585858L, 0x05050505L,
  0xb8b8b8b8L, 0xb3b3b3b3L, 0x45454545L, 0x06060606L,
  0xd0d0d0d0L, 0x2c2c2c2cL, 0x1e1e1e1eL, 0x8f8f8f8fL,
  0xcacacacaL, 0x3f3f3f3fL, 0x0f0f0f0fL, 0x02020202L,
  0xc1c1c1c1L, 0xafafafafL, 0xbdbdbdbdL, 0x03030303L,
  0x01010101L, 0x13131313L, 0x8a8a8a8aL, 0x6b6b6b6bL,
  0x3a3a3a3aL, 0x91919191L, 0x11111111L, 0x41414141L,
  0x4f4f4f4fL, 0x67676767L, 0xdcdcdcdcL, 0xeaeaeaeaL,
  0x97979797L, 0xf2f2f2f2L, 0xcfcfcfcfL, 0xcecececeL,
  0xf0f0f0f0L, 0xb4b4b4b4L, 0xe6e6e6e6L, 0x73737373L,
  0x96969696L, 0xacacacacL, 0x74747474L, 0x22222222L,
  0xe7e7e7e7L, 0xadadadadL, 0x35353535L, 0x85858585L,
  0xe2e2e2e2L, 0xf9f9f9f9L, 0x37373737L, 0xe8e8e8e8L,
  0x1c1c1c1cL, 0x75757575L, 0xdfdfdfdfL, 0x6e6e6e6eL,
  0x47474747L, 0xf1f1f1f1L, 0x1a1a1a1aL, 0x71717171L,
  0x1d1d1d1dL, 0x29292929L, 0xc5c5c5c5L, 0x89898989L,
  0x6f6f6f6fL, 0xb7b7b7b7L, 0x62626262L, 0x0e0e0e0eL,
  0xaaaaaaaaL, 0x18181818L, 0xbebebebeL, 0x1b1b1b1bL,
  0xfcfcfcfcL, 0x56565656L, 0x3e3e3e3eL, 0x4b4b4b4bL,
  0xc6c6c6c6L, 0xd2d2d2d2L, 0x79797979L, 0x20202020L,
  0x9a9a9a9aL, 0xdbdbdbdbL, 0xc0c0c0c0L, 0xfefefefeL,
  0x78787878L, 0xcdcdcdcdL, 0x5a5a5a5aL, 0xf4f4f4f4L,
  0x1f1f1f1fL, 0xddddddddL, 0xa8a8a8a8L, 0x33333333L,
  0x88888888L, 0x07070707L, 0xc7c7c7c7L, 0x31313131L,
  0xb1b1b1b1L, 0x12121212L, 0x10101010L, 0x59595959L,
  0x27272727L, 0x80808080L, 0xececececL, 0x5f5f5f5fL,
  0x60606060L, 0x51515151L, 0x7f7f7f7fL, 0xa9a9a9a9L,
  0x19191919L, 0xb5b5b5b5L, 0x4a4a4a4aL, 0x0d0d0d0dL,
  0x2d2d2d2dL, 0xe5e5e5e5L, 0x7a7a7a7aL, 0x9f9f9f9fL,
  0x93939393L, 0xc9c9c9c9L, 0x9c9c9c9cL, 0xefefefefL,
  0xa0a0a0a0L, 0xe0e0e0e0L, 0x3b3b3b3bL, 0x4d4d4d4dL,
  0xaeaeaeaeL, 0x2a2a2a2aL, 0xf5f5f5f5L, 0xb0b0b0b0L,
  0xc8c8c8c8L, 0xebebebebL, 0xbbbbbbbbL, 0x3c3c3c3cL,
  0x83838383L, 0x53535353L, 0x99999999L, 0x61616161L,
  0x17171717L, 0x2b2b2b2bL, 0x04040404L, 0x7e7e7e7eL,
  0xbabababaL, 0x77777777L, 0xd6d6d6d6L, 0x26262626L,
  0xe1e1e1e1L, 0x69696969L, 0x14141414L, 0x63636363L,
  0x55555555L, 0x21212121L, 0x0c0c0c0cL, 0x7d7d7d7dL,
  ]

rcon = [
  0x01000000, 0x02000000, 0x04000000, 0x08000000,
  0x10000000, 0x20000000, 0x40000000, 0x80000000,
  0x1B000000, 0x36000000,
  # 128-bit blocks, Rijndael never uses more than 10 rcon values
  ]

if len(pack('L',0)) == 4:
    # 32bit
    def GETU32(x): return unpack('>L', x)[0]
    def PUTU32(x): return pack('>L', x)
else:
    # 64bit
    def GETU32(x): return unpack('>I', x)[0]
    def PUTU32(x): return pack('>I', x)

# Expand the cipher key into the encryption key schedule.
#
# @return the number of rounds for the given cipher key size.
def rijndaelSetupEncrypt(key, keybits):
    i = p = 0
    rk = [0]*RKLENGTH(keybits)
    rk[0] = GETU32(key[0:4])
    rk[1] = GETU32(key[4:8])
    rk[2] = GETU32(key[8:12])
    rk[3] = GETU32(key[12:16])
    if keybits == 128:
        while 1:
            temp = rk[p+3]
            rk[p+4] = (rk[p+0] ^
                       (Te4[(temp >> 16) & 0xff] & 0xff000000) ^
                       (Te4[(temp >>  8) & 0xff] & 0x00ff0000) ^
                       (Te4[(temp      ) & 0xff] & 0x0000ff00) ^
                       (Te4[(temp >> 24)       ] & 0x000000ff) ^
                       rcon[i])
            rk[p+5] = rk[p+1] ^ rk[p+4]
            rk[p+6] = rk[p+2] ^ rk[p+5]
            rk[p+7] = rk[p+3] ^ rk[p+6]
            i += 1
            if i == 10: return (rk, 10)
            p += 4

    rk[4] = GETU32(key[16:20])
    rk[5] = GETU32(key[20:24])
    if keybits == 192:
        while 1:
            temp = rk[p+5]
            rk[p+6] = (rk[p+0] ^
                       (Te4[(temp >> 16) & 0xff] & 0xff000000) ^
                       (Te4[(temp >>  8) & 0xff] & 0x00ff0000) ^
                       (Te4[(temp      ) & 0xff] & 0x0000ff00) ^
                       (Te4[(temp >> 24)       ] & 0x000000ff) ^
                       rcon[i])
            rk[p+7] = rk[p+1] ^ rk[p+6]
            rk[p+8] = rk[p+2] ^ rk[p+7]
            rk[p+9] = rk[p+3] ^ rk[p+8]
            i += 1
            if i == 8: return (rk, 12)
            rk[p+10] = rk[p+4] ^ rk[p+9]
            rk[p+11] = rk[p+5] ^ rk[p+10]
            p += 6

    rk[6] = GETU32(key[24:28])
    rk[7] = GETU32(key[28:32])
    if keybits == 256:
        while 1:
            temp = rk[p+7]
            rk[p+8] = (rk[p+0] ^
                       (Te4[(temp >> 16) & 0xff] & 0xff000000) ^
                       (Te4[(temp >>  8) & 0xff] & 0x00ff0000) ^
                       (Te4[(temp      ) & 0xff] & 0x0000ff00) ^
                       (Te4[(temp >> 24)       ] & 0x000000ff) ^
                       rcon[i])
            rk[p+9] = rk[p+1] ^ rk[p+8]
            rk[p+10] = rk[p+2] ^ rk[p+9]
            rk[p+11] = rk[p+3] ^ rk[p+10]
            i += 1
            if i == 7: return (rk, 14)
            temp = rk[p+11]
            rk[p+12] = (rk[p+4] ^
                        (Te4[(temp >> 24)       ] & 0xff000000) ^
                        (Te4[(temp >> 16) & 0xff] & 0x00ff0000) ^
                        (Te4[(temp >>  8) & 0xff] & 0x0000ff00) ^
                        (Te4[(temp      ) & 0xff] & 0x000000ff))
            rk[p+13] = rk[p+5] ^ rk[p+12]
            rk[p+14] = rk[p+6] ^ rk[p+13]
            rk[p+15] = rk[p+7] ^ rk[p+14]
            p += 8

    raise ValueError(keybits)


# Expand the cipher key into the decryption key schedule.
#
# @return the number of rounds for the given cipher key size.
def rijndaelSetupDecrypt(key, keybits):

    # expand the cipher key:
    (rk, nrounds) = rijndaelSetupEncrypt(key, keybits)
    # invert the order of the round keys:
    i = 0
    j = 4*nrounds
    while i < j:
        temp = rk[i    ]; rk[i    ] = rk[j    ]; rk[j    ] = temp
        temp = rk[i + 1]; rk[i + 1] = rk[j + 1]; rk[j + 1] = temp
        temp = rk[i + 2]; rk[i + 2] = rk[j + 2]; rk[j + 2] = temp
        temp = rk[i + 3]; rk[i + 3] = rk[j + 3]; rk[j + 3] = temp
        i += 4
        j -= 4
    # apply the inverse MixColumn transform to all round keys but the first and the last:
    p = 0
    for i in xrange(1, nrounds):
        p += 4
        rk[p+0] = (
          Td0[Te4[(rk[p+0] >> 24)       ] & 0xff] ^
          Td1[Te4[(rk[p+0] >> 16) & 0xff] & 0xff] ^
          Td2[Te4[(rk[p+0] >>  8) & 0xff] & 0xff] ^
          Td3[Te4[(rk[p+0]      ) & 0xff] & 0xff])
        rk[p+1] = (
          Td0[Te4[(rk[p+1] >> 24)       ] & 0xff] ^
          Td1[Te4[(rk[p+1] >> 16) & 0xff] & 0xff] ^
          Td2[Te4[(rk[p+1] >>  8) & 0xff] & 0xff] ^
          Td3[Te4[(rk[p+1]      ) & 0xff] & 0xff])
        rk[p+2] = (
          Td0[Te4[(rk[p+2] >> 24)       ] & 0xff] ^
          Td1[Te4[(rk[p+2] >> 16) & 0xff] & 0xff] ^
          Td2[Te4[(rk[p+2] >>  8) & 0xff] & 0xff] ^
          Td3[Te4[(rk[p+2]      ) & 0xff] & 0xff])
        rk[p+3] = (
          Td0[Te4[(rk[p+3] >> 24)       ] & 0xff] ^
          Td1[Te4[(rk[p+3] >> 16) & 0xff] & 0xff] ^
          Td2[Te4[(rk[p+3] >>  8) & 0xff] & 0xff] ^
          Td3[Te4[(rk[p+3]      ) & 0xff] & 0xff])

    return (rk, nrounds)


def rijndaelEncrypt(rk, nrounds, plaintext):
    assert len(plaintext) == 16

    # map byte array block to cipher state
    # and add initial round key:
    s0 = GETU32(plaintext[0:4]) ^ rk[0]
    s1 = GETU32(plaintext[4:8]) ^ rk[1]
    s2 = GETU32(plaintext[8:12]) ^ rk[2]
    s3 = GETU32(plaintext[12:16]) ^ rk[3]

    # nrounds - 1 full rounds:
    r = nrounds >> 1
    p = 0
    while 1:
        t0 = (
          Te0[(s0 >> 24)       ] ^
          Te1[(s1 >> 16) & 0xff] ^
          Te2[(s2 >>  8) & 0xff] ^
          Te3[(s3      ) & 0xff] ^
          rk[p+4])
        t1 = (
          Te0[(s1 >> 24)       ] ^
          Te1[(s2 >> 16) & 0xff] ^
          Te2[(s3 >>  8) & 0xff] ^
          Te3[(s0      ) & 0xff] ^
          rk[p+5])
        t2 = (
          Te0[(s2 >> 24)       ] ^
          Te1[(s3 >> 16) & 0xff] ^
          Te2[(s0 >>  8) & 0xff] ^
          Te3[(s1      ) & 0xff] ^
          rk[p+6])
        t3 = (
          Te0[(s3 >> 24)       ] ^
          Te1[(s0 >> 16) & 0xff] ^
          Te2[(s1 >>  8) & 0xff] ^
          Te3[(s2      ) & 0xff] ^
          rk[p+7])
        p += 8
        r -= 1
        if r == 0: break
        s0 = (
          Te0[(t0 >> 24)       ] ^
          Te1[(t1 >> 16) & 0xff] ^
          Te2[(t2 >>  8) & 0xff] ^
          Te3[(t3      ) & 0xff] ^
          rk[p+0])
        s1 = (
          Te0[(t1 >> 24)       ] ^
          Te1[(t2 >> 16) & 0xff] ^
          Te2[(t3 >>  8) & 0xff] ^
          Te3[(t0      ) & 0xff] ^
          rk[p+1])
        s2 = (
          Te0[(t2 >> 24)       ] ^
          Te1[(t3 >> 16) & 0xff] ^
          Te2[(t0 >>  8) & 0xff] ^
          Te3[(t1      ) & 0xff] ^
          rk[p+2])
        s3 = (
          Te0[(t3 >> 24)       ] ^
          Te1[(t0 >> 16) & 0xff] ^
          Te2[(t1 >>  8) & 0xff] ^
          Te3[(t2      ) & 0xff] ^
          rk[p+3])

    ciphertext = ''

    # apply last round and
    # map cipher state to byte array block:
    s0 = (
      (Te4[(t0 >> 24)       ] & 0xff000000) ^
      (Te4[(t1 >> 16) & 0xff] & 0x00ff0000) ^
      (Te4[(t2 >>  8) & 0xff] & 0x0000ff00) ^
      (Te4[(t3      ) & 0xff] & 0x000000ff) ^
      rk[p+0])
    ciphertext += PUTU32(s0)
    s1 = (
      (Te4[(t1 >> 24)       ] & 0xff000000) ^
      (Te4[(t2 >> 16) & 0xff] & 0x00ff0000) ^
      (Te4[(t3 >>  8) & 0xff] & 0x0000ff00) ^
      (Te4[(t0      ) & 0xff] & 0x000000ff) ^
      rk[p+1])
    ciphertext += PUTU32(s1)
    s2 = (
      (Te4[(t2 >> 24)       ] & 0xff000000) ^
      (Te4[(t3 >> 16) & 0xff] & 0x00ff0000) ^
      (Te4[(t0 >>  8) & 0xff] & 0x0000ff00) ^
      (Te4[(t1      ) & 0xff] & 0x000000ff) ^
      rk[p+2])
    ciphertext += PUTU32(s2)
    s3 = (
      (Te4[(t3 >> 24)       ] & 0xff000000) ^
      (Te4[(t0 >> 16) & 0xff] & 0x00ff0000) ^
      (Te4[(t1 >>  8) & 0xff] & 0x0000ff00) ^
      (Te4[(t2      ) & 0xff] & 0x000000ff) ^
      rk[p+3])
    ciphertext += PUTU32(s3)

    assert len(ciphertext) == 16
    return ciphertext


def rijndaelDecrypt(rk, nrounds, ciphertext):
    assert len(ciphertext) == 16

    # map byte array block to cipher state
    # and add initial round key:
    s0 = GETU32(ciphertext[0:4]) ^ rk[0]
    s1 = GETU32(ciphertext[4:8]) ^ rk[1]
    s2 = GETU32(ciphertext[8:12]) ^ rk[2]
    s3 = GETU32(ciphertext[12:16]) ^ rk[3]

    # nrounds - 1 full rounds:
    r = nrounds >> 1
    p = 0
    while 1:
        t0 = (
          Td0[(s0 >> 24)       ] ^
          Td1[(s3 >> 16) & 0xff] ^
          Td2[(s2 >>  8) & 0xff] ^
          Td3[(s1      ) & 0xff] ^
          rk[p+4])
        t1 = (
          Td0[(s1 >> 24)       ] ^
          Td1[(s0 >> 16) & 0xff] ^
          Td2[(s3 >>  8) & 0xff] ^
          Td3[(s2      ) & 0xff] ^
          rk[p+5])
        t2 = (
          Td0[(s2 >> 24)       ] ^
          Td1[(s1 >> 16) & 0xff] ^
          Td2[(s0 >>  8) & 0xff] ^
          Td3[(s3      ) & 0xff] ^
          rk[p+6])
        t3 = (
          Td0[(s3 >> 24)       ] ^
          Td1[(s2 >> 16) & 0xff] ^
          Td2[(s1 >>  8) & 0xff] ^
          Td3[(s0      ) & 0xff] ^
          rk[p+7])
        p += 8
        r -= 1
        if r == 0: break
        s0 = (
          Td0[(t0 >> 24)       ] ^
          Td1[(t3 >> 16) & 0xff] ^
          Td2[(t2 >>  8) & 0xff] ^
          Td3[(t1      ) & 0xff] ^
          rk[p+0])
        s1 = (
          Td0[(t1 >> 24)       ] ^
          Td1[(t0 >> 16) & 0xff] ^
          Td2[(t3 >>  8) & 0xff] ^
          Td3[(t2      ) & 0xff] ^
          rk[p+1])
        s2 = (
          Td0[(t2 >> 24)       ] ^
          Td1[(t1 >> 16) & 0xff] ^
          Td2[(t0 >>  8) & 0xff] ^
          Td3[(t3      ) & 0xff] ^
          rk[p+2])
        s3 = (
          Td0[(t3 >> 24)       ] ^
          Td1[(t2 >> 16) & 0xff] ^
          Td2[(t1 >>  8) & 0xff] ^
          Td3[(t0      ) & 0xff] ^
          rk[p+3])

    plaintext = ''

    # apply last round and
    # map cipher state to byte array block:
    s0 = (
      (Td4[(t0 >> 24)       ] & 0xff000000) ^
      (Td4[(t3 >> 16) & 0xff] & 0x00ff0000) ^
      (Td4[(t2 >>  8) & 0xff] & 0x0000ff00) ^
      (Td4[(t1      ) & 0xff] & 0x000000ff) ^
      rk[p+0])
    plaintext += PUTU32(s0)
    s1 = (
      (Td4[(t1 >> 24)       ] & 0xff000000) ^
      (Td4[(t0 >> 16) & 0xff] & 0x00ff0000) ^
      (Td4[(t3 >>  8) & 0xff] & 0x0000ff00) ^
      (Td4[(t2      ) & 0xff] & 0x000000ff) ^
      rk[p+1])
    plaintext += PUTU32(s1)
    s2 = (
      (Td4[(t2 >> 24)       ] & 0xff000000) ^
      (Td4[(t1 >> 16) & 0xff] & 0x00ff0000) ^
      (Td4[(t0 >>  8) & 0xff] & 0x0000ff00) ^
      (Td4[(t3      ) & 0xff] & 0x000000ff) ^
      rk[p+2])
    plaintext += PUTU32(s2)
    s3 = (
      (Td4[(t3 >> 24)       ] & 0xff000000) ^
      (Td4[(t2 >> 16) & 0xff] & 0x00ff0000) ^
      (Td4[(t1 >>  8) & 0xff] & 0x0000ff00) ^
      (Td4[(t0      ) & 0xff] & 0x000000ff) ^
      rk[p+3])
    plaintext += PUTU32(s3)

    assert len(plaintext) == 16
    return plaintext


# decrypt(key, fin, fout, keybits=256)
class RijndaelDecryptor(object):

    """
    >>> key = '00010203050607080a0b0c0d0f101112'.decode('hex')
    >>> ciphertext = 'd8f532538289ef7d06b506a4fd5be9c9'.decode('hex')
    >>> RijndaelDecryptor(key, 128).decrypt(ciphertext).encode('hex')
    '506812a45f08c889b97f5980038b8359'
    """

    def __init__(self, key, keybits=256):
        assert len(key) == KEYLENGTH(keybits)
        (self.rk, self.nrounds) = rijndaelSetupDecrypt(key, keybits)
        assert len(self.rk) == RKLENGTH(keybits)
        assert self.nrounds == NROUNDS(keybits)
        return

    def decrypt(self, ciphertext):
        assert len(ciphertext) == 16
        return rijndaelDecrypt(self.rk, self.nrounds, ciphertext)

# encrypt(key, fin, fout, keybits=256)
class RijndaelEncryptor(object):

    """
    >>> key = '00010203050607080a0b0c0d0f101112'.decode('hex')
    >>> plaintext = '506812a45f08c889b97f5980038b8359'.decode('hex')
    >>> RijndaelEncryptor(key, 128).encrypt(plaintext).encode('hex')
    'd8f532538289ef7d06b506a4fd5be9c9'
    """
    
    def __init__(self, key, keybits=256):
        assert len(key) == KEYLENGTH(keybits)
        (self.rk, self.nrounds) = rijndaelSetupEncrypt(key, keybits)
        assert len(self.rk) == RKLENGTH(keybits)
        assert self.nrounds == NROUNDS(keybits)
        return

    def encrypt(self, plaintext):
        assert len(plaintext) == 16
        return rijndaelEncrypt(self.rk, self.nrounds, plaintext)


if __name__ == '__main__':
    import doctest
    doctest.testmod()
