#!/usr/bin/env python3
from operator import itemgetter

from io import BytesIO

from netaddr.ip import IPNetwork
from pathlib import Path
import pequod_cli
from pequod_cli.console import action, ok, error, print_table
from pequod_cli.context import Context
from pequod_cli.service import CORE_PORTS
from pequod_cli.utils import generate_random_name, AliasedGroup
import click
import collections
import logging
import os
import random
import requests
import tarfile
import time
import yaml

DEFAULT_CONFIG_FILE='~/.pequod-cli.yaml'

CONTEXT_SETTINGS = dict(help_option_names=['-h', '--help'])

def print_version(ctx, param, value):
    if not value or ctx.resilient_parsing:
        return
    click.echo('Pequod CLI {}'.format(pequod_cli.__version__))
    ctx.exit()

def configure_logging(loglevel):
    # configure file logger to not clutter stdout with log lines
    logging.basicConfig(level=loglevel, filename='/tmp/pequod-cli.log', format='%(asctime)s %(levelname)s %(name)s: %(message)s')
    logging.getLogger('urllib3.connectionpool').setLevel(logging.WARNING)
    logging.getLogger('requests.packages.urllib3.connectionpool').setLevel(logging.WARNING)

@click.group(invoke_without_command=True, cls=AliasedGroup, context_settings=CONTEXT_SETTINGS)
@click.option('--config-file', help='Use alternative config file', default=DEFAULT_CONFIG_FILE, metavar='PATH')
@click.option('-c', '--cluster', envvar='PEQUOD_CLUSTER', help='Name of the cluster to use', default='default', metavar='NAME')
@click.option('-v', '--verbose', help='Verbose logging', is_flag=True)
@click.option('-V', '--version', is_flag=True, callback=print_version,  expose_value=False, is_eager=True)
@click.pass_context
def cli(ctx, config_file, cluster, verbose):
    """
    Pequod cluster command line interface
    """
    configure_logging(logging.DEBUG if verbose else logging.INFO)
    fn = os.path.expanduser(config_file)
    data = {}
    if os.path.exists(fn):
        with open(fn) as fd:
            data = yaml.safe_load(fd)
    if ctx.invoked_subcommand != 'login':
        if cluster not in data:
            raise click.UsageError('Cluster "{}" is not configured yet. Please use the "login" command or edit the configuration file {} manually.'.format(cluster, fn))
        endpoints = data[cluster]['endpoints']
        credentials = data[cluster]['credentials']
    else:
        endpoints = {}
        credentials = None
    ctx.obj = Context(fn, cluster, endpoints, credentials)
    if not ctx.invoked_subcommand:
        _state(ctx.obj)


class Endpoints(click.ParamType):
    name = 'Endpoints'

    def convert(self, value, param, ctx):
        if isinstance(value, dict):
            return value
        if value.startswith('{'):
            return yaml.safe_load(value)
        endpoints = {}
        parts = [v.strip() for v in value.split(',')]

        if parts[0].startswith('http'):
            for k, port in CORE_PORTS.items():
                endpoints[k] = [h.format(service=k) for h in parts]
        else:
            hosts = parts
            for k, port in CORE_PORTS.items():
                    endpoints[k] = ['http://{}:{}/'.format(h, port) for h in hosts]
        return endpoints

@cli.command()
@click.option('-e', '--endpoints', help='Endpoints', prompt='Service endpoints (list of IPs/URLs or complete JSON dict)', metavar='CONFIG', type=Endpoints())
@click.option('-u', '--username', help='Username', prompt='Username', metavar='USER')
@click.option('-p', '--password', help='Password', prompt=True, hide_input=True, metavar='PASSWD')
@click.pass_context
def login(click_context, endpoints, username, password):
    """
    Login to a cluster and store credentials
    """
    click.secho('Trying authenticating to core services..', bold=True, fg='blue')

    credentials = {'username': username, 'password': password}
    ctx = click_context.obj
    ctx = Context(ctx.config_file, ctx.cluster, endpoints, credentials)

    has_errors = False
    for svc in ctx.all_core_services:
        action('Checking {}..'.format(svc.name))
        try:
            svc.get('/status')
        except Exception as e:
            error('ERROR: {}'.format(e))
            has_errors = True
        else:
            ok()

    if has_errors:
        click.secho('WARNING: At least one core service could not be reached. Check your username/password and endpoint IPs.', bold=True, fg='yellow')

    try:
        with open(ctx.config_file) as fd:
            data = yaml.safe_load(fd)
    except FileNotFoundError:
        data = {}

    click.confirm('Save credentials for cluster {cluster} at {fn} in plaintext?'.format(cluster=ctx.cluster, fn=ctx.config_file), abort=True)

    action('Storing configuration for cluster {cluster} at {fn}..', cluster=ctx.cluster, fn=ctx.config_file)
    data[ctx.cluster] = {'endpoints': endpoints, 'credentials': credentials}

    with open(ctx.config_file, 'w') as fd:
        yaml.safe_dump(data, fd)
    ok()



@cli.command()
@click.pass_obj
def state(ctx):
    """
    Display the current cluster nodes and running application instances
    """
    _state(ctx)

@cli.command()
@click.pass_context
def dashboard(ctx):
    """
    Display the current cluster nodes and running application instances and updates every second
    """
    while True:
        click.clear()
        _state(ctx.obj)
        time.sleep(1)

def _state(ctx):
    rawdata = ctx.controller.get('/state')

    rows = []
    for node_name, node_data in sorted(rawdata['nodes'].items()):
        data = {'node_name': node_name, 'status': node_data['status'], 'reboot': False, 'maintenance': False}
        try:
            node_status = ctx.manager.get('/nodes/{}'.format(node_name))
            data['reboot'] = node_status['reboot']
            data['maintenance'] = node_status['maintenance']
        except requests.HTTPError as e:
            if e.response.status_code != 404:
                # ignore nodes not configure in core manager
                # TODO: better error handling
                logging.exception('Failed to contact core manager')
        data.update(node_data['resources'])
        data.update(node_data['metadata'])
        rows.append(data)

    COLS = 'node_name status memory_mb cpu_vmhz processes_count filehandles_count node_start_time agent_start_time template_version reboot maintenance'.split()
    print_table(COLS, rows)

    click.echo('')
    rows = []
    for instance_name, instance_data in sorted(rawdata['instances'].items()):
        data = {'instance_name': instance_name, 'node_name': instance_data['node_name'], 'lifecycle_phase': instance_data['lifecycle_phase']}
        data.update(instance_data['resources'] or {})
        data.update(instance_data['metadata'])
        rows.append(data)

    COLS = 'instance_name node_name repository_name application_name application_version zone_name lifecycle_phase memory_mb cpu_vmhz processes_count filehandles_count ipv6 instance_started_time'.split()
    print_table(COLS, rows)



ApplicationVersion = collections.namedtuple('ApplicationVersion', 'repository_name application_name application_version')

def parse_application_spec(spec):
    try:
        repo, app = spec.split('/')
        app, version = app.split(':')
    except:
        raise click.UsageError('Invalid application spec, must be <repo>/<app>:<version>')
    return ApplicationVersion(repository_name=repo, application_name=app, application_version=version)

class ApplicationSpec(click.ParamType):
    name = 'ApplicationSpec'

    def convert(self, value, param, ctx):
        if isinstance(value, ApplicationVersion):
            return value
        return parse_application_spec(value)

@cli.command()
@click.argument('app', metavar='REPO/APP:VERSION', type=ApplicationSpec())
@click.argument('zone_name')
@click.option('--node', help='Start instance on a specific node')
@click.option('--count', help='Number of instances to start', default=1, type=click.IntRange(1, None))
@click.option('--memory-mb', help='Memory in MB', type=click.IntRange(1, None))
@click.option('--cpu-vmhz', help='CPU in VMHz', type=click.IntRange(1, None))
@click.option('--processes-count', help='Processes', type=click.IntRange(1, None))
@click.pass_obj
def start(ctx, app, zone_name, node, count, **kwargs):
    """
    Start one or more application instances
    """
    manifest = ctx.registry.get_manifest(app.repository_name, app.application_name, app.application_version)

    if not manifest:
        raise click.UsageError('Application manifest not found: {}'.format(app))

    zones = ctx.manager.get_zones()
    if zone_name not in zones:
        raise click.UsageError('Zone does not exist: {}'.format(zone_name))

    if node:
        node_name = node
    else:
        rawdata = ctx.controller.get_state()
        node_candidates = []
        for _node_name, node_data in sorted(rawdata['nodes'].items()):
            if node_data['status'] == 'REGISTERED':
                node_candidates.append(_node_name)
        node_name = random.choice(node_candidates)
    resources = {}
    for key, val in manifest['resources'].items():
        default_key = [ k for k in val.keys() if k.startswith('default_') ][0]
        unit = default_key.split('_')[1]
        resources['{}_{}'.format(key, unit)] = val[default_key]
    for key in 'memory_mb', 'cpu_vmhz', 'processes_count':
        val = kwargs.get(key)
        if val:
            resources[key] = val
    action('Starting {} instances on {}..'.format(count, node_name))
    for i in range(count):
        data = {'actions_create': [{
                'instance_name': generate_random_name('{}-{}-{}-'.format(app.repository_name, app.application_name, app.application_version)[:24], 4),
                'node_name': node_name,
                'resources': resources,
                'metadata': {
                    'repository_name': app.repository_name,
                    'application_name': app.application_name,
                    'application_version': app.application_version,
                    'zone_name': zone_name,
                    }
                }], 'allow_partial_commit': False}
        result = ctx.controller.put('/scheduler', data=data)
    ok()

@cli.command()
@click.argument('instance_name')
@click.pass_obj
def stop(ctx, instance_name):
    """
    Stop a single application instance
    """
    rawdata = ctx.controller.get_state()
    node_name = None
    for _instance_name, instance_data in sorted(rawdata['instances'].items()):
        if _instance_name == instance_name:
            node_name = instance_data['node_name']
            break
    if node_name:
        action('Stopping instance {} on {}..'.format(instance_name, node_name))
        data = {'actions_kill': [{
                'instance_name': instance_name,
                'node_name': instance_data['node_name'],
                }], 'allow_partial_commit': False}
        result = ctx.controller.put('/scheduler', data=data)
        ok()


@cli.command()
@click.argument('node_name')
@click.pass_obj
def reboot(ctx, node_name):
    """
    Reboot a cluster app node
    """
    action('Rebooting {}..'.format(node_name))
    data = {'maintenance': False,
            'reboot': True
            }
    ctx.manager.patch('/nodes/{}'.format(node_name), data=data)
    ok()

@cli.group(invoke_without_command=True, cls=AliasedGroup)
@click.pass_context
def registry(ctx):
    """
    Interact with application registry, list all applications
    """
    if not ctx.invoked_subcommand:
        _tags(ctx.obj)

@cli.command()
@click.pass_obj
def status(ctx):
    """
    Ping all core cluster services
    """
    for svc in ctx.all_core_services:
        action('Checking {}..'.format(svc.name))
        has_errors = 0
        total = 0
        try:
            for ep, success, msg in svc.status():
                click.secho(' .', nl=False)
                if not success:
                    error(msg, nl=False)
                    has_errors += 1
                total += 1
        except Exception as e:
            error('ERROR: {}'.format(e))
        else:
            if has_errors:
                error('{}/{} FAILED'.format(has_errors, total))
            else:
                ok()


def _tags(ctx, query=None):
    data = ctx.registry.get('/tags')
    rows = []
    for row in data:
        if query and query not in row['id']:
            continue
        row['last_update_time'] = row['last_update'] / 1000
        row['command'] = ' '.join(row['container_setup']['command'])
        row['has_manifest'] = bool(row['container_setup']['manifest_image_id'])
        rows.append(row)

    rows.sort(key=lambda x: x['last_update_time'])

    COLS = 'repository_name application_name application_version command has_manifest last_update_time'.split()
    print_table(COLS, rows)

@registry.command()
@click.option('-q', '--query', help='List only tags matching the query')
@click.pass_obj
def tags(ctx, query):
    """
    Show all available tags/applications
    """
    _tags(ctx, query)

@registry.command()
@click.pass_obj
def services(ctx):
    """
    Show all available services and applications providing them
    """
    data = ctx.registry.get('/tags')
    rows = []
    for row in data:
        manifest = ctx.registry.get_manifest(row['repository_name'], row['application_name'], row['application_version'])
        if not manifest:
            continue
        for service in manifest['services'].get('provide', []):

            rows.append({'service_name': service['service_name'],
                         'service_version': service['service_version'],
                         'repository_name': row['repository_name'],
                         'application_name': row['application_name'],
                         'application_version': row['application_version'],
                         })

    rows.sort(key=lambda x: (x['service_name'], x['service_version'], x['repository_name'], x['application_name'], x['application_version']) )

    COLS = 'service_name service_version repository_name application_name application_version'.split()
    print_table(COLS, rows)

@registry.command()
@click.argument('app', metavar='REPO/APP:VERSION', type=ApplicationSpec())
@click.pass_obj
def manifest(ctx, app):
    data = ctx.registry.get_manifest(app.repository_name, app.application_name, app.application_version)
    click.echo(yaml.safe_dump(data))

@cli.group(cls=AliasedGroup)
def manager():
    """
    Interact with core manager
    """
    pass

@manager.group(invoke_without_command=True, cls=AliasedGroup)
@click.pass_context
def nodes(ctx):
    """
    Show all configured nodes
    """
    if not ctx.invoked_subcommand:
        data = ctx.obj.manager.get('/nodes')
        rows = []
        for row in data['nodes']:
            data = ctx.obj.manager.get('/nodes/{}'.format(row['name']))
            row.update(data)
            rows.append(row)
        rows.sort(key=itemgetter('name'))
        print_table('name mac ipv4 ipv6 template_name template_version reboot maintenance'.split(), rows)

def parse_version(v):
    return tuple([int(x) for x in v.split(".")])

@nodes.command()
@click.argument('node_name')
@click.argument('key_value', nargs=-1)
@click.pass_obj
def update(ctx, node_name, key_value):
    """
    Update node configuration (e.g. "template_version")
    """
    data = ctx.manager.get('/nodes/{}'.format(node_name))
    del data['reboot']
    del data['maintenance']
    for pair in key_value:
        try:
            key, val = pair.split('=', 1)
        except:
            raise click.UsageError('Key pairs need to be specified as KEY=VAL')
        if key not in data:
            raise click.UsageError('Invalid key: "{}" is not allowed'.format(key))
        if key == 'template_version' and val == 'latest':
            # special case: find out latest template and use it
            latest_version = '0'
            for row in ctx.manager.get_templates():
                if row['template_name'] == data['template_name']:
                    # TODO: only supports numeric versions right now :-P
                    try:
                        if parse_version(row['template_version']) > parse_version(latest_version):
                            latest_version = row['template_version']
                    except ValueError:
                        # ignore non-int versions
                        pass
            val = latest_version
        data[key] = val
    action('Updating {}..'.format(node_name))
    ctx.manager.put('/nodes/{}'.format(node_name), data=data)
    ok()

@manager.group(invoke_without_command=True, cls=AliasedGroup)
@click.pass_context
def templates(ctx):
    """
    Show all available templates
    """
    if not ctx.invoked_subcommand:
        rows = []
        for row in ctx.obj.manager.get_templates():
            rows.append(row)
        rows.sort(key=lambda x: (x['template_name'], x['template_version'], x['image_type']))
        print_table('template_name template_version image_type'.split(), rows)

@templates.command()
@click.argument("files", nargs=-1)
@click.pass_obj
def upload(ctx, files):
    """
    Upload template files
    """
    existing_templates = set([(x['template_name'], x['template_version'], x['image_type']) for x in ctx.manager.get_templates()])
    for fn in files:
        path = Path(fn)
        try:
            _, template_name, template_version, image_type = path.name.rsplit('-', 3)
        except ValueError:
            raise click.UsageError('Invalid file name "{}". Files must be named like "foobar-<TEMPLATE_NAME>-<TEMPLATE_VERSION>-<IMAGE_TYPE>".'.format(path.name))
        image_type = image_type.upper()
        identity = template_name, template_version, image_type
        if identity in existing_templates:
            raise click.UsageError('Template {}:{} {} already exists'.format(*identity))
        action('Uploading template {} version {} {}..'.format(*identity))
        with path.open('rb') as fd:
            ctx.manager.put('/templates/{}/{}/{}'.format(template_name, template_version, image_type), data=fd, headers={'Content-Type': 'application/octet-stream'}, timeout=300)
        ok()


@manager.group(invoke_without_command=True, cls=AliasedGroup)
@click.pass_context
def vaults(ctx):
    """
    Show all available vaults
    """
    if not ctx.invoked_subcommand:
        rows = []
        for row in ctx.obj.manager.get_vaults():
            rows.append(row)
        rows.sort(key=lambda x: (x['name'], x['zone_name']))
        print_table('name zone_name'.split(), rows)

@vaults.command('upload')
@click.argument("vault_name")
@click.argument("zone_name")
@click.argument("directory")
@click.pass_obj
def upload_vault(ctx, vault_name, zone_name, directory):
    """
    Upload vault
    """

    if not os.path.isdir(directory):
        raise click.UsageError('"{}" is not a directory'.format(directory))

    temp_tar_io = BytesIO()
    temp_tar = tarfile.open(fileobj=temp_tar_io, mode='w')
    temp_tar.add(directory, arcname='/')
    temp_tar.close()
    raw_tar = temp_tar_io.getvalue()
    result = ctx.manager.put('/vaults/{}/{}/'.format(vault_name, zone_name),
                     data=raw_tar,
                     headers={'Content-Type': 'application/octet-stream'})
    ok()


@manager.command()
@click.pass_obj
def zones(ctx):
    """
    Show all available zones
    """
    data = ctx.manager.get('/zones')
    rows = []
    for row in data['zones']:
        rows.append(row)
    rows.sort(key=itemgetter('name'))
    print_table('name secure'.split(), rows)

@manager.command()
@click.pass_obj
def locations(ctx):
    """
    Show all available locations
    """
    data = ctx.manager.get('/locations')
    rows = []
    for row in data['locations']:
        row['ipv4_subnet'] = IPNetwork('{}/{}'.format(row['ipv4_subnet_ip'], row['ipv4_subnet_prefixlen']))
        rows.append(row)
    rows.sort(key=itemgetter('name'))
    print_table('name ipv4_subnet'.split(), rows)


@manager.command()
@click.pass_obj
def networks(ctx):
    """
    Show all available networks
    """
    data = ctx.manager.get('/networks')
    rows = []
    for row in data['networks']:
        row['ipv6_subnet'] = IPNetwork('{}/{}'.format(row['ipv6_subnet_ip'], row['ipv6_subnet_prefixlen']))
        rows.append(row)
    rows.sort(key=itemgetter('zone_name'))
    print_table('zone_name location_name ipv6_subnet ipv6_gateway ipv6_dns'.split(), rows)

@manager.group(invoke_without_command=True, cls=AliasedGroup)
@click.pass_context
def external_services(ctx):
    """
    Show all available external services
    """
    if not ctx.invoked_subcommand:
        data = ctx.obj.manager.get('/external-services')
        rows = []
        for row in data['external_services']:
            rows.append(row)
        rows.sort(key=itemgetter('service_name'))
        print_table('service_name service_version zone_name external_host external_port'.split(), rows)

@external_services.command()
@click.argument('zone_name')
@click.argument('service_name')
@click.argument('service_version')
@click.argument('external_host')
@click.argument('external_port')
@click.pass_obj
def add(ctx, zone_name, service_name, service_version, external_host, external_port):
    action('Adding {}:{} in zone {}..'.format(service_name, service_version, zone_name))
    data = {
        'external_host': external_host,
        'external_port': external_port,
    }
    result = ctx.manager.put('/external-services/{}/{}/{}'.format(zone_name, service_name, service_version), data=data)
    ok()

def main():
    cli()

