#!/usr/bin/env python3
import pequod_cli
from pequod_cli.auth.cli import auth
from pequod_cli.console import action, ok, error
from pequod_cli.context import Context
from pequod_cli.controller.cli import _state, controller
from pequod_cli.manager.cli import manager
from pequod_cli.registry.cli import registry
from pequod_cli.replicator.cli import replicator
from pequod_cli.service import CORE_PORTS
from pequod_cli.utils import AliasedGroup
import click
import logging
import os
import requests
import yaml

DEFAULT_CONFIG_FILE = '~/.pequod-cli.yaml'

CONTEXT_SETTINGS = dict(help_option_names=['-h', '--help'])


def print_version(ctx, param, value):
    if not value or ctx.resilient_parsing:
        return
    click.echo('Pequod CLI {}'.format(pequod_cli.__version__))
    ctx.exit()


def configure_logging(loglevel):
    # configure file logger to not clutter stdout with log lines
    logging.basicConfig(level=loglevel, filename='/tmp/pequod-cli.log',
                        format='%(asctime)s %(levelname)s %(name)s: %(message)s')
    logging.getLogger('urllib3.connectionpool').setLevel(logging.WARNING)
    logging.getLogger('requests.packages.urllib3.connectionpool').setLevel(logging.WARNING)


@click.group(invoke_without_command=True, cls=AliasedGroup, context_settings=CONTEXT_SETTINGS)
@click.option('--config-file', help='Use alternative config file', default=DEFAULT_CONFIG_FILE, metavar='PATH')
@click.option('-c', '--cluster', envvar='PEQUOD_CLUSTER',
              help='Name of the cluster to use', default='default', metavar='NAME')
@click.option('-v', '--verbose', help='Verbose logging', is_flag=True)
@click.option('-V', '--version', is_flag=True, callback=print_version, expose_value=False, is_eager=True)
@click.pass_context
def cli(ctx, config_file, cluster, verbose):
    """
    Pequod cluster command line interface
    """
    configure_logging(logging.DEBUG if verbose else logging.INFO)
    fn = os.path.expanduser(config_file)
    data = {}
    if os.path.exists(fn):
        with open(fn) as fd:
            data = yaml.safe_load(fd)
    if ctx.invoked_subcommand != 'login':
        if cluster not in data:
            raise click.UsageError('''Cluster "{}" is not configured yet.
Please use the "login" command or edit the configuration file {} manually.'''.format(cluster, fn))
        endpoints = data[cluster]['endpoints']
        credentials = data[cluster]['credentials']
    else:
        endpoints = {}
        credentials = None
    ctx.obj = Context(fn, cluster, endpoints, credentials)
    if not ctx.invoked_subcommand:
        _state(ctx.obj)


class Endpoints(click.ParamType):
    name = 'Endpoints'

    def convert(self, value, param, ctx):
        if isinstance(value, dict):
            return value
        if value.startswith('{'):
            return yaml.safe_load(value)
        endpoints = {}
        parts = [v.strip() for v in value.split(',')]

        if parts[0].startswith('http'):
            for k, port in CORE_PORTS.items():
                endpoints[k] = [h.format(service=k) for h in parts]
        else:
            hosts = parts
            for k, port in CORE_PORTS.items():
                    endpoints[k] = ['http://{}:{}/'.format(h, port) for h in hosts]
        return endpoints


@cli.command()
@click.option('-e', '--endpoints', help='Endpoints',
              prompt='Service endpoints (list of IPs/URLs or complete JSON dict)', metavar='CONFIG', type=Endpoints())
@click.option('-u', '--username', help='Username', prompt='Username', metavar='USER')
@click.option('-p', '--password', help='Password', prompt=True, hide_input=True, metavar='PASSWD')
@click.pass_context
def login(click_context, endpoints, username, password):
    """
    Login to a cluster and store credentials
    """
    click.secho('Trying authenticating to core services..', bold=True, fg='blue')

    credentials = {'username': username, 'password': password}
    ctx = click_context.obj
    ctx = Context(ctx.config_file, ctx.cluster, endpoints, credentials)

    has_errors = False
    for svc in ctx.all_core_services:
        action('Checking {}..'.format(svc.name))
        try:
            svc.get('/status')
        except Exception as e:
            error('ERROR: {}'.format(e))
            has_errors = True
        else:
            ok()

    if has_errors:
        click.secho('''WARNING: At least one core service could not be reached.
Check your username/password and endpoint IPs.''', bold=True, fg='yellow')

    try:
        with open(ctx.config_file) as fd:
            data = yaml.safe_load(fd)
    except FileNotFoundError:
        data = {}

    click.confirm('Save credentials for cluster {cluster} at {fn} in plaintext?'.format(
        cluster=ctx.cluster, fn=ctx.config_file), abort=True)

    action('Storing configuration for cluster {cluster} at {fn}..', cluster=ctx.cluster, fn=ctx.config_file)
    data[ctx.cluster] = {'endpoints': endpoints, 'credentials': credentials}

    with open(ctx.config_file, 'w') as fd:
        yaml.safe_dump(data, fd)
    ok()


@cli.command()
@click.pass_obj
def status(ctx):
    """
    Ping all core cluster services
    """
    for svc in ctx.all_core_services:
        action('Checking {}..'.format(svc.name))
        has_errors = 0
        total = 0
        try:
            for ep, success, msg in svc.status():
                click.secho(' .', nl=False)
                if not success:
                    error(msg, nl=False)
                    has_errors += 1
                total += 1
        except Exception as e:
            error('ERROR: {}'.format(e))
        else:
            if has_errors:
                error('{}/{} FAILED'.format(has_errors, total))
            else:
                ok()


def register_subcommands():
    cli.add_command(auth)
    cli.add_command(manager)
    cli.add_command(registry)
    cli.add_command(controller)
    cli.add_command(replicator)


def main():
    register_subcommands()
    try:
        cli()
    except requests.HTTPError as e:
        click.secho('\nHTTP error from {}: {}\nHTTP response was: {}'.format(e.response.url, e, e.response.text),
                    bold=True, fg='red')
