import json
from logging import getLogger
import platform
import urllib
import pequod_cli
import requests

AUTH_PORT = 2192
REPLICATOR_PORT = 2193
MANAGER_PORT = 2194
REGISTRY_PORT = 2195
CONTROLLER_PORT = 2196
CASSANDRA_PORT = 9042

CORE_PORTS = {
    'auth-service': AUTH_PORT,
    'file-replicator': REPLICATOR_PORT,
    'core-manager': MANAGER_PORT,
    'application-registry': REGISTRY_PORT,
    'cluster-controller': CONTROLLER_PORT,
    'core-cassandra': CASSANDRA_PORT,
}


def format_ip(ip: str) -> str:
    """
    IPv6 IPs need to be enclosed for URLs in brackets
    """
    if ':' in ip:
        return '[{}]'.format(ip)
    return ip


def url(ip, port, path):
    url = 'http://{}:{}'.format(format_ip(ip), port)
    return urllib.parse.urljoin(url, path)


def parse_header_ip(s: str) -> str:
    return s.strip().rsplit(':', 1)[0].strip('[]')


class ClusterService:

    def __init__(self, credentials):
        self.log = getLogger('pequod.cluster-service')
        self.credentials = credentials
        self.user_agent = 'pequod-cli/{} {}/{} {}/{}'.format(pequod_cli.__version__, platform.python_implementation(),
                                                             platform.python_version(), platform.system(),
                                                             platform.release())

    @property
    def endpoints(self) -> list:
        return []

    @property
    def name(self) -> str:
        raise NotImplementedError()

    def _request(self, method_name, method, path, use_credentials=True, *args, **kwargs):
        self._set_timeout(kwargs)
        if use_credentials:
            self._set_credentials(kwargs)
        else:
            kwargs['auth'] = None

        parse_json = kwargs.get('parse_json', True)
        try:
            del kwargs['parse_json']
        except KeyError:
            pass  # if it doesn't exist great

        headers = kwargs.setdefault('headers', {})
        headers['User-Agent'] = self.user_agent

        endpoints = self.endpoints
        if not endpoints:
            raise requests.ConnectionError('No endpoints found for {}'.format(self.name))
        # sort endpoints, this is necessary to ensure calls go to the same server
        # (e.g. for template upload in Core Manager)
        endpoints.sort()
        last_error = None
        for ep in endpoints:
            url = '{}{path}'.format(ep.rstrip('/'), path=path)
            try:
                response = method(url, *args, **kwargs)
            except requests.ConnectionError as e:
                # HACK to make large file uploads possible without authentication.
                # Server will respond with 401 and "connection reset" and we have to retry
                # without authentication.
                if method_name == 'PUT' and use_credentials:
                    return self._request(method_name, method, path, False, *args, **kwargs)

                last_error = 'Connection error for {}: {}'.format(url, e)
                self.log.warning(last_error)
                continue
            if response.ok:
                self.log.debug('%s %s', method_name, response.url)
                try:
                    return response.json() if parse_json else response.content
                except ValueError:
                    return None
            elif response.status_code == 401 and use_credentials:
                self.log.debug('Got status 401 on %s. Retrying request without credentials..', response.url)
                # we need to pass parse_json status
                kwargs['parse_json'] = parse_json
                return self._request(method_name, method, path, False, *args, **kwargs)
            elif 500 <= response.status_code <= 599:  # Server errors: Retry
                last_error = 'Error {} on {}: {}'.format(response.status_code, response.url, response.text)
                self.log.warning(last_error)
            elif 400 <= response.status_code < 500:  # Client Error: Raise
                http_error_msg = '{} Client Error: {}'.format(response.status_code, response.reason)
                # do not log as we are a command line interface
                # self.log.warning('Client Error: %s', response.text)
                raise requests.HTTPError(http_error_msg, response=response)
        raise requests.ConnectionError('Request to {} failed for all {} endpoint(s): {}'.format(
            self.name, len(endpoints), last_error))

    def _prepare_data(self, kwargs: dict):
        """
        Prepares data to be sent to server (on POST, PATCH and PUT). By default serializes the data in JSON.
        Kwargs is modified in place.
        """
        if 'headers' not in kwargs:
            # By default send data as json but still leave the possibility to use something else downstream
            kwargs['headers'] = {'Content-Type': 'application/json'}
            kwargs['data'] = json.dumps(kwargs.get('data', {}))

    def _set_timeout(self, kwargs: dict):
        """
        Sets a default timeout if it's not set already
        """
        if 'timeout' not in kwargs:
            kwargs['timeout'] = 10

    def _set_credentials(self, kwargs: dict):
        if self.credentials:
            kwargs['auth'] = (self.credentials['username'], self.credentials['password'])

    def delete(self, path, *args, **kwargs):
        return self._request('DELETE', requests.delete, path, *args, **kwargs)

    def get(self, path, *args, **kwargs):
        return self._request('GET', requests.get, path, *args, **kwargs)

    def patch(self, path, *args, **kwargs):
        self._prepare_data(kwargs)
        return self._request('PATCH', requests.patch, path, *args, **kwargs)

    def post(self, path, *args, **kwargs):
        self._prepare_data(kwargs)
        return self._request('POST', requests.post, path, *args, **kwargs)

    def put(self, path, *args, **kwargs):
        self._prepare_data(kwargs)
        return self._request('PUT', requests.put, path, *args, **kwargs)


class CoreService(ClusterService):
    def __init__(self, endpoints, service_name, credentials):
        super().__init__(credentials)
        self.log = getLogger('pequod.core-service')
        self._endpoints = set(endpoints.get(service_name, []))
        self._rotated_endpoints = sorted(self._endpoints)
        self.master_endpoint = None
        self.endpoints
        self.service_name = service_name
        self.port = CORE_PORTS[service_name]

    @property
    def endpoints(self):
        if self.master_endpoint:
            return [self.master_endpoint]
        return list(self._endpoints)

    @property
    def name(self):
        return self.service_name

    def _request(self, method_name, method, path, use_credentials=True, recursion_depth=0, *args, **kwargs):
        try:
            return super()._request(method_name, method, path, use_credentials, *args, **kwargs)
        except requests.HTTPError as e:
            if e.response.status_code == 451:
                if recursion_depth > len(self._rotated_endpoints):
                    raise

                # we are contacting a service which is not the master
                master = e.response.headers.get('X-Pequod-Master')
                if not master:
                    raise
                url = urllib.parse.urlparse(e.response.url)
                for ep in self._rotated_endpoints:
                    ep_url = urllib.parse.urlparse(ep)
                    if ep_url.hostname != url.hostname:
                        self.master_endpoint = ep
                        break
                self._rotated_endpoints = self._rotated_endpoints[1:] + self._rotated_endpoints[:1]

                return self._request(method_name, method, path, use_credentials,
                                     recursion_depth=recursion_depth + 1, *args, **kwargs)
            raise

    def status(self):
        all_endpoints = sorted(self._endpoints)
        try:
            for ep in all_endpoints:
                self._endpoints = [ep]
                try:
                    try:
                        self.get('/status')
                    except requests.HTTPError as e:
                        if e.response.status_code != 451:
                            raise
                except Exception as e:
                    yield (ep, False, str(e))
                yield (ep, True, 'OK')
        finally:
            self._endpoints = all_endpoints