import logging
import random
import click
import time
from pequod_cli.console import print_table, action, ok
import requests
from pequod_cli.model import ApplicationSpec
from pequod_cli.utils import AliasedGroup, generate_random_name


@click.group(cls=AliasedGroup)
def controller():
    """
    Interact with Cluster Controller, show current cluster state
    """
    pass


@controller.command()
@click.pass_obj
def state(ctx):
    """
    Display the current cluster nodes and running application instances
    """
    _state(ctx)


@controller.command()
@click.pass_context
def dashboard(ctx):
    """
    Display the current cluster nodes and running application instances and updates every second
    """
    while True:
        click.clear()
        _state(ctx.obj)
        time.sleep(1)


def _state(ctx):
    rawdata = ctx.controller.get('/state')

    rows = []
    for node_name, node_data in sorted(rawdata['nodes'].items()):
        data = {'node_name': node_name, 'status': node_data['status'], 'reboot': False, 'maintenance': False}
        try:
            node_status = ctx.manager.get('/nodes/{}'.format(node_name))
            data['reboot'] = node_status.get('reboot')
            data['maintenance'] = node_status.get('maintenance')
        except requests.HTTPError as e:
            if e.response.status_code != 404:
                # ignore nodes not configure in core manager
                # TODO: better error handling
                logging.exception('Failed to contact core manager')
        data.update(node_data['resources'])
        data.update(node_data['metadata'])
        rows.append(data)

    COLS = ('node_name status memory_mb cpu_vmhz processes_count filehandles_count'
            + ' node_start_time agent_start_time template_version reboot maintenance').split()
    print_table(COLS, rows)

    click.echo('')
    rows = []
    for instance_name, instance_data in sorted(rawdata['instances'].items()):
        data = {'instance_name': instance_name,
                'node_name': instance_data['node_name'],
                'lifecycle_phase': instance_data['lifecycle_phase']}
        data.update(instance_data['resources'] or {})
        data.update(instance_data['metadata'])
        rows.append(data)

    COLS = ('instance_name node_name repository_name application_name application_version zone_name lifecycle_phase'
            + ' memory_mb cpu_vmhz processes_count filehandles_count ipv6 instance_started_time').split()
    print_table(COLS, rows)


@controller.command()
@click.argument('app', metavar='REPO/APP:VERSION', type=ApplicationSpec())
@click.argument('zone_name')
@click.option('--node', help='Start instance on a specific node')
@click.option('--count', help='Number of instances to start', default=1, type=click.IntRange(1, None))
@click.option('--memory-mb', help='Memory in MB', type=click.IntRange(1, None))
@click.option('--cpu-vmhz', help='CPU in VMHz', type=click.IntRange(1, None))
@click.option('--processes-count', help='Processes', type=click.IntRange(1, None))
@click.pass_obj
def start(ctx, app, zone_name, node, count, **kwargs):
    """
    Start one or more application instances
    """
    manifest = ctx.registry.get_manifest(app.repository_name, app.application_name, app.application_version)

    if not manifest:
        raise click.UsageError('Application manifest not found: {}'.format(app))

    zones = ctx.manager.get_zones()
    if zone_name not in zones:
        raise click.UsageError('Zone does not exist: {}'.format(zone_name))

    if node:
        node_name = node
    else:
        rawdata = ctx.controller.get_state()
        node_candidates = []
        for _node_name, node_data in sorted(rawdata['nodes'].items()):
            if node_data['status'] == 'REGISTERED':
                node_candidates.append(_node_name)
        node_name = random.choice(node_candidates)
    resources = {}
    for key, val in manifest['resources'].items():
        default_key = [k for k in val.keys() if k.startswith('default_')][0]
        unit = default_key.split('_')[1]
        resources['{}_{}'.format(key, unit)] = val[default_key]
    for key in 'memory_mb', 'cpu_vmhz', 'processes_count':
        val = kwargs.get(key)
        if val:
            resources[key] = val
    action('Starting {} instances on {}..'.format(count, node_name))
    for i in range(count):
        data = {'actions_create': [{
                'instance_name': generate_random_name('{}-{}-{}-'.format(
                    app.repository_name, app.application_name, app.application_version)[:24], 4),
                'node_name': node_name,
                'resources': resources,
                'metadata': {
                    'repository_name': app.repository_name,
                    'application_name': app.application_name,
                    'application_version': app.application_version,
                    'zone_name': zone_name,
                }
                }], 'allow_partial_commit': False}
        ctx.controller.put('/scheduler', data=data)
    ok()


@controller.command()
@click.argument('instance_name')
@click.pass_obj
def stop(ctx, instance_name):
    """
    Stop a single application instance
    """
    rawdata = ctx.controller.get_state()
    node_name = None
    for _instance_name, instance_data in sorted(rawdata['instances'].items()):
        if _instance_name == instance_name:
            node_name = instance_data['node_name']
            break
    if node_name:
        action('Stopping instance {} on {}..'.format(instance_name, node_name))
        data = {'actions_kill': [{
                'instance_name': instance_name,
                'node_name': instance_data['node_name'],
                }], 'allow_partial_commit': False}
        ctx.controller.put('/scheduler', data=data)
        ok()