from operator import itemgetter
import os
import tarfile
import click
from pathlib import Path
from netaddr import IPNetwork
from pequod_cli.console import print_table, action, ok, print_permissions
from pequod_cli.utils import AliasedGroup
from io import BytesIO


@click.group(cls=AliasedGroup)
def manager():
    """
    Interact with Core Manager
    """
    pass


@manager.group(invoke_without_command=True, cls=AliasedGroup)
@click.pass_context
def nodes(ctx):
    """
    Show all configured nodes
    """
    if not ctx.invoked_subcommand:
        data = ctx.obj.manager.get('/nodes')
        rows = []
        for row in data['nodes']:
            data = ctx.obj.manager.get('/nodes/{}'.format(row['name']))
            row.update(data)
            rows.append(row)
        rows.sort(key=itemgetter('name'))
        print_table('name mac ipv4 ipv6 template_name template_version reboot maintenance'.split(), rows)


@nodes.command()
@click.argument('node_name')
@click.pass_obj
def reboot(ctx, node_name):
    """
    Reboot a cluster app node
    """
    action('Rebooting {}..'.format(node_name))
    data = {'maintenance': False,
            'reboot': True
            }
    ctx.manager.patch('/nodes/{}'.format(node_name), data=data)
    ok()


def parse_version(v):
    return tuple([int(x) for x in v.split(".")])


@nodes.command()
@click.argument('node_name')
@click.argument('key_value', nargs=-1)
@click.pass_obj
def update(ctx, node_name, key_value):
    """
    Update node configuration (e.g. "template_version")
    """
    data = ctx.manager.get('/nodes/{}'.format(node_name))
    del data['reboot']
    del data['maintenance']
    for pair in key_value:
        try:
            key, val = pair.split('=', 1)
        except:
            raise click.UsageError('Key pairs need to be specified as KEY=VAL')
        if key not in data:
            raise click.UsageError('Invalid key: "{}" is not allowed'.format(key))
        if key == 'template_version' and val == 'latest':
            # special case: find out latest template and use it
            latest_version = '0'
            for row in ctx.manager.get_templates():
                if row['template_name'] == data['template_name']:
                    # TODO: only supports numeric versions right now :-P
                    try:
                        if parse_version(row['template_version']) > parse_version(latest_version):
                            latest_version = row['template_version']
                    except ValueError:
                        # ignore non-int versions
                        pass
            val = latest_version
        data[key] = val
    action('Updating {}..'.format(node_name))
    ctx.manager.put('/nodes/{}'.format(node_name), data=data)
    ok()


@manager.group(invoke_without_command=True, cls=AliasedGroup)
@click.pass_context
def templates(ctx):
    """
    Show all available templates
    """
    if not ctx.invoked_subcommand:
        rows = []
        for row in ctx.obj.manager.get_templates():
            rows.append(row)
        rows.sort(key=lambda x: (x['template_name'], x['template_version'], x['image_type']))
        print_table('template_name template_version image_type'.split(), rows)


@templates.command()
@click.argument("files", nargs=-1)
@click.pass_obj
def upload(ctx, files):
    """
    Upload template files
    """
    existing_templates = set([(x['template_name'], x['template_version'], x['image_type'])
                              for x in ctx.manager.get_templates()])
    for fn in files:
        path = Path(fn)
        try:
            _, template_name, template_version, image_type = path.name.rsplit('-', 3)
        except ValueError:
            raise click.UsageError('Invalid file name "{}". Files must be named like '
                                   + '"foobar-<TEMPLATE_NAME>-<TEMPLATE_VERSION>-<IMAGE_TYPE>".'.format(path.name))
        image_type = image_type.upper()
        identity = template_name, template_version, image_type
        if identity in existing_templates:
            raise click.UsageError('Template {}:{} {} already exists'.format(*identity))
        action('Uploading template {} version {} {}..'.format(*identity))
        with path.open('rb') as fd:
            ctx.manager.put('/templates/{}/{}/{}'.format(template_name, template_version, image_type), data=fd,
                            headers={'Content-Type': 'application/octet-stream'}, timeout=300)
        ok()


@manager.group(invoke_without_command=True, cls=AliasedGroup)
@click.pass_context
def vaults(ctx):
    """
    Show all available vaults
    """
    if not ctx.invoked_subcommand:
        rows = []
        for row in ctx.obj.manager.get_vaults():
            rows.append(row)
        rows.sort(key=lambda x: (x['name'], x['zone_name']))
        print_table('name zone_name'.split(), rows)


@vaults.command('upload')
@click.argument("vault_name")
@click.argument("zone_name")
@click.argument("directory")
@click.pass_obj
def upload_vault(ctx, vault_name, zone_name, directory):
    """
    Upload vault
    """

    if not os.path.isdir(directory):
        raise click.UsageError('"{}" is not a directory'.format(directory))

    temp_tar_io = BytesIO()
    temp_tar = tarfile.open(fileobj=temp_tar_io, mode='w')
    temp_tar.add(directory, arcname='/')
    temp_tar.close()
    raw_tar = temp_tar_io.getvalue()
    ctx.manager.put('/vaults/{}/{}/'.format(vault_name, zone_name),
                    data=raw_tar,
                    headers={'Content-Type': 'application/octet-stream'})
    ok()


@manager.command()
@click.pass_obj
def zones(ctx):
    """
    Show all available zones
    """
    data = ctx.manager.get('/zones')
    rows = []
    for row in data['zones']:
        rows.append(row)
    rows.sort(key=itemgetter('name'))
    print_table('name secure'.split(), rows)


@manager.command()
@click.pass_obj
def locations(ctx):
    """
    Show all available locations
    """
    data = ctx.manager.get('/locations')
    rows = []
    for row in data['locations']:
        row['ipv4_subnet'] = IPNetwork('{}/{}'.format(row['ipv4_subnet_ip'], row['ipv4_subnet_prefixlen']))
        rows.append(row)
    rows.sort(key=itemgetter('name'))
    print_table('name ipv4_subnet'.split(), rows)


@manager.command()
@click.pass_obj
def networks(ctx):
    """
    Show all available networks
    """
    data = ctx.manager.get('/networks')
    rows = []
    for row in data['networks']:
        row['ipv6_subnet'] = IPNetwork('{}/{}'.format(row['ipv6_subnet_ip'], row['ipv6_subnet_prefixlen']))
        rows.append(row)
    rows.sort(key=itemgetter('zone_name'))
    print_table('zone_name location_name ipv6_subnet ipv6_gateway ipv6_dns'.split(), rows)


@manager.group(invoke_without_command=True, cls=AliasedGroup)
@click.pass_context
def external_services(ctx):
    """
    Show all available external services
    """
    if not ctx.invoked_subcommand:
        data = ctx.obj.manager.get('/external-services')
        rows = []
        for row in data['external_services']:
            rows.append(row)
        rows.sort(key=itemgetter('service_name'))
        print_table('service_name service_version zone_name external_host external_port'.split(), rows)


@external_services.command()
@click.argument('zone_name')
@click.argument('service_name')
@click.argument('service_version')
@click.argument('external_host')
@click.argument('external_port', type=int)
@click.pass_obj
def add(ctx, zone_name, service_name, service_version, external_host, external_port):
    action('Adding {}:{} in zone {}..'.format(service_name, service_version, zone_name))
    data = {
        'external_host': external_host,
        'external_port': external_port,
    }
    ctx.manager.put('/external-services/{}/{}/{}'.format(zone_name, service_name, service_version), data=data)
    ok()


@manager.command()
@click.pass_obj
def permissions(ctx):
    """
    List all permissions
    """
    print_permissions(ctx.manager.permissions())