import logging
import random
import subprocess
import click

# see http://golang.org/src/pkg/crypto/x509/root_unix.go
from pathlib import Path

CERT_FILES = [
    "/etc/ssl/certs/ca-certificates.crt",     # Debian/Ubuntu/Gentoo etc.
    "/etc/pki/tls/certs/ca-bundle.crt",       # Fedora/RHEL
    "/etc/ssl/ca-bundle.pem",                 # OpenSUSE
    "/etc/ssl/cert.pem",                      # OpenBSD
    "/usr/local/share/certs/ca-root-nss.crt"  # FreeBSD/DragonFly
]


def generate_random_name(prefix: str, size: int) -> str:
    """
    See GenerateRandomName in vendor/src/github.com/docker/libcontainer/utils/utils.go

    >>> len(generate_random_name('abc', 4))
    7
    """
    return '{}%0{}x'.format(prefix, size) % random.randrange(16 ** size)


class AliasedGroup(click.Group):

    def get_command(self, ctx, cmd_name):
        rv = click.Group.get_command(self, ctx, cmd_name)
        if rv is not None:
            return rv
        matches = [x for x in self.list_commands(ctx)
                   if x.startswith(cmd_name)]
        if not matches:
            return None
        elif len(matches) == 1:
            return click.Group.get_command(self, ctx, matches[0])
        ctx.fail('Too many matches: %s' % ', '.join(sorted(matches)))


def find_cacert_bundle():
    for path in CERT_FILES:
        if Path(path).is_file():
            logging.debug('Found CA bundle %s', path)
            return path

    try:
        # really stupid way to get CA bundle for darwin
        # see http://golang.org/src/pkg/crypto/x509/root_darwin.go
        output = subprocess.check_output(["/usr/bin/security", "find-certificate", "-a", "-p",
                                          "/System/Library/Keychains/SystemRootCertificates.keychain"])
        path = '/tmp/pequod-cli-ca-bundle.pem'
        with open(path, 'wb') as fd:
            fd.write(output)
        return path
    except Exception as e:
        logging.exception('Failed to find Mac CA bundle: %s', e)
