import logging
import random
import subprocess
import click
from collections import OrderedDict

# see http://golang.org/src/pkg/crypto/x509/root_unix.go
from pathlib import Path

CERT_FILES = [
    "/etc/ssl/certs/ca-certificates.crt",     # Debian/Ubuntu/Gentoo etc.
    "/etc/pki/tls/certs/ca-bundle.crt",       # Fedora/RHEL
    "/etc/ssl/ca-bundle.pem",                 # OpenSUSE
    "/etc/ssl/cert.pem",                      # OpenBSD
    "/usr/local/share/certs/ca-root-nss.crt"  # FreeBSD/DragonFly
]


def generate_random_name(prefix: str, size: int) -> str:
    """
    See GenerateRandomName in vendor/src/github.com/docker/libcontainer/utils/utils.go

    >>> len(generate_random_name('abc', 4))
    7
    """
    return '{}%0{}x'.format(prefix, size) % random.randrange(16 ** size)


def generate_help_tree(commands):
    def make_structure(commands, index=0, structure=None):

        if not structure:
            structure = []

        if not commands or len(commands) == index:
            return structure

        commands = OrderedDict(sorted(commands.items(), key=lambda t: t[0]))
        keys = list(commands.keys())

        if isinstance(commands[keys[index]], click.core.Group):
            structure.append({
                'name': commands[keys[index]].name,
                'commands': make_structure(commands[keys[index]].commands)})

        elif isinstance(commands[keys[index]], click.core.Command):
            structure.append({'name': commands[keys[index]].name})

        return make_structure(commands, index + 1, structure)

    def get_longest_name(structure, index=0, result=0):

        if not structure or len(structure) == index:
            return result

        if 'commands' in structure[index]:
            result = max(result, len(structure[index]['name']), get_longest_name(structure[index]['commands']))

        else:
            result = max(result, len(structure[index]['name']))

        return get_longest_name(structure, index + 1, result)

    def make_tree(structure, block):

        def build(structure, index=0, level=0, tree=''):

            if not structure or len(structure) == index:
                return tree

            padding = (' ' * (block + 6) + '|') * level
            label = '-- \033[95m' + structure[index]['name'] + '\033[0m '
            extension = '-' * (block - len(structure[index]['name']) + 2) + '\n'

            if 'commands' in structure[index]:
                tree += padding + label + extension + build(structure[index]['commands'], 0, level + 1, '')
            else:
                tree += padding + label + '\n'

            return build(structure, index + 1, level, tree)

        return build(structure)

    structure = make_structure(commands)
    block = get_longest_name(structure)

    return make_tree(structure, block)


class AliasedGroup(click.Group):

    def get_command(self, ctx, cmd_name):
        rv = click.Group.get_command(self, ctx, cmd_name)
        if rv is not None:
            return rv
        matches = [x for x in self.list_commands(ctx)
                   if x.startswith(cmd_name)]
        if not matches:
            return None
        elif len(matches) == 1:
            return click.Group.get_command(self, ctx, matches[0])
        ctx.fail('Too many matches: %s' % ', '.join(sorted(matches)))


def find_cacert_bundle():
    for path in CERT_FILES:
        if Path(path).is_file():
            logging.debug('Found CA bundle %s', path)
            return path

    try:
        # really stupid way to get CA bundle for darwin
        # see http://golang.org/src/pkg/crypto/x509/root_darwin.go
        output = subprocess.check_output(["/usr/bin/security", "find-certificate", "-a", "-p",
                                          "/System/Library/Keychains/SystemRootCertificates.keychain"])
        path = '/tmp/pequod-cli-ca-bundle.pem'
        with open(path, 'wb') as fd:
            fd.write(output)
        return path
    except Exception as e:
        logging.exception('Failed to find Mac CA bundle: %s', e)
