from operator import itemgetter
import click
from pequod_cli.console import print_table, action, ok
from pequod_cli.utils import AliasedGroup


@click.group(cls=AliasedGroup)
def lbupdater():
    """
    Interact with LB Updater
    """
    pass


@lbupdater.command('load-balancers')
@click.pass_obj
def load_balancers(ctx):
    """
    List all defined load balancers
    """
    cols = 'name load_balancer_type location'.split()

    rows = []
    for lb in ctx.lbupdater.get('/load-balancers'):
        rows.append(lb)

    print_table(cols, rows)


@lbupdater.group(invoke_without_command=True, cls=AliasedGroup)
@click.pass_context
def pools(ctx):
    """
    List all defined load balancer pools
    """
    if not ctx.invoked_subcommand:
        ctx = ctx.obj
        cols = ('name service_name service_version zone_name'
                + ' load_balancer_configuration_name virtual_server_name health_check_http_path').split()

        rows = []
        for pool in ctx.lbupdater.get('/load-balancer-pools'):
            rows.append(pool)

        rows.sort(key=itemgetter('name'))

        print_table(cols, rows)


@pools.command()
@click.argument('name')
@click.argument('service_name')
@click.argument('service_version')
@click.argument('zone_name')
@click.argument('load_balancer_configuration_name')
@click.argument('virtual_server_name')
@click.option('--health-check-http-path', help='HTTP path for health check')
@click.pass_obj
def add(ctx, name, **kwargs):
    """
    Update load balancer pool configuration
    """
    action('Adding {}..'.format(name))
    data = kwargs
    ctx.lbupdater.put('/load-balancer-pools/{}'.format(name), data=data)
    ok()


@pools.command()
@click.argument('name')
@click.argument('key_value', nargs=-1)
@click.pass_obj
def update(ctx, name, key_value):
    """
    Update load balancer pool configuration
    """
    data = ctx.lbupdater.get('/load-balancer-pools/{}'.format(name))
    for pair in key_value:
        try:
            key, val = pair.split('=', 1)
        except:
            raise click.UsageError('Key pairs need to be specified as KEY=VAL')
        if key not in data:
            raise click.UsageError('Invalid key: "{}" is not allowed'.format(key))
        data[key] = val
    action('Updating {}..'.format(name))
    ctx.lbupdater.put('/load-balancer-pools/{}'.format(name), data=data)
    ok()
