#!/usr/bin/env python3
import pequod_cli
from pequod_cli.auth.cli import auth
from pequod_cli.console import action, ok, error
from pequod_cli.context import Context
from pequod_cli.controller.cli import _state, controller
from pequod_cli.lbupdater.cli import lbupdater
from pequod_cli.manager.cli import manager
from pequod_cli.registry.cli import registry
from pequod_cli.replicator.cli import replicator
from pequod_cli.service import CORE_DOMAINS
from pequod_cli.utils import AliasedGroup, generate_help_tree, find_cacert_bundle
import click
import logging
import os
import requests
import yaml

DEFAULT_CONFIG_FILE = '~/.pequod-cli.yaml'

CONTEXT_SETTINGS = dict(help_option_names=['-h', '--help'])

COMMANDS = {
    'auth': auth,
    'manager': manager,
    'registry': registry,
    'controller': controller,
    'replicator': replicator,
    'lbupdater': lbupdater
}

NON_RESTRICTED = frozenset(['login', 'help'])


def print_version(ctx, param, value):
    if not value or ctx.resilient_parsing:
        return
    click.echo('Pequod CLI {}'.format(pequod_cli.__version__))
    ctx.exit()


def configure_logging(loglevel):
    # configure file logger to not clutter stdout with log lines
    logging.basicConfig(level=loglevel, filename='/tmp/pequod-cli.log',
                        format='%(asctime)s %(levelname)s %(name)s: %(message)s')
    logging.getLogger('urllib3.connectionpool').setLevel(logging.WARNING)
    logging.getLogger('requests.packages.urllib3.connectionpool').setLevel(logging.WARNING)


@click.group(invoke_without_command=True, cls=AliasedGroup, context_settings=CONTEXT_SETTINGS)
@click.option('--config-file', help='Use alternative config file', default=DEFAULT_CONFIG_FILE, metavar='PATH')
@click.option('-c', '--cluster', envvar='PEQUOD_CLUSTER',
              help='Name of the cluster to use', default='default', metavar='NAME')
@click.option('-v', '--verbose', help='Verbose logging', is_flag=True)
@click.option('-V', '--version', is_flag=True, callback=print_version, expose_value=False, is_eager=True)
@click.pass_context
def cli(ctx, config_file, cluster, verbose):
    """
    Pequod cluster command line interface
    """
    configure_logging(logging.DEBUG if verbose else logging.INFO)
    fn = os.path.expanduser(config_file)
    data = {}
    if os.path.exists(fn):
        with open(fn) as fd:
            data = yaml.safe_load(fd)
    if ctx.invoked_subcommand not in NON_RESTRICTED:
        if cluster not in data:
            raise click.UsageError('''Cluster "{}" is not configured yet.
Please use the "login" command or edit the configuration file {} manually.'''.format(cluster, fn))
        endpoints = data[cluster]['endpoints']
        credentials = data[cluster]['credentials']
        cacert = data[cluster].get('cacert')
    else:
        endpoints = {}
        credentials = None
        cacert = None
    ctx.obj = Context(fn, cluster, endpoints, credentials, cacert)
    if not ctx.invoked_subcommand:
        _state(ctx.obj)


class Endpoints(click.ParamType):
    name = 'Endpoints'

    def convert(self, value, param, ctx):
        if isinstance(value, dict):
            return value
        if value.startswith('{'):
            return yaml.safe_load(value)
        endpoints = {}
        parts = [v.strip() for v in value.split(',')]

        if parts[0].startswith('http'):
            # pequod login -d http(s)://{domain}.pequod.example.org
            for service, domain in CORE_DOMAINS.items():
                endpoints[service] = [h.format(service=service, domain=domain) for h in parts]
        else:
            # pequod login -d pequod.example.org
            # pequod login -d .pequod.example.org
            domains = parts
            for service, subdomain in CORE_DOMAINS.items():
                endpoints[service] = ['https://{}.{}/'.format(subdomain, domain.strip('.')) for domain in domains]
        return endpoints


@cli.command()
@click.option('-d', '--domain', 'endpoints', help='Cluster domain address',
              prompt='Cluster domain (e.g. "pequod.example.org")', metavar='DOMAIN', type=Endpoints())
@click.option('-u', '--username', help='Username', prompt='Username', metavar='USER')
@click.option('-p', '--password', help='Password', prompt=True, hide_input=True, metavar='PASSWD')
@click.option('--cacert', help='CA certificate to verify peer against (SSL)', metavar='FILE')
@click.pass_context
def login(click_context, endpoints, username, password, cacert):
    """
    Login to a cluster and store credentials
    """
    click.secho('Trying authenticating to core services..', bold=True, fg='blue')

    if not cacert:
        cacert = find_cacert_bundle()

    credentials = {'username': username, 'password': password}
    ctx = click_context.obj
    ctx = Context(ctx.config_file, ctx.cluster, endpoints, credentials, cacert)

    has_errors = False
    for svc in ctx.all_services:
        action('Checking {}..'.format(svc.name))
        try:
            svc.get('/status')
        except Exception as e:
            error('ERROR: {}'.format(e))
            has_errors = True
        else:
            ok()

    if has_errors:
        click.secho('''WARNING: At least one core service could not be reached.
Check your username/password and cluster domain address.''', bold=True, fg='yellow')

    try:
        with open(ctx.config_file) as fd:
            data = yaml.safe_load(fd)
    except FileNotFoundError:
        data = {}

    click.confirm('Save credentials for cluster {cluster} at {fn} in plaintext?'.format(
        cluster=ctx.cluster, fn=ctx.config_file), abort=True)

    action('Storing configuration for cluster {cluster} at {fn}..', cluster=ctx.cluster, fn=ctx.config_file)
    data[ctx.cluster] = {'endpoints': endpoints, 'credentials': credentials, 'cacert': cacert}

    with open(ctx.config_file, 'w') as fd:
        yaml.safe_dump(data, fd)
    ok()


@cli.command()
@click.pass_obj
def status(ctx):
    """
    Ping all core cluster services
    """
    for svc in ctx.all_services:
        action('Checking {}..'.format(svc.name))
        has_errors = 0
        total = 0
        try:
            for ep, success, msg in svc.status():
                click.secho(' .', nl=False)
                if not success:
                    error(msg, nl=False)
                    has_errors += 1
                total += 1
        except Exception as e:
            error('ERROR: {}'.format(e))
        else:
            if has_errors:
                error('{}/{} FAILED'.format(has_errors, total))
            else:
                ok()


@cli.command()
@click.pass_context
def help(ctx):
    """
    Show tree of commands and subcommands
    """
    commands = dict(list(COMMANDS.items()) + list(cli.commands.items()))
    click.echo('%s'
               '\n**In order to get more information regarding'
               ' command execute "pequod [COMMAND] --help"\n'
               % generate_help_tree(commands))


def register_subcommands():
    for name, command in COMMANDS.items():
        cli.add_command(command)


def main():
    register_subcommands()
    try:
        cli()
    except requests.HTTPError as e:
        click.secho('\nHTTP error from {}: {}\nHTTP response was: {}'.format(e.response.url, e, e.response.text),
                    bold=True, fg='red')
