import click
import yaml
from pequod_cli.console import print_table, print_permissions
from pequod_cli.model import ApplicationSpec
from pequod_cli.utils import AliasedGroup


@click.group(invoke_without_command=True, cls=AliasedGroup)
@click.pass_context
def registry(ctx):
    """
    Interact with Application Registry, list all applications
    """
    if not ctx.invoked_subcommand:
        _tags(ctx.obj)


def _tags(ctx, query=None):
    data = ctx.registry.get('/tags')
    rows = []
    for row in data:
        if query and query not in row['id']:
            continue
        row['last_update_time'] = row['last_update'] / 1000
        row['command'] = ' '.join(row['container_setup']['command'])
        row['has_manifest'] = bool(row['container_setup'].get('manifest_image_id'))
        rows.append(row)

    rows.sort(key=lambda x: x['last_update_time'])

    COLS = 'repository_name application_name application_version command has_manifest last_update_time'.split()
    print_table(COLS, rows)


@registry.command()
@click.option('-q', '--query', help='List only tags matching the query')
@click.pass_obj
def tags(ctx, query):
    """
    Show all available tags/applications
    """
    _tags(ctx, query)


@registry.command()
@click.pass_obj
def services(ctx):
    """
    Show all available services and applications providing them
    """
    data = ctx.registry.get('/tags')
    rows = []
    for row in data:
        manifest = ctx.registry.get_manifest(row['repository_name'],
                                             row['application_name'], row['application_version'])
        if not manifest:
            continue
        for service in manifest['services'].get('provide', []):

            rows.append({'service_name': service['service_name'],
                         'service_version': service['service_version'],
                         'repository_name': row['repository_name'],
                         'application_name': row['application_name'],
                         'application_version': row['application_version'],
                         })

    rows.sort(key=lambda x: (x['service_name'], x['service_version'],
                             x['repository_name'], x['application_name'], x['application_version']))

    COLS = 'service_name service_version repository_name application_name application_version'.split()
    print_table(COLS, rows)


@registry.command()
@click.argument('app', metavar='REPO/APP:VERSION', type=ApplicationSpec())
@click.pass_obj
def manifest(ctx, app):
    """
    Print a single application manifest
    """
    data = ctx.registry.get_manifest(app.repository_name, app.application_name, app.application_version)
    click.echo(yaml.safe_dump(data))


@registry.command()
@click.pass_obj
def permissions(ctx):
    """
    List all permissions
    """
    print_permissions(ctx.registry.permissions())
