import fnmatch
from operator import itemgetter
import os
import tarfile
import click
from pathlib import Path
from netaddr import IPNetwork
from pequod_cli.console import print_table, action, ok, print_permissions
from pequod_cli.utils import AliasedGroup
from io import BytesIO


@click.group(cls=AliasedGroup)
def manager():
    """
    Interact with Core Manager
    """
    pass


@manager.group(invoke_without_command=True, cls=AliasedGroup)
@click.pass_context
def nodes(ctx):
    """
    Show all configured nodes
    """
    if not ctx.invoked_subcommand:
        data = ctx.obj.manager.get('/nodes')
        rows = []
        for row in data['nodes']:
            data = ctx.obj.manager.get('/nodes/{}'.format(row['name']))
            row.update(data)
            rows.append(row)
        rows.sort(key=itemgetter('name'))
        print_table('name mac ipv4 ipv6 template_name template_version reboot maintenance'.split(), rows)


def get_matching_node_names(manager, pattern):
    data = manager.get('/nodes')
    names = [row['name'] for row in data['nodes']]
    matching_node_names = fnmatch.filter(names, pattern)
    return sorted(matching_node_names)


@nodes.command()
@click.argument('node_name_pattern')
@click.pass_obj
def reboot(ctx, node_name_pattern):
    """
    Reboot one or more cluster app node(s)
    """
    for node_name in get_matching_node_names(ctx.manager, node_name_pattern):
        action('Rebooting {}..'.format(node_name))
        data = {'maintenance': False,
                'reboot': True
                }
        ctx.manager.patch('/nodes/{}'.format(node_name), data=data)
        ok()


def parse_version(v):
    return tuple([int(x) for x in v.split(".")])


def get_latest_template_version(manager, template_name):
    # special case: find out latest template and use it
    latest_version = '0'
    for row in manager.get_templates():
        if row['template_name'] == template_name:
            # TODO: only supports numeric versions right now :-P
            try:
                if parse_version(row['template_version']) > parse_version(latest_version):
                    latest_version = row['template_version']
            except ValueError:
                # ignore non-int versions
                pass
    return latest_version


@nodes.command('add')
@click.argument('node_name')
@click.argument('mac')
@click.argument('ipv4')
@click.argument('ipv6')
@click.argument('template_name')
@click.option('--template-version', help='Use specific template version')
@click.pass_obj
def add_node(ctx, node_name, mac, ipv4, ipv6, template_name, template_version):
    """
    Add a new app node configuration, uses latest template version by default.
    """
    if not template_version:
        template_version = get_latest_template_version(ctx.manager, template_name)
    data = {
        'mac': mac,
        'ipv4': ipv4,
        'ipv6': ipv6,
        'template_name': template_name,
        'template_version': template_version,
    }
    action('Adding node {node_name} with MAC address {mac}..', **vars())
    ctx.manager.put('/nodes/{}'.format(node_name), data=data)
    ok()


@nodes.command()
@click.argument('node_name_pattern')
@click.argument('key_value', nargs=-1)
@click.pass_obj
def update(ctx, node_name_pattern, key_value):
    """
    Update node configuration (e.g. "template_version")
    """
    for node_name in get_matching_node_names(ctx.manager, node_name_pattern):
        data = ctx.manager.get('/nodes/{}'.format(node_name))
        del data['reboot']
        del data['maintenance']
        for pair in key_value:
            try:
                key, val = pair.split('=', 1)
            except:
                raise click.UsageError('Key pairs need to be specified as KEY=VAL')
            if key not in data:
                raise click.UsageError('Invalid key: "{}" is not allowed'.format(key))
            if key == 'template_version' and val == 'latest':
                val = get_latest_template_version(ctx.manager, data['template_name'])
            data[key] = val
        action('Updating {}..'.format(node_name))
        ctx.manager.put('/nodes/{}'.format(node_name), data=data)
        ok()


@manager.group(invoke_without_command=True, cls=AliasedGroup)
@click.pass_context
def templates(ctx):
    """
    Show all available templates
    """
    if not ctx.invoked_subcommand:
        rows = []
        for row in ctx.obj.manager.get_templates():
            rows.append(row)
        rows.sort(key=lambda x: (x['template_name'], x['template_version'], x['image_type']))
        print_table('template_name template_version image_type'.split(), rows)


@templates.command()
@click.argument("files", nargs=-1)
@click.pass_obj
def upload(ctx, files):
    """
    Upload template files
    """
    existing_templates = set([(x['template_name'], x['template_version'], x['image_type'])
                              for x in ctx.manager.get_templates()])
    for fn in files:
        path = Path(fn)
        try:
            _, template_name, template_version, image_type = path.name.rsplit('-', 3)
        except ValueError:
            raise click.UsageError('Invalid file name "{}". Files must be named like '
                                   + '"foobar-<TEMPLATE_NAME>-<TEMPLATE_VERSION>-<IMAGE_TYPE>".'.format(path.name))
        image_type = image_type.upper()
        identity = template_name, template_version, image_type
        if identity in existing_templates:
            raise click.UsageError('Template {}:{} {} already exists'.format(*identity))
        action('Uploading template {} version {} {}..'.format(*identity))
        with path.open('rb') as fd:
            ctx.manager.put('/templates/{}/{}/{}'.format(template_name, template_version, image_type), data=fd,
                            headers={'Content-Type': 'application/octet-stream'}, timeout=300)
        ok()


@manager.group(invoke_without_command=True, cls=AliasedGroup)
@click.pass_context
def vaults(ctx):
    """
    Show all available vaults
    """
    if not ctx.invoked_subcommand:
        rows = []
        for row in ctx.obj.manager.get_vaults():
            rows.append(row)
        rows.sort(key=lambda x: (x['name'], x['zone_name']))
        print_table('name zone_name'.split(), rows)


@vaults.command('upload')
@click.argument("vault_name")
@click.argument("zone_name")
@click.argument("directory")
@click.pass_obj
def upload_vault(ctx, vault_name, zone_name, directory):
    """
    Upload vault
    """

    if not os.path.isdir(directory):
        raise click.UsageError('"{}" is not a directory'.format(directory))

    temp_tar_io = BytesIO()
    temp_tar = tarfile.open(fileobj=temp_tar_io, mode='w')
    temp_tar.add(directory, arcname='/')
    temp_tar.close()
    raw_tar = temp_tar_io.getvalue()
    ctx.manager.put('/vaults/{}/{}/'.format(vault_name, zone_name),
                    data=raw_tar,
                    headers={'Content-Type': 'application/octet-stream'})
    ok()


@manager.group(invoke_without_command=True, cls=AliasedGroup)
@click.pass_context
def zones(ctx):
    """
    Show all available zones
    """
    if not ctx.invoked_subcommand:
        ctx = ctx.obj
        data = ctx.manager.get('/zones')
        rows = []
        for row in data['zones']:
            rows.append(row)
        rows.sort(key=itemgetter('name'))
        print_table('name secure'.split(), rows)


@zones.command('add')
@click.argument('zone_name')
@click.argument('secure', type=bool)
@click.pass_obj
def add_zone(ctx, zone_name, secure):
    """
    Add a new zone
    """
    data = {
        'secure': secure
    }
    action('Adding zone {zone_name} with secure={secure}..', **vars())
    ctx.manager.put('/zones/{}'.format(zone_name), data=data)
    ok()


@manager.group(invoke_without_command=True, cls=AliasedGroup)
@click.pass_context
def locations(ctx):
    """
    Show all available locations
    """
    if not ctx.invoked_subcommand:
        ctx = ctx.obj
        data = ctx.manager.get('/locations')
        rows = []
        for row in data['locations']:
            row['ipv4_subnet'] = IPNetwork('{}/{}'.format(row['ipv4_subnet_ip'], row['ipv4_subnet_prefixlen']))
            rows.append(row)
        rows.sort(key=itemgetter('name'))
        print_table('name ipv4_subnet'.split(), rows)


@locations.command('add')
@click.argument('location_name')
@click.argument('ipv4_subnet')
@click.pass_obj
def add_location(ctx, location_name, ipv4_subnet):
    """
    Add a new location (data center)
    """
    ipv4_subnet_ip, ipv4_subnet_prefixlen = ipv4_subnet.split('/')
    ipv4_subnet_prefixlen = int(ipv4_subnet_prefixlen)
    data = {
        'ipv4_subnet_ip': ipv4_subnet_ip,
        'ipv4_subnet_prefixlen': ipv4_subnet_prefixlen
    }
    action('Adding location {location_name} with IPv4 subnet {ipv4_subnet}..', **vars())
    ctx.manager.put('/locations/{}'.format(location_name), data=data)
    ok()


@manager.group(invoke_without_command=True, cls=AliasedGroup)
@click.pass_context
def networks(ctx):
    """
    Show all available networks
    """
    if not ctx.invoked_subcommand:
        ctx = ctx.obj
        data = ctx.manager.get('/networks')
        rows = []
        for row in data['networks']:
            row['ipv6_subnet'] = IPNetwork('{}/{}'.format(row['ipv6_subnet_ip'], row['ipv6_subnet_prefixlen']))
            row['ipv6_routes'] = ', '.join([
                '{destination_ip}/{destination_prefixlen} via {gateway}'.format(**r) for r in row['ipv6_routes']])
            rows.append(row)
        rows.sort(key=itemgetter('zone_name'))
        print_table('zone_name location_name ipv6_subnet ipv6_gateway ipv6_dns ipv6_routes'.split(), rows)


@networks.command('add')
@click.argument('zone_name')
@click.argument('location_name')
@click.argument('ipv6_subnet')
@click.argument('ipv6_gateway')
@click.option('--ipv6-dns', help='Use specific nameserver')
@click.pass_obj
def add_network(ctx, zone_name, location_name, ipv6_subnet, ipv6_gateway, ipv6_dns):
    """
    Add a new network
    """
    ipv6_subnet_ip, ipv6_subnet_prefixlen = ipv6_subnet.split('/')
    ipv6_subnet_prefixlen = int(ipv6_subnet_prefixlen)
    data = {
        'ipv6_subnet_ip': ipv6_subnet_ip,
        'ipv6_subnet_prefixlen': ipv6_subnet_prefixlen,
        'ipv6_gateway': ipv6_gateway,
        'ipv6_dns': ipv6_dns
    }
    action('Adding network for zone {zone_name} and location {location_name} with subnet {ipv6_subnet}..', **vars())
    ctx.manager.put('/networks/{}/{}'.format(location_name, zone_name), data=data)
    ok()


@manager.group('external-services', invoke_without_command=True, cls=AliasedGroup)
@click.pass_context
def external_services(ctx):
    """
    Show all available external services
    """
    if not ctx.invoked_subcommand:
        data = ctx.obj.manager.get('/external-services')
        rows = []
        for row in data['external_services']:
            rows.append(row)
        rows.sort(key=itemgetter('service_name'))
        print_table('service_name service_version zone_name external_host external_port'.split(), rows)


@external_services.command()
@click.argument('zone_name')
@click.argument('service_name')
@click.argument('service_version')
@click.argument('external_host')
@click.argument('external_port', type=int)
@click.pass_obj
def add(ctx, zone_name, service_name, service_version, external_host, external_port):
    action('Adding {}:{} in zone {}..'.format(service_name, service_version, zone_name))
    data = {
        'external_host': external_host,
        'external_port': external_port,
    }
    ctx.manager.put('/external-services/{}/{}/{}'.format(zone_name, service_name, service_version), data=data)
    ok()


@manager.command()
@click.pass_obj
def permissions(ctx):
    """
    List all permissions
    """
    print_permissions(ctx.manager.permissions())
