/*
    PGExt - pygame extension - filter
    Author: Josef Vanžura
    Required: Python 2.5+, PyGame 1.8.1+, SDL 1.2
*/

#include "pgext.h"

#define PGEXT_VERSION 6

#define MAX_SHADOW_VALUE 20

/* simple horizontal/vertical blur */
static void _blur( SDL_Surface *source, SDL_Surface *target, int radius, char vertical ){
    Uint8 r,g,b,a;
    double sr,sg,sb,sa;
    int color_cnt = 0;
    int x,y,ra,n,s,o1,o2,pxn;
    //double r2;
    int pxcnt = 0;

    /* Get sdl_surface attributes */
    int w = source->w;
    int h = source->h;

    /* Vertical/horizonal depending values */
    s = (vertical)?h:w;
    o1 = (vertical)?1:0;
    o2 = (vertical)?0:1;
    //r2 = radius*2+1;

    SDL_LockSurface(source);
    SDL_LockSurface(target);
    for( y = 0; y < h; y++ ){
        for( x = 0; x < w; x++ ){
            color_cnt = 0;sr = 0;sg = 0;sb = 0;sa = 0;
            ra = -radius;
            n = (vertical)?y:x;

            /* Sum of pixel within radius */
            while( ra <= radius ){
                if( (n+ra) >= 0 && (n+ra) < s ){
                    pxn = pxcnt + (int)( ((ra * o1) * w) + (ra * o2));
                    getPixelN_RGBA(source, pxn, &r, &g, &b, &a );
                    sr += r;sg += g;sb += b;sa += a;
                    color_cnt++;
                }
                ra++;
            }
            pxcnt++;
            if(sa == 0) continue;
            if(color_cnt <= 1) continue;

            /* average channel value */
            sr = sr / color_cnt;
            sg = sg / color_cnt;
            sb = sb / color_cnt;
            sa = sa / color_cnt;

            /* map & set pixel */
            setPixel_RGBA(target, x, y, (Uint8)sr, (Uint8)sg, (Uint8)sb, (Uint8)sa);
        }
    }
    SDL_UnlockSurface(source);
    SDL_UnlockSurface(target);
}


/*
    Simple random noise. [RGB]+RAND
*/
static PyObject* noise( PyObject* self, PyObject* args )
{
    PyObject* py_surface = NULL;
    SDL_Surface *surface;
    int w,h,x,y = 0;
    Uint32 pixel;
    SDL_PixelFormat *format = NULL;

    int cnt = 0;
    int value = 255;
    int density = 5;
    Uint8 r,g,b,a;
    int rnd,tr,tg,tb;

    PY_PARSEARGS_BEGIN
        "O|ii", &py_surface, &value, &density
    PY_PARSEARGS_END

    getSurfaceAttrsSurf(py_surface, &w, &h, &format, &surface);
    //srand();

    cnt = (w * h) / density;
    for( ;; ){

        x = (rand() % w);
        y = (rand() % h);

        pixel = getPixel(surface, x, y);
        SDL_GetRGBA( pixel, format, &r, &g, &b, &a );

        rnd = (value / 2) - (rand() % value);
        tr = r + rnd;
        tg = g + rnd;
        tb = b + rnd;
        r = clampUint8(tr);
        g = clampUint8(tg);
        b = clampUint8(tb);

        pixel = SDL_MapRGBA( format, r, g, b, a );
        setPixel(surface, x, y, pixel);

        cnt--;
        if(cnt < 0) break;
    }

/*    for( y = 0; y < h; y ++ ){
        for( x = 0; x < w; x ++ ){
            if( rand()%density != 0 ) continue;
            pixel = getPixel(surface, x, y);
            SDL_GetRGBA( pixel, format, &r, &g, &b, &a );

            rnd = (value / 2) - (rand() % value);
            tr = r + rnd;
            tg = g + rnd;
            tb = b + rnd;
            r = clampUint8(tr);
            g = clampUint8(tg);
            b = clampUint8(tb);

            pixel = SDL_MapRGBA( format, r, g, b, a );
            setPixel(surface, x, y, pixel);
        }
    }
*/
    return Py_None;
}


/*
    noise blur
*/
static PyObject* noiseBlur( PyObject* self, PyObject* args )
{
    PyObject* py_surface = NULL;
    SDL_Surface *surface;
    int w,h,x,y;
    SDL_PixelFormat *format = NULL;

    int radius = 5;
    char blend = 0;
    Uint8 r,g,b,a,tr,tg,tb,ta;
    int dx,dy;

    PY_PARSEARGS_BEGIN
        "O|ib", &py_surface, &radius, &blend
    PY_PARSEARGS_END

    getSurfaceAttrsSurf(py_surface, &w, &h, &format, &surface);
    //srand();

    for( y = 0; y < h; y ++ ){
        for( x = 0; x < w; x ++ ){
            dx = x + ((rand() % (radius * 2)) - radius);
            dy = y + ((rand() % (radius * 2)) - radius);
            if( dx < 0 || dx >= w || dx == x ) continue;
            if( dy < 0 || dy >= h || dy == y ) continue;
            getPixel_RGBA(surface, dx, dy, &r, &g, &b, &a );
            if(blend){
                getPixel_RGBA(surface, x, y, &tr, &tg, &tb, &ta );
                r = (tr + r) / 2;
                g = (tg + g) / 2;
                b = (tb + b) / 2;
            }
            setPixel_RGBA(surface, x, y, r, g, b, a);
        }
    }

    return Py_None;
}


/*
    Simple image scratching.
*/
static PyObject* scratch( PyObject* self, PyObject* args )
{
    PyObject* py_surface = NULL;
    SDL_Surface *surface;
    int w,h,x,y = 0;
    Uint32 pixel;
    SDL_PixelFormat *format = NULL;

    int offset = 10;
    int rnd,tx;

    SDL_Surface *surface_copy = NULL;

    PY_PARSEARGS_BEGIN
        "O|i", &py_surface, &offset
    PY_PARSEARGS_END

    getSurfaceAttrsSurf(py_surface, &w, &h, &format, &surface);

    surface_copy = SDL_ConvertSurface(surface, format, 0);

    //srand();

    for( y = 0; y < h; y ++ ){
        rnd = (rand() % (offset*2)) - offset;
        for( x = 0; x < w; x ++ ){
            tx = rnd + x;
            if( tx < 0 ) continue;
            if( tx >= w ) continue;
            pixel = getPixel(surface_copy, x, y);
            setPixel(surface, tx, y, pixel);
        }
    }

    SDL_FreeSurface(surface_copy);
    return Py_None;
}

/*
    Simple image scratching.
*/
static PyObject* pixelize( PyObject* self, PyObject* args )
{
    PyObject* py_surface = NULL;
    SDL_Surface *surface;
    int w,h,x,y,rx,ry = 0;
    Uint32 pixel;
    SDL_PixelFormat *format = NULL;

    int offset = 10;
    int halfoffset = 5;

    PY_PARSEARGS_BEGIN
        "O|ib", &py_surface, &offset
    PY_PARSEARGS_END

    halfoffset = (int) (((float) offset) / 2);

    getSurfaceAttrsSurf(py_surface, &w, &h, &format, &surface);

    for( y = 0; y < h; y ++ ){
        for( x = 0; x < w; x ++ ){
            rx = ((x / offset) * offset) + halfoffset;
            ry = ((y / offset) * offset) + halfoffset;
            if(x == rx && y == ry) continue;
            if(rx >= w) rx = w - 1;
            if(ry >= h) ry = h - 1;
            pixel = getPixel(surface, rx, ry);
            setPixel(surface, x, y, pixel);

        }
    }

    return Py_None;
}


/*
    Horizontal/vertical gradient.
*/
static PyObject* gradient( PyObject* self, PyObject* args )
{

    SDL_Surface *surface = NULL;
    PyObject *py_fcolor = NULL, *py_tcolor = NULL;
    SDL_PixelFormat *format = NULL;

    Uint32 pixel;
    Uint8 color[4];
    Uint8 fcolor[4];
    Uint8 tcolor[4];
    int x,y,i,q,w,h,dR,dG,dB,dA,rows,cols;
    char vertical = 0;
    float ratio = 0.0;
    double nn,delta;


    PY_PARSEARGS_BEGIN
        "(ii)OO|bf", &w, &h, &py_fcolor, &py_tcolor, &vertical, &ratio
    PY_PARSEARGS_END

    if(!RGBAFromColorObj (py_fcolor, fcolor))
        return Py_None;

    if(!RGBAFromColorObj (py_tcolor, tcolor))
        return Py_None;

    dR = (int)(tcolor[0]-fcolor[0]);
    dG = (int)(tcolor[1]-fcolor[1]);
    dB = (int)(tcolor[2]-fcolor[2]);
    dA = (int)(tcolor[3]-fcolor[3]);

    surface = createSurface( w, h );
    format = surface->format;

    rows = vertical?h:w;
    cols = vertical?w:h;

    SDL_LockSurface (surface);
    for( i = 0; i < rows; i ++ ){
        delta = ((i+1)/(float)(rows+1));
        nn = pow( delta, ratio );

        color[0] = (Uint8)( fcolor[0] + ( dR * nn ) );
        color[1] = (Uint8)( fcolor[1] + ( dG * nn ) );
        color[2] = (Uint8)( fcolor[2] + ( dB * nn ) );
        color[3] = (Uint8)( fcolor[3] + ( dA * nn ) );

        pixel = SDL_MapRGBA( format, color[0], color[1], color[2], color[3] );
        for( q = 0; q < cols; q ++ ){
            x = vertical?q:i;
            y = vertical?i:q;
            setPixel(surface, x, y, pixel);
        }
    }
    SDL_UnlockSurface (surface);

    return PySurface_New( surface );
}

/*
    Create shadow from RGBA surface.
*/
static PyObject* shadow(PyObject* self, PyObject* args)
{
    PyObject *py_surface = NULL, *py_color = NULL;
    SDL_Surface *surface;
    int w,h,x,y;
    int rw,rh;
    int offset = 0;
    SDL_PixelFormat *format;
    Uint8 r,g,b,a;

    SDL_Surface *target_surface = NULL;
    SDL_Surface *temp_surface = NULL;

    Uint8 color[4];

    int radius = 5;
    float opacity = 1.0;
    char resized = 0;

    PY_PARSEARGS_BEGIN
        "OO|ibf", &py_surface, &py_color, &radius, &resized, &opacity
    PY_PARSEARGS_END

    if(!RGBAFromColorObj (py_color, color))
        return Py_None;

    if( radius > MAX_SHADOW_VALUE )
        radius = MAX_SHADOW_VALUE;

    getSurfaceAttrsSurf(py_surface, &w, &h, &format, &surface);

    if(format->BytesPerPixel == 3){
        return RAISE (PyExc_ValueError, "input surface must be 32bit surface not 24bit");
    }

    if( resized ){
        /* Set offsets for resized surface */
        rw = w + (radius * 2);
        rh = h + (radius * 2);
        offset = radius;
    } else {
        rw = w;
        rh = h;
        offset = 0;
    }

    /* Helper surfaces */
    target_surface = createSurface( rw, rh );
    temp_surface = createSurface( rw, rh );

    /* Lock surfaces for direct pixel set */
    SDL_LockSurface (temp_surface);
    SDL_LockSurface (target_surface);
    /* Set color & copy surface to target */
    for( y = 0; y < h; y ++ ){
        for( x = 0; x < w; x ++ ){
            getPixel_RGBA(surface, x, y, &r, &g, &b, &a );
            a = (Uint8) (((double)a)*opacity);
            setPixel_RGBA(target_surface, x + offset, y + offset, color[0], color[1], color[2], a);
        }
    }
    SDL_UnlockSurface (temp_surface);
    SDL_UnlockSurface (target_surface);

    /* Switch target => temp with horizontal blur */
    _blur( target_surface, temp_surface, radius, 0 );
    /* Switch temp => target with vertical blur */
    _blur( temp_surface, target_surface, radius, 1 );

    /* Temp must be free */
    SDL_FreeSurface( temp_surface );

    return PySurface_New( target_surface );
}

/*
    Simple&fast blur
*/
static PyObject* blur(PyObject* self, PyObject* args)
{

    PyObject* py_surface = NULL;
    PySurfaceObject *surface;
    int w,h;

    SDL_Surface *temp_surface = NULL;
    unsigned int radius = 5;

    /* Parsovani pyArgumentu - radius by melo byt unsigned int */
    PY_PARSEARGS_BEGIN
        "O|i", &py_surface, &radius
    PY_PARSEARGS_END

    surface = (PySurfaceObject*) py_surface;
    w = surface->surf->w;
    h = surface->surf->h;

    /* Kopie je potreba kvuli zdrojovym pixelum */
    temp_surface = createSurface( w, h );

    /* Switch original => temp with horizontal blur */
    _blur( surface->surf, temp_surface, radius, 0 );
    /* Switch temp => target with vertical blur */
    _blur( temp_surface, surface->surf, radius, 1 );

    SDL_FreeSurface( temp_surface );

    return Py_None;
}

/*
    Simple&fast blur
*/
static PyObject* hvBlur(PyObject* self, PyObject* args)
{

    PyObject* py_surface = NULL;
    PySurfaceObject *surface;
    SDL_Surface *copy_surface = NULL;
    SDL_PixelFormat *format;
    unsigned int radius = 5;
    char vertical = 0;

    /* Parsovani pyArgumentu - radius by melo byt unsigned int */
    PY_PARSEARGS_BEGIN
        "O|ib", &py_surface, &radius, &vertical
    PY_PARSEARGS_END

    surface = (PySurfaceObject*) py_surface;
    format = (SDL_PixelFormat *)surface->surf->format;

    /* Kopie je potreba kvuli zdrojovym pixelum */
    copy_surface = SDL_ConvertSurface(surface->surf, format, 0);

    /* Switch original => temp with vertical blur */
    _blur( copy_surface, surface->surf, radius, vertical );

    SDL_FreeSurface(copy_surface);

    return Py_None;
}

static PyMethodDef FiltersMethods[] = {
    {"gradient", gradient, METH_VARARGS, "Create vertical/horizontal gradient"},
    {"blur", blur, METH_VARARGS, "Blur surface"},
    {"hvBlur", hvBlur, METH_VARARGS, "Horizonal/vertical blur"},
    {"noise", noise, METH_VARARGS, "Simple surface noise"},
    {"noiseBlur", noiseBlur, METH_VARARGS, "Simple surface noise"},
    {"shadow", shadow, METH_VARARGS, "Create RGBA surface shadow"},
    {"scratch", scratch, METH_VARARGS, "Create RGBA surface shadow"},
    {"pixelize", pixelize, METH_VARARGS, "Fast pixelize effect"},
    {NULL, NULL, NULL, NULL}
};

PyMODINIT_FUNC initfilters(void)
{
    PyObject *module = Py_InitModule("filters", FiltersMethods);
    import_pygame_surface();
    import_pygame_color();
    PyObject_SetAttrString(module, "version", PyLong_FromLong(PGEXT_VERSION));
}
