#!/usr/bin/python
# -*- coding: utf8 -*-
# -*- Mode: Python; py-indent-offset: 4 -*-

"""Method that send message on RabbitMQ or display it on stdout.

.. moduleauthor:: Pierre Leray <pierreleray64@gmail.com>

"""

import logging
import pika
import sys
from messager.utils import RABBIT_MQ_CREDENTIAL_PROBLEM
from messager.utils import RABBIT_MQ_CONNECTION_PROBLEM
from messager.utils import INCORRECT_MESSAGE, MISSING_SITE_ID
from messager.utils import MISSING_NON_DST_TIMEZONE, MISSING_VARIABLE_ID
from messager.utils import MISSING_DST_TIMEZONE, MISSING_DATE, MISSING_VALUE
from time import tzname
from datetime import datetime
import json
import smtplib


def validate_message(message):
    """Method that validate entrance message.

        :param message: Message receive through RabbitMQ in JSON.
        :type topic: str.
    """
    error = None
    validated_message = None

    try:
        validated_message = json.loads(message)
        if "siteID" not in validated_message:
            error = INCORRECT_MESSAGE % MISSING_SITE_ID % message
        elif "variableID" not in validated_message:
            error = INCORRECT_MESSAGE % MISSING_VARIABLE_ID % message
        elif "value" not in validated_message:
            error = INCORRECT_MESSAGE % MISSING_VALUE % message
        elif "date" not in validated_message:
            error = INCORRECT_MESSAGE % MISSING_DATE % message
        elif "dstTimezone" not in validated_message:
            error = INCORRECT_MESSAGE % MISSING_DST_TIMEZONE % message
        elif "nonDstTimezone" not in validated_message:
            error = INCORRECT_MESSAGE % MISSING_NON_DST_TIMEZONE % message
    except ValueError:
        error = INCORRECT_MESSAGE % message

    return error, validated_message


class RabbitMQMessager(object):

    """This class send message to RabbitMQ or in stdout following user choice.

    """

    def __init__(self, username, password, host):
        """Constructor.

            :param username: Name of the RabbitMQ user.
            :type username: str.

            :param password: Password of the RabbitMQ user.
            :type password: str.

            :param host: Host target for RabbitMQ.
            :type host: str.
        """
        # Logger and channel initialization
        self.logger = logging.getLogger("messager.messager")
        self.channel = None
        # If we have a username and password for RabbitMQ
        if username is not None and password is not None:
            try:
                # We try to connect to RabbitMQ with this credential
                self.connection = pika.BlockingConnection(
                    pika.ConnectionParameters(
                        host=host,
                        credentials=pika.PlainCredentials(username, password)))
                # If our connection was successful we retrieve a channel
                self.channel = self.connection.channel()
            except pika.exceptions.ConnectionClosed:
                # Else if our credential were wrong, we log according message
                error = RABBIT_MQ_CREDENTIAL_PROBLEM % (
                    username,
                    password,
                    host)
                self.logger.error(error)
            except pika.exceptions.AMQPConnectionError:
                # Else if RabbitMQ is not available in this location,
                # we log according message
                error = RABBIT_MQ_CONNECTION_PROBLEM % (
                    username,
                    password,
                    host)
                self.logger.error(error)

    def is_connected(self):
        """Method that return connection status

            :returns:  bool -- Connection status. True for correct connection.
        """
        return True if self.channel is not None else False

    def send_message(self, topic, site_name, var_name, data, out=sys.stdout):
        """Format parameter into json string for phase platform.

            :param topic: Channel where to send message (RabbitMQ filter).
            :type topic: str.

            :param site_name: Name of the site.
            :type site_name: str.

            :param var_name: Variable name.
            :type var_name: str.

            :param data: Data send to send.
            :type data: str.

            :param out: Stdout redirection in case of test (default: stdout).
            :type out: StringIO.
        """
        # We create our phase message
        json_message = {
            'siteID': site_name,
            'variableID': var_name,
            'message': data,
            'date': datetime.utcnow().isoformat('T'),
            'dstTimezone': tzname[1],
            'nonDstTimezone': tzname[0]
        }
        # And we send it using our send method defined previously
        self.send(topic, json.dumps(json_message), out)

    def send_worker(self, topic, message, out=sys.stdout):
        """Method that send a message with a topic.

            :param topic: Channel where to send message (RabbitMQ filter).
            :type topic: str.

            :param message: Message to send to RabbitMQ.
            :type message: str.

            :param out: Stdout redirection in case of test (default: stdout).
            :type out: StringIO.
        """
        # We log message we want to send to keep a trace
        self.logger.error(message)
        # If channel is available
        if self.channel is not None:
            # We send a message on this channel
            self.channel.queue_declare(queue=topic)
            self.channel.basic_publish(
                exchange='', routing_key=topic, body=message)
        else:
            # Else we print it in stdout
            out.write("%s %s" % (unicode(message), "\n"))

    def consume_worker(self, topics, callback):
        """Method that wait for a message on RabbitMQ topics channels.

            :param topic: Channel where to send message (RabbitMQ filter).
            :type topic: str.

            :param callback: Function that compute result of message (async).
            :type callback: Function.
        """
        # We try to connect to our channel and wait for a message
        try:
            for topic in topics:
                self.channel.queue_declare(queue=topic)
                self.channel.basic_consume(callback, queue=topic, no_ack=True)
            self.channel.start_consuming()
        except pika.exceptions.ConnectionClosed:
            # If something goes wrong we pass silently
            pass

    def send(self, topic, message, out=sys.stdout):
        """Method that send a message with a topic.

            :param topic: Channel where to send message (RabbitMQ filter).
            :type topic: str.

            :param message: Message to send to RabbitMQ.
            :type message: str.

            :param out: Stdout redirection in case of test (default: stdout).
            :type out: StringIO.
        """
        # We log message we want to send to keep a trace
        self.logger.error(message)
        # If channel is available
        if self.channel is not None:
            # We send a message on this channel
            self.channel.exchange_declare(exchange=topic, type='fanout')
            self.channel.basic_publish(
                exchange=topic,
                routing_key=topic,
                body=message)
        else:
            # Else we print it in stdout
            out.write("%s %s" % (unicode(message), "\n"))

    def consume(self, topics, callback):
        """Method that wait for a message on RabbitMQ topics channels.

            :param topic: Channel where to send message (RabbitMQ filter).
            :type topic: str.

            :param callback: Function that compute result of message (async).
            :type callback: Function.
        """
        # We try to connect to our channel and wait for a message
        try:
            for topic in topics:
                self.channel.exchange_declare(exchange=topic, type='fanout')
                result = self.channel.queue_declare(exclusive=True)
                queue_name = result.method.queue
                self.channel.queue_bind(exchange=topic, queue=queue_name)
                self.channel.basic_consume(
                    callback,
                    queue=queue_name,
                    no_ack=True)
            self.channel.start_consuming()
        except pika.exceptions.ConnectionClosed as excpt:
            # If something goes wrong we pass silently
            msg = "Something goes wrong: %s" % excpt
            self.logger.error(msg)

    def send_email(self, options, gmail_username, gmail_password):
        """Method that send an email using gmail SMTP

            :param target: Dict containing subject, body, e-mail target.
            :type target: Dict.

            :param gmail_username: Gmail username account.
            :type gmail_username: str.

            :param gmail_password: Password of gmail account.
            :type gmail_password: str.
        """
        headers = ["From: %s" % gmail_username,
                   "Subject: %s" % options["subject"],
                   "To: %s" % options["target"],
                   "MIME-Version: 1.0",
                   "Content-Type: text/html"]
        headers = "\r\n".join(headers)

        message = "%s\r\n\r\n%s" % (headers, options["body"])

        smtp_url = "smtp.gmail.com:587"

        log_message = "\n%s" % message
        self.logger.error(log_message)

        server = smtplib.SMTP(smtp_url)
        server.starttls()
        server.login(gmail_username, gmail_password)
        server.sendmail(options["target"], options["target"], message)
        server.quit()
