#!/usr/bin/env python
"""Unit tests for asyoutypeformatter.py"""

# Based on original Java code:
#     java/test/com/google/i18n/phonenumbers/PhoneNumberTest.java
# Copyright (C) 2009 Google Inc.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
# http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
import unittest

import pathfix
pathfix.fix()

from phonenumbers import AsYouTypeFormatter
from testphonenumberutil import insert_test_metadata, reinstate_real_metadata


class AsYouTypeFormatterTest(unittest.TestCase):
    """Unit tests for AsYouTypeFormatter.java

    Note that these tests use the metadata contained in the files in
    tests/data, not the normal metadata files, so should not be used for
    regression test purposes - these tests are illustrative only and test
    functionality.
    """

    def setUp(self):
        insert_test_metadata()

    def tearDown(self):
        reinstate_real_metadata()

    def testInvalidRegion(self):
        formatter = AsYouTypeFormatter("ZZ")
        self.assertEquals("+", formatter.input_digit('+'))
        self.assertEquals("+4", formatter.input_digit('4'))
        self.assertEquals("+48 ", formatter.input_digit('8'))
        self.assertEquals("+48 8", formatter.input_digit('8'))
        self.assertEquals("+48 88", formatter.input_digit('8'))
        self.assertEquals("+48 88 1", formatter.input_digit('1'))
        self.assertEquals("+48 88 12", formatter.input_digit('2'))
        self.assertEquals("+48 88 123", formatter.input_digit('3'))
        self.assertEquals("+48 88 123 1", formatter.input_digit('1'))
        self.assertEquals("+48 88 123 12", formatter.input_digit('2'))

        formatter.clear()
        self.assertEquals("6", formatter.input_digit('6'))
        self.assertEquals("65", formatter.input_digit('5'))
        self.assertEquals("650", formatter.input_digit('0'))
        self.assertEquals("6502", formatter.input_digit('2'))
        self.assertEquals("65025", formatter.input_digit('5'))
        self.assertEquals("650253", formatter.input_digit('3'))

    def testAYTFUS(self):
        formatter = AsYouTypeFormatter("US")
        self.assertEquals("6", formatter.input_digit('6'))
        self.assertEquals("65", formatter.input_digit('5'))
        self.assertEquals("650", formatter.input_digit('0'))
        self.assertEquals("650 2", formatter.input_digit('2'))
        self.assertEquals("650 25", formatter.input_digit('5'))
        self.assertEquals("650 253", formatter.input_digit('3'))
        # Note this is how a US local number (without area code) should be formatted.
        self.assertEquals("650 2532", formatter.input_digit('2'))
        self.assertEquals("650 253 22", formatter.input_digit('2'))
        self.assertEquals("650 253 222", formatter.input_digit('2'))
        self.assertEquals("650 253 2222", formatter.input_digit('2'))

        formatter.clear()
        self.assertEquals("1", formatter.input_digit('1'))
        self.assertEquals("16", formatter.input_digit('6'))
        self.assertEquals("1 65", formatter.input_digit('5'))
        self.assertEquals("1 650", formatter.input_digit('0'))
        self.assertEquals("1 650 2", formatter.input_digit('2'))
        self.assertEquals("1 650 25", formatter.input_digit('5'))
        self.assertEquals("1 650 253", formatter.input_digit('3'))
        self.assertEquals("1 650 253 2", formatter.input_digit('2'))
        self.assertEquals("1 650 253 22", formatter.input_digit('2'))
        self.assertEquals("1 650 253 222", formatter.input_digit('2'))
        self.assertEquals("1 650 253 2222", formatter.input_digit('2'))

        formatter.clear()
        self.assertEquals("0", formatter.input_digit('0'))
        self.assertEquals("01", formatter.input_digit('1'))
        self.assertEquals("011 ", formatter.input_digit('1'))
        self.assertEquals("011 4", formatter.input_digit('4'))
        self.assertEquals("011 44 ", formatter.input_digit('4'))
        self.assertEquals("011 44 6", formatter.input_digit('6'))
        self.assertEquals("011 44 61", formatter.input_digit('1'))
        self.assertEquals("011 44 6 12", formatter.input_digit('2'))
        self.assertEquals("011 44 6 123", formatter.input_digit('3'))
        self.assertEquals("011 44 6 123 1", formatter.input_digit('1'))
        self.assertEquals("011 44 6 123 12", formatter.input_digit('2'))
        self.assertEquals("011 44 6 123 123", formatter.input_digit('3'))
        self.assertEquals("011 44 6 123 123 1", formatter.input_digit('1'))
        self.assertEquals("011 44 6 123 123 12", formatter.input_digit('2'))
        self.assertEquals("011 44 6 123 123 123", formatter.input_digit('3'))

        formatter.clear()
        self.assertEquals("0", formatter.input_digit('0'))
        self.assertEquals("01", formatter.input_digit('1'))
        self.assertEquals("011 ", formatter.input_digit('1'))
        self.assertEquals("011 5", formatter.input_digit('5'))
        self.assertEquals("011 54 ", formatter.input_digit('4'))
        self.assertEquals("011 54 9", formatter.input_digit('9'))
        self.assertEquals("011 54 91", formatter.input_digit('1'))
        self.assertEquals("011 54 9 11", formatter.input_digit('1'))
        self.assertEquals("011 54 9 11 2", formatter.input_digit('2'))
        self.assertEquals("011 54 9 11 23", formatter.input_digit('3'))
        self.assertEquals("011 54 9 11 231", formatter.input_digit('1'))
        self.assertEquals("011 54 9 11 2312", formatter.input_digit('2'))
        self.assertEquals("011 54 9 11 2312 1", formatter.input_digit('1'))
        self.assertEquals("011 54 9 11 2312 12", formatter.input_digit('2'))
        self.assertEquals("011 54 9 11 2312 123", formatter.input_digit('3'))
        self.assertEquals("011 54 9 11 2312 1234", formatter.input_digit('4'))

        formatter.clear()
        self.assertEquals("0", formatter.input_digit('0'))
        self.assertEquals("01", formatter.input_digit('1'))
        self.assertEquals("011 ", formatter.input_digit('1'))
        self.assertEquals("011 2", formatter.input_digit('2'))
        self.assertEquals("011 24", formatter.input_digit('4'))
        self.assertEquals("011 244 ", formatter.input_digit('4'))
        self.assertEquals("011 244 2", formatter.input_digit('2'))
        self.assertEquals("011 244 28", formatter.input_digit('8'))
        self.assertEquals("011 244 280", formatter.input_digit('0'))
        self.assertEquals("011 244 280 0", formatter.input_digit('0'))
        self.assertEquals("011 244 280 00", formatter.input_digit('0'))
        self.assertEquals("011 244 280 000", formatter.input_digit('0'))
        self.assertEquals("011 244 280 000 0", formatter.input_digit('0'))
        self.assertEquals("011 244 280 000 00", formatter.input_digit('0'))
        self.assertEquals("011 244 280 000 000", formatter.input_digit('0'))

        formatter.clear()
        self.assertEquals("+", formatter.input_digit('+'))
        self.assertEquals("+4", formatter.input_digit('4'))
        self.assertEquals("+48 ", formatter.input_digit('8'))
        self.assertEquals("+48 8", formatter.input_digit('8'))
        self.assertEquals("+48 88", formatter.input_digit('8'))
        self.assertEquals("+48 88 1", formatter.input_digit('1'))
        self.assertEquals("+48 88 12", formatter.input_digit('2'))
        self.assertEquals("+48 88 123", formatter.input_digit('3'))
        self.assertEquals("+48 88 123 1", formatter.input_digit('1'))
        self.assertEquals("+48 88 123 12", formatter.input_digit('2'))
        self.assertEquals("+48 88 123 12 1", formatter.input_digit('1'))
        self.assertEquals("+48 88 123 12 12", formatter.input_digit('2'))

    def testAYTFUSFullWidthCharacters(self):
        formatter = AsYouTypeFormatter("US")
        self.assertEquals(u"\uFF16", formatter.input_digit(u'\uFF16'))
        self.assertEquals(u"\uFF16\uFF15", formatter.input_digit(u'\uFF15'))
        self.assertEquals("650", formatter.input_digit(u'\uFF10'))
        self.assertEquals("650 2", formatter.input_digit(u'\uFF12'))
        self.assertEquals("650 25", formatter.input_digit(u'\uFF15'))
        self.assertEquals("650 253", formatter.input_digit(u'\uFF13'))
        self.assertEquals("650 2532", formatter.input_digit(u'\uFF12'))
        self.assertEquals("650 253 22", formatter.input_digit(u'\uFF12'))
        self.assertEquals("650 253 222", formatter.input_digit(u'\uFF12'))
        self.assertEquals("650 253 2222", formatter.input_digit(u'\uFF12'))

    def testAYTFUSMobileShortCode(self):
        formatter = AsYouTypeFormatter("US")
        self.assertEquals("*", formatter.input_digit('*'))
        self.assertEquals("*1", formatter.input_digit('1'))
        self.assertEquals("*12", formatter.input_digit('2'))
        self.assertEquals("*121", formatter.input_digit('1'))
        self.assertEquals("*121#", formatter.input_digit('#'))

    def testAYTFUSVanityNumber(self):
        formatter = AsYouTypeFormatter("US")
        self.assertEquals("8", formatter.input_digit('8'))
        self.assertEquals("80", formatter.input_digit('0'))
        self.assertEquals("800", formatter.input_digit('0'))
        self.assertEquals("800 ", formatter.input_digit(' '))
        self.assertEquals("800 M", formatter.input_digit('M'))
        self.assertEquals("800 MY", formatter.input_digit('Y'))
        self.assertEquals("800 MY ", formatter.input_digit(' '))
        self.assertEquals("800 MY A", formatter.input_digit('A'))
        self.assertEquals("800 MY AP", formatter.input_digit('P'))
        self.assertEquals("800 MY APP", formatter.input_digit('P'))
        self.assertEquals("800 MY APPL", formatter.input_digit('L'))
        self.assertEquals("800 MY APPLE", formatter.input_digit('E'))

    def testAYTFAndRememberPositionUS(self):
        formatter = AsYouTypeFormatter("US")
        self.assertEquals("1", formatter.input_digit('1', remember_position=True))
        self.assertEquals(1, formatter.get_remembered_position())
        self.assertEquals("16", formatter.input_digit('6'))
        self.assertEquals("1 65", formatter.input_digit('5'))
        self.assertEquals(1, formatter.get_remembered_position())
        self.assertEquals("1 650", formatter.input_digit('0', remember_position=True))
        self.assertEquals(5, formatter.get_remembered_position())
        self.assertEquals("1 650 2", formatter.input_digit('2'))
        self.assertEquals("1 650 25", formatter.input_digit('5'))
        # Note the remembered position for digit "0" changes from 4 to 5, because a space is now
        # inserted in the front.
        self.assertEquals(5, formatter.get_remembered_position())
        self.assertEquals("1 650 253", formatter.input_digit('3'))
        self.assertEquals("1 650 253 2", formatter.input_digit('2'))
        self.assertEquals("1 650 253 22", formatter.input_digit('2'))
        self.assertEquals(5, formatter.get_remembered_position())
        self.assertEquals("1 650 253 222", formatter.input_digit('2', remember_position=True))
        self.assertEquals(13, formatter.get_remembered_position())
        self.assertEquals("1 650 253 2222", formatter.input_digit('2'))
        self.assertEquals(13, formatter.get_remembered_position())
        self.assertEquals("165025322222", formatter.input_digit('2'))
        self.assertEquals(10, formatter.get_remembered_position())
        self.assertEquals("1650253222222", formatter.input_digit('2'))
        self.assertEquals(10, formatter.get_remembered_position())

        formatter.clear()
        self.assertEquals("1", formatter.input_digit('1'))
        self.assertEquals("16", formatter.input_digit('6', remember_position=True))
        self.assertEquals(2, formatter.get_remembered_position())
        self.assertEquals("1 65", formatter.input_digit('5'))
        self.assertEquals("1 650", formatter.input_digit('0'))
        self.assertEquals(3, formatter.get_remembered_position())
        self.assertEquals("1 650 2", formatter.input_digit('2'))
        self.assertEquals("1 650 25", formatter.input_digit('5'))
        self.assertEquals(3, formatter.get_remembered_position())
        self.assertEquals("1 650 253", formatter.input_digit('3'))
        self.assertEquals("1 650 253 2", formatter.input_digit('2'))
        self.assertEquals("1 650 253 22", formatter.input_digit('2'))
        self.assertEquals(3, formatter.get_remembered_position())
        self.assertEquals("1 650 253 222", formatter.input_digit('2'))
        self.assertEquals("1 650 253 2222", formatter.input_digit('2'))
        self.assertEquals("165025322222", formatter.input_digit('2'))
        self.assertEquals(2, formatter.get_remembered_position())
        self.assertEquals("1650253222222", formatter.input_digit('2'))
        self.assertEquals(2, formatter.get_remembered_position())

        formatter.clear()
        self.assertEquals("6", formatter.input_digit('6'))
        self.assertEquals("65", formatter.input_digit('5'))
        self.assertEquals("650", formatter.input_digit('0'))
        self.assertEquals("650 2", formatter.input_digit('2'))
        self.assertEquals("650 25", formatter.input_digit('5'))
        self.assertEquals("650 253", formatter.input_digit('3'))
        self.assertEquals("650 2532", formatter.input_digit('2', remember_position=True))
        self.assertEquals(8, formatter.get_remembered_position())
        self.assertEquals("650 253 22", formatter.input_digit('2'))
        self.assertEquals(9, formatter.get_remembered_position())
        self.assertEquals("650 253 222", formatter.input_digit('2'))
        # No more formatting when semicolon is entered.
        self.assertEquals("650253222;", formatter.input_digit(';'))
        self.assertEquals(7, formatter.get_remembered_position())
        self.assertEquals("650253222;2", formatter.input_digit('2'))

        formatter.clear()
        self.assertEquals("6", formatter.input_digit('6'))
        self.assertEquals("65", formatter.input_digit('5'))
        self.assertEquals("650", formatter.input_digit('0'))
        # No more formatting when users choose to do their own formatting.
        self.assertEquals("650-", formatter.input_digit('-'))
        self.assertEquals("650-2", formatter.input_digit('2', remember_position=True))
        self.assertEquals(5, formatter.get_remembered_position())
        self.assertEquals("650-25", formatter.input_digit('5'))
        self.assertEquals(5, formatter.get_remembered_position())
        self.assertEquals("650-253", formatter.input_digit('3'))
        self.assertEquals(5, formatter.get_remembered_position())
        self.assertEquals("650-253-", formatter.input_digit('-'))
        self.assertEquals("650-253-2", formatter.input_digit('2'))
        self.assertEquals("650-253-22", formatter.input_digit('2'))
        self.assertEquals("650-253-222", formatter.input_digit('2'))
        self.assertEquals("650-253-2222", formatter.input_digit('2'))

        formatter.clear()
        self.assertEquals("0", formatter.input_digit('0'))
        self.assertEquals("01", formatter.input_digit('1'))
        self.assertEquals("011 ", formatter.input_digit('1'))
        self.assertEquals("011 4", formatter.input_digit('4', remember_position=True))
        self.assertEquals("011 48 ", formatter.input_digit('8'))
        self.assertEquals(5, formatter.get_remembered_position())
        self.assertEquals("011 48 8", formatter.input_digit('8'))
        self.assertEquals(5, formatter.get_remembered_position())
        self.assertEquals("011 48 88", formatter.input_digit('8'))
        self.assertEquals("011 48 88 1", formatter.input_digit('1'))
        self.assertEquals("011 48 88 12", formatter.input_digit('2'))
        self.assertEquals(5, formatter.get_remembered_position())
        self.assertEquals("011 48 88 123", formatter.input_digit('3'))
        self.assertEquals("011 48 88 123 1", formatter.input_digit('1'))
        self.assertEquals("011 48 88 123 12", formatter.input_digit('2'))
        self.assertEquals("011 48 88 123 12 1", formatter.input_digit('1'))
        self.assertEquals("011 48 88 123 12 12", formatter.input_digit('2'))

        formatter.clear()
        self.assertEquals("+", formatter.input_digit('+'))
        self.assertEquals("+1", formatter.input_digit('1'))
        self.assertEquals("+1 6", formatter.input_digit('6', remember_position=True))
        self.assertEquals("+1 65", formatter.input_digit('5'))
        self.assertEquals("+1 650", formatter.input_digit('0'))
        self.assertEquals(4, formatter.get_remembered_position())
        self.assertEquals("+1 650 2", formatter.input_digit('2'))
        self.assertEquals(4, formatter.get_remembered_position())
        self.assertEquals("+1 650 25", formatter.input_digit('5'))
        self.assertEquals("+1 650 253", formatter.input_digit('3', remember_position=True))
        self.assertEquals("+1 650 253 2", formatter.input_digit('2'))
        self.assertEquals("+1 650 253 22", formatter.input_digit('2'))
        self.assertEquals("+1 650 253 222", formatter.input_digit('2'))
        self.assertEquals(10, formatter.get_remembered_position())

        formatter.clear()
        self.assertEquals("+", formatter.input_digit('+'))
        self.assertEquals("+1", formatter.input_digit('1'))
        self.assertEquals("+1 6", formatter.input_digit('6', remember_position=True))
        self.assertEquals("+1 65", formatter.input_digit('5'))
        self.assertEquals("+1 650", formatter.input_digit('0'))
        self.assertEquals(4, formatter.get_remembered_position())
        self.assertEquals("+1 650 2", formatter.input_digit('2'))
        self.assertEquals(4, formatter.get_remembered_position())
        self.assertEquals("+1 650 25", formatter.input_digit('5'))
        self.assertEquals("+1 650 253", formatter.input_digit('3'))
        self.assertEquals("+1 650 253 2", formatter.input_digit('2'))
        self.assertEquals("+1 650 253 22", formatter.input_digit('2'))
        self.assertEquals("+1 650 253 222", formatter.input_digit('2'))
        self.assertEquals("+1650253222;", formatter.input_digit(';'))
        self.assertEquals(3, formatter.get_remembered_position())

    def testAYTFGBFixedLine(self):
        formatter = AsYouTypeFormatter("GB")
        self.assertEquals("0", formatter.input_digit('0'))
        self.assertEquals("02", formatter.input_digit('2'))
        self.assertEquals("020", formatter.input_digit('0'))
        self.assertEquals("020 7", formatter.input_digit('7', remember_position=True))
        self.assertEquals(5, formatter.get_remembered_position())
        self.assertEquals("020 70", formatter.input_digit('0'))
        self.assertEquals("020 703", formatter.input_digit('3'))
        self.assertEquals(5, formatter.get_remembered_position())
        self.assertEquals("020 7031", formatter.input_digit('1'))
        self.assertEquals("020 7031 3", formatter.input_digit('3'))
        self.assertEquals("020 7031 30", formatter.input_digit('0'))
        self.assertEquals("020 7031 300", formatter.input_digit('0'))
        self.assertEquals("020 7031 3000", formatter.input_digit('0'))

    def testAYTFGBTollFree(self):
        formatter = AsYouTypeFormatter("gb")
        self.assertEquals("0", formatter.input_digit('0'))
        self.assertEquals("08", formatter.input_digit('8'))
        self.assertEquals("080", formatter.input_digit('0'))
        self.assertEquals("080 7", formatter.input_digit('7'))
        self.assertEquals("080 70", formatter.input_digit('0'))
        self.assertEquals("080 703", formatter.input_digit('3'))
        self.assertEquals("080 7031", formatter.input_digit('1'))
        self.assertEquals("080 7031 3", formatter.input_digit('3'))
        self.assertEquals("080 7031 30", formatter.input_digit('0'))
        self.assertEquals("080 7031 300", formatter.input_digit('0'))
        self.assertEquals("080 7031 3000", formatter.input_digit('0'))

    def testAYTFGBPremiumRate(self):
        formatter = AsYouTypeFormatter("GB")
        self.assertEquals("0", formatter.input_digit('0'))
        self.assertEquals("09", formatter.input_digit('9'))
        self.assertEquals("090", formatter.input_digit('0'))
        self.assertEquals("090 7", formatter.input_digit('7'))
        self.assertEquals("090 70", formatter.input_digit('0'))
        self.assertEquals("090 703", formatter.input_digit('3'))
        self.assertEquals("090 7031", formatter.input_digit('1'))
        self.assertEquals("090 7031 3", formatter.input_digit('3'))
        self.assertEquals("090 7031 30", formatter.input_digit('0'))
        self.assertEquals("090 7031 300", formatter.input_digit('0'))
        self.assertEquals("090 7031 3000", formatter.input_digit('0'))

    def testAYTFNZMobile(self):
        formatter = AsYouTypeFormatter("NZ")
        self.assertEquals("0", formatter.input_digit('0'))
        self.assertEquals("02", formatter.input_digit('2'))
        self.assertEquals("021", formatter.input_digit('1'))
        self.assertEquals("02-11", formatter.input_digit('1'))
        self.assertEquals("02-112", formatter.input_digit('2'))
        # Note the unittest is using fake metadata which might produce non-ideal results.
        self.assertEquals("02-112 3", formatter.input_digit('3'))
        self.assertEquals("02-112 34", formatter.input_digit('4'))
        self.assertEquals("02-112 345", formatter.input_digit('5'))
        self.assertEquals("02-112 3456", formatter.input_digit('6'))

    def testAYTFDE(self):
        formatter = AsYouTypeFormatter("DE")
        self.assertEquals("0", formatter.input_digit('0'))
        self.assertEquals("03", formatter.input_digit('3'))
        self.assertEquals("030", formatter.input_digit('0'))
        self.assertEquals("030/1", formatter.input_digit('1'))
        self.assertEquals("030/12", formatter.input_digit('2'))
        self.assertEquals("030/123", formatter.input_digit('3'))
        self.assertEquals("030/1234", formatter.input_digit('4'))

        # 04134 1234
        formatter.clear()
        self.assertEquals("0", formatter.input_digit('0'))
        self.assertEquals("04", formatter.input_digit('4'))
        self.assertEquals("041", formatter.input_digit('1'))
        self.assertEquals("041 3", formatter.input_digit('3'))
        self.assertEquals("041 34", formatter.input_digit('4'))
        self.assertEquals("04134 1", formatter.input_digit('1'))
        self.assertEquals("04134 12", formatter.input_digit('2'))
        self.assertEquals("04134 123", formatter.input_digit('3'))
        self.assertEquals("04134 1234", formatter.input_digit('4'))

        # 08021 2345
        formatter.clear()
        self.assertEquals("0", formatter.input_digit('0'))
        self.assertEquals("08", formatter.input_digit('8'))
        self.assertEquals("080", formatter.input_digit('0'))
        self.assertEquals("080 2", formatter.input_digit('2'))
        self.assertEquals("080 21", formatter.input_digit('1'))
        self.assertEquals("08021 2", formatter.input_digit('2'))
        self.assertEquals("08021 23", formatter.input_digit('3'))
        self.assertEquals("08021 234", formatter.input_digit('4'))
        self.assertEquals("08021 2345", formatter.input_digit('5'))

        # 00 1 650 253 2250
        formatter.clear()
        self.assertEquals("0", formatter.input_digit('0'))
        self.assertEquals("00", formatter.input_digit('0'))
        self.assertEquals("00 1 ", formatter.input_digit('1'))
        self.assertEquals("00 1 6", formatter.input_digit('6'))
        self.assertEquals("00 1 65", formatter.input_digit('5'))
        self.assertEquals("00 1 650", formatter.input_digit('0'))
        self.assertEquals("00 1 650 2", formatter.input_digit('2'))
        self.assertEquals("00 1 650 25", formatter.input_digit('5'))
        self.assertEquals("00 1 650 253", formatter.input_digit('3'))
        self.assertEquals("00 1 650 253 2", formatter.input_digit('2'))
        self.assertEquals("00 1 650 253 22", formatter.input_digit('2'))
        self.assertEquals("00 1 650 253 222", formatter.input_digit('2'))
        self.assertEquals("00 1 650 253 2222", formatter.input_digit('2'))

    def testAYTFAR(self):
        formatter = AsYouTypeFormatter("AR")
        self.assertEquals("0", formatter.input_digit('0'))
        self.assertEquals("01", formatter.input_digit('1'))
        self.assertEquals("011", formatter.input_digit('1'))
        self.assertEquals("011 7", formatter.input_digit('7'))
        self.assertEquals("011 70", formatter.input_digit('0'))
        self.assertEquals("011 703", formatter.input_digit('3'))
        self.assertEquals("011 7031", formatter.input_digit('1'))
        self.assertEquals("011 7031-3", formatter.input_digit('3'))
        self.assertEquals("011 7031-30", formatter.input_digit('0'))
        self.assertEquals("011 7031-300", formatter.input_digit('0'))
        self.assertEquals("011 7031-3000", formatter.input_digit('0'))

    def testAYTFARMobile(self):
        formatter = AsYouTypeFormatter("AR")
        self.assertEquals("+", formatter.input_digit('+'))
        self.assertEquals("+5", formatter.input_digit('5'))
        self.assertEquals("+54 ", formatter.input_digit('4'))
        self.assertEquals("+54 9", formatter.input_digit('9'))
        self.assertEquals("+54 91", formatter.input_digit('1'))
        self.assertEquals("+54 9 11", formatter.input_digit('1'))
        self.assertEquals("+54 9 11 2",
                                 formatter.input_digit('2'))
        self.assertEquals("+54 9 11 23", formatter.input_digit('3'))
        self.assertEquals("+54 9 11 231", formatter.input_digit('1'))
        self.assertEquals("+54 9 11 2312", formatter.input_digit('2'))
        self.assertEquals("+54 9 11 2312 1", formatter.input_digit('1'))
        self.assertEquals("+54 9 11 2312 12", formatter.input_digit('2'))
        self.assertEquals("+54 9 11 2312 123", formatter.input_digit('3'))
        self.assertEquals("+54 9 11 2312 1234", formatter.input_digit('4'))

    def testAYTFKR(self):
        # +82 51 234 5678
        formatter = AsYouTypeFormatter("KR")
        self.assertEquals("+", formatter.input_digit('+'))
        self.assertEquals("+8", formatter.input_digit('8'))
        self.assertEquals("+82 ", formatter.input_digit('2'))
        self.assertEquals("+82 5", formatter.input_digit('5'))
        self.assertEquals("+82 51", formatter.input_digit('1'))
        self.assertEquals("+82 51-2", formatter.input_digit('2'))
        self.assertEquals("+82 51-23", formatter.input_digit('3'))
        self.assertEquals("+82 51-234", formatter.input_digit('4'))
        self.assertEquals("+82 51-234-5", formatter.input_digit('5'))
        self.assertEquals("+82 51-234-56", formatter.input_digit('6'))
        self.assertEquals("+82 51-234-567", formatter.input_digit('7'))
        self.assertEquals("+82 51-234-5678", formatter.input_digit('8'))

        # +82 2 531 5678
        formatter.clear()
        self.assertEquals("+", formatter.input_digit('+'))
        self.assertEquals("+8", formatter.input_digit('8'))
        self.assertEquals("+82 ", formatter.input_digit('2'))
        self.assertEquals("+82 2", formatter.input_digit('2'))
        self.assertEquals("+82 25", formatter.input_digit('5'))
        self.assertEquals("+82 2-53", formatter.input_digit('3'))
        self.assertEquals("+82 2-531", formatter.input_digit('1'))
        self.assertEquals("+82 2-531-5", formatter.input_digit('5'))
        self.assertEquals("+82 2-531-56", formatter.input_digit('6'))
        self.assertEquals("+82 2-531-567", formatter.input_digit('7'))
        self.assertEquals("+82 2-531-5678", formatter.input_digit('8'))

        # +82 2 3665 5678
        formatter.clear()
        self.assertEquals("+", formatter.input_digit('+'))
        self.assertEquals("+8", formatter.input_digit('8'))
        self.assertEquals("+82 ", formatter.input_digit('2'))
        self.assertEquals("+82 2", formatter.input_digit('2'))
        self.assertEquals("+82 23", formatter.input_digit('3'))
        self.assertEquals("+82 2-36", formatter.input_digit('6'))
        self.assertEquals("+82 2-366", formatter.input_digit('6'))
        self.assertEquals("+82 2-3665", formatter.input_digit('5'))
        self.assertEquals("+82 2-3665-5", formatter.input_digit('5'))
        self.assertEquals("+82 2-3665-56", formatter.input_digit('6'))
        self.assertEquals("+82 2-3665-567", formatter.input_digit('7'))
        self.assertEquals("+82 2-3665-5678", formatter.input_digit('8'))

        # 02-114
        formatter.clear()
        self.assertEquals("0", formatter.input_digit('0'))
        self.assertEquals("02", formatter.input_digit('2'))
        self.assertEquals("021", formatter.input_digit('1'))
        self.assertEquals("02-11", formatter.input_digit('1'))
        self.assertEquals("02-114", formatter.input_digit('4'))

        # 02-1300
        formatter.clear()
        self.assertEquals("0", formatter.input_digit('0'))
        self.assertEquals("02", formatter.input_digit('2'))
        self.assertEquals("021", formatter.input_digit('1'))
        self.assertEquals("02-13", formatter.input_digit('3'))
        self.assertEquals("02-130", formatter.input_digit('0'))
        self.assertEquals("02-1300", formatter.input_digit('0'))

        # 011-456-7890
        formatter.clear()
        self.assertEquals("0", formatter.input_digit('0'))
        self.assertEquals("01", formatter.input_digit('1'))
        self.assertEquals("011", formatter.input_digit('1'))
        self.assertEquals("011-4", formatter.input_digit('4'))
        self.assertEquals("011-45", formatter.input_digit('5'))
        self.assertEquals("011-456", formatter.input_digit('6'))
        self.assertEquals("011-456-7", formatter.input_digit('7'))
        self.assertEquals("011-456-78", formatter.input_digit('8'))
        self.assertEquals("011-456-789", formatter.input_digit('9'))
        self.assertEquals("011-456-7890", formatter.input_digit('0'))

        # 011-9876-7890
        formatter.clear()
        self.assertEquals("0", formatter.input_digit('0'))
        self.assertEquals("01", formatter.input_digit('1'))
        self.assertEquals("011", formatter.input_digit('1'))
        self.assertEquals("011-9", formatter.input_digit('9'))
        self.assertEquals("011-98", formatter.input_digit('8'))
        self.assertEquals("011-987", formatter.input_digit('7'))
        self.assertEquals("011-9876", formatter.input_digit('6'))
        self.assertEquals("011-9876-7", formatter.input_digit('7'))
        self.assertEquals("011-9876-78", formatter.input_digit('8'))
        self.assertEquals("011-9876-789", formatter.input_digit('9'))
        self.assertEquals("011-9876-7890", formatter.input_digit('0'))

    def testAYTFMultipleLeadingDigitPatterns(self):
        # +81 50 2345 6789
        formatter = AsYouTypeFormatter("JP")
        self.assertEquals("+", formatter.input_digit('+'))
        self.assertEquals("+8", formatter.input_digit('8'))
        self.assertEquals("+81 ", formatter.input_digit('1'))
        self.assertEquals("+81 5", formatter.input_digit('5'))
        self.assertEquals("+81 50", formatter.input_digit('0'))
        self.assertEquals("+81 50 2", formatter.input_digit('2'))
        self.assertEquals("+81 50 23", formatter.input_digit('3'))
        self.assertEquals("+81 50 234", formatter.input_digit('4'))
        self.assertEquals("+81 50 2345", formatter.input_digit('5'))
        self.assertEquals("+81 50 2345 6", formatter.input_digit('6'))
        self.assertEquals("+81 50 2345 67", formatter.input_digit('7'))
        self.assertEquals("+81 50 2345 678", formatter.input_digit('8'))
        self.assertEquals("+81 50 2345 6789", formatter.input_digit('9'))

        # +81 222 12 5678
        formatter.clear()
        self.assertEquals("+", formatter.input_digit('+'))
        self.assertEquals("+8", formatter.input_digit('8'))
        self.assertEquals("+81 ", formatter.input_digit('1'))
        self.assertEquals("+81 2", formatter.input_digit('2'))
        self.assertEquals("+81 22", formatter.input_digit('2'))
        self.assertEquals("+81 22 2", formatter.input_digit('2'))
        self.assertEquals("+81 22 21", formatter.input_digit('1'))
        self.assertEquals("+81 2221 2", formatter.input_digit('2'))
        self.assertEquals("+81 222 12 5", formatter.input_digit('5'))
        self.assertEquals("+81 222 12 56", formatter.input_digit('6'))
        self.assertEquals("+81 222 12 567", formatter.input_digit('7'))
        self.assertEquals("+81 222 12 5678", formatter.input_digit('8'))

        # +81 3332 2 5678
        formatter.clear()
        self.assertEquals("+", formatter.input_digit('+'))
        self.assertEquals("+8", formatter.input_digit('8'))
        self.assertEquals("+81 ", formatter.input_digit('1'))
        self.assertEquals("+81 3", formatter.input_digit('3'))
        self.assertEquals("+81 33", formatter.input_digit('3'))
        self.assertEquals("+81 33 3", formatter.input_digit('3'))
        self.assertEquals("+81 3332", formatter.input_digit('2'))
        self.assertEquals("+81 3332 2", formatter.input_digit('2'))
        self.assertEquals("+81 3332 2 5", formatter.input_digit('5'))
        self.assertEquals("+81 3332 2 56", formatter.input_digit('6'))
        self.assertEquals("+81 3332 2 567", formatter.input_digit('7'))
        self.assertEquals("+81 3332 2 5678", formatter.input_digit('8'))
