# ----------------------------------------------------------------------------
#       Copyright 2013 Huynh Vi Lam  <domovilam@gmail.com>
#
#       This file is part of pimucha.
#
#	This program is free software: you can redistribute it and/or modify
#	it under the terms of the GNU General Public License as published by
#	the Free Software Foundation, either version 3 of the License, or
#	(at your option) any later version.
#	
#	This program is distributed in the hope that it will be useful,
#	but WITHOUT ANY WARRANTY; without even the implied warranty of
#	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#	GNU General Public License for more details.
#	
#	You should have received a copy of the GNU General Public License
#	along with this program.  If not, see <http://www.gnu.org/licenses/>.
# ----------------------------------------------------------------------------



# ----------------------------------------------------------------------------
# Use to receive RF events
# from TellStick Duo interface
# Receiving data / USB FTDI Serial
# Require python-ftdi and libfti1 from www.intra2net.com
# with Python 2.7.x only
# ----------------------------------------------------------------------------
# Company Telldus Technologies
# ----------------------------
# - Website: http://www.telldus.com
# - Documentation and source code TellStick Duo is used as base for development of scripts
# ----------------------------------------------------------------------------


import logging,time
from .devcomm.devftdi import DEVftdi
from .ctrlfcts.tsduofct import (requestcmd)

logger = logging.getLogger()


class TSDUO(DEVftdi):

    def __init__(self,port=None):
        DEVftdi.__init__(self,port)
        self.idcontroller = 'TSDUO'
        self._id = '1781:0c31'
        self.port = port
        self.setup = False
        self.TXcap = True
        self.RXcap = True

    def tsetup(self):
        """
        Setup interface with request version test
        """
        self.open()
        if not self.opened:
            return False
        logger.debug("Starting %s with version request....",self.idcontroller)
        self.flush()
        data = self.request('VERS')
        self.flush()
        logger.warning("%s Ready....",self.idcontroller)
        self.setup = True
        self.device = self.idcontroller + '-ftdiport'

    def flush(self):
        logger.debug("Flush interface buffer by read %s...",self.idcontroller)
        n = 0
        while n < 5:
            n += 1
            res = self.read(100)
            if not res:
                time.sleep(0.1)
        return res

    def request(self,rcmd):
        return requestcmd(self,rcmd)

    def plsend(self):
        return None

    def rfsend(self,data,ack=None,msgq=None):
        if isinstance(data,list):
            databytes = data
        else:
            databytes = [ord(i) for i in data]
        self.flush()
        for byte in databytes:
            self.write(byte)
        logger.debug("Write to controller nb : %s",repr(len(data)))
        time.sleep(1)
        res = self.read(15)
        if not res:
            logger.debug("No response from controller")
            return None
        logger.debug("Result %s...",repr(res))
        if (len(res) > 0) and ack:
            if ack == res[0:2]:
                logger.debug("Success of command : %s",repr(ack))
                return ack
        return res

    def rxevent(self,maxlen=100):
        """
        Read one event after flush/pause
        Cut down data
        and eliminates same events
        """
        LEVT = "+W".encode('utf-8')
        REVT = ";\r\n".encode('utf-8')
        data = self.read(maxlen)
        if not data :
            return None
        events = []
        n = data.count(LEVT)
        if n == 0 :
            return None
        a = data.split(REVT)[0]            
        first = a[2:]
        if n == 1:
            events.append(first)
            return events
        else:
            next = self.read(10 * maxlen)
            if not next: return events
            n1 = next.count(LEVT)
            data += next
            n += n1
            evts = data.split(REVT)
            for a in evts[1:-1]:
                 e = a[2:]
                 if e not in events:
                      events.append(e)
            logger.warning("Cut down data into %s fragments with %s different events",str(n),str(len(events)))
            return events
