# ----------------------------------------------------------------------------
#       Copyright (C) 2013-2014 Huynh Vi Lam  <domovilam@gmail.com>
#
#       This file is part of pimucha.
#
#	This program is free software: you can redistribute it and/or modify
#	it under the terms of the GNU General Public License as published by
#	the Free Software Foundation, either version 3 of the License, or
#	(at your option) any later version.
#	
#	This program is distributed in the hope that it will be useful,
#	but WITHOUT ANY WARRANTY; without even the implied warranty of
#	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#	GNU General Public License for more details.
#	
#	You should have received a copy of the GNU General Public License
#	along with this program.  If not, see <http://www.gnu.org/licenses/>.
# ----------------------------------------------------------------------------

# ----------------------------------------------------------------------------
# Use for receive events incoming from controller
# Format the incoming event for sending to next target
# Use for transmit command to controller
# ----------------------------------------------------------------------------

import logging,sys,threading,time
if sys.hexversion > 0x030000F0:
    import queue as Queue
else:
    import Queue
QEmpty = Queue.Empty
from piHAcontrollers import Controllers
from .corelibs import (broadcast,unduplicate,multievtqueue,printer)
from .servers import (MESSAGEq,SetupControllers,TXControllers,evtnetqueue)


logger = logging.getLogger()


class CONTROLLER(object):
    def __init__(self,idcontroller,port=None):
        self.idcontroller = idcontroller
        self.name = self.idcontroller
        self.port = port
        self.stop = False
        self.opened = False
        self.setup = False
        self.type = 'SCTRL'
        self.TXcap = False
        self.featurenames = ['PRINTER','EVTNETQUEUE']
        self.features = [printer(),evtnetqueue()]
        self.decodernames = []
        self.decoders = []
        self.loccmdq = Queue.Queue()
        self.servercmds = {
            'ALLRXTX':self.allrxtx,
            'DROPDUPRX':self.dropduprx,
            'STOPS':self.stops}
        self.initcontroller()

    def initcontroller(self):
        if self.idcontroller in Controllers:
            self.controller = Controllers[self.idcontroller](self.port)
            if self.controller.TXcap : self.TXcap = True
        else:
            logger.critical("No controller associated with %s",idcontroller)
            self.controller = None

    def __del__(self):
        if self.setup:
            self.controller.close()

    def close(self):
        self.controller.close()

    def tsetup(self):
        if self.controller:
            self.controller.tsetup()
            self.opened = self.controller.opened
            self.setup = self.controller.setup
        if self.setup:
            self.namedev = self.name + ':' + self.idcontroller
            self.device = self.controller.device
            if self.namedev not in SetupControllers:
                SetupControllers.setdefault(self.namedev)
                SetupControllers[self.namedev] =  self
                logger.debug("Added in SetupControllers %s",repr(SetupControllers))
            if (self.idcontroller not in TXControllers) and (self.TXcap):
                TXControllers.setdefault(self.idcontroller)
                TXControllers[self.idcontroller] = self
                logger.debug("Added in TXControllers %s",repr(TXControllers))

    def cleanm(self):
        if not self.setup: return
        for d in self.decoders:
            d.close()
            self.decoders.remove(d)
        for d in self.decodernames:
            self.decodernames.remove(d)
        for f in self.features:
            f.close()
            self.features.remove(f)
        for f in self.featurenames:
            self.featurenames.remove(f)
        if self.TXcap : del TXControllers[self.idcontroller]          
        del SetupControllers[self.namedev]
        self.controller.close()

    def control(self):
        """
        Control the receive loop for incoming events
        """
        if self.opened == False:
            logger.debug("Device %s disconnected - Break Stop",self.name)
            self.stop = True
            self.setup = False
            self.close()

    def dispatch(self,target):
        if not self.setup: return
        while not self.stop:
            event = self.controller.rxevent()
            if event:
                target.send([self.namedev,event])
            # getting cmd from loccmdq for sending to local controller TX
            self.getloccmdq(MESSAGEq)
            # Control the receive loop
            self.control()
        self.cleanm()


    def getloccmdq(self,msgq=None):
        """
        Get from local TX queue
        Send cmd
        cmd = [idencoder,txcmdtosend]
        """
        try:
            cmd = self.loccmdq.get_nowait()
        except QEmpty:
            return
        if (self.controller.plsend) and (cmd[0] == 'PL'):
            self.controller.plsend(cmd[1],msgq)            
        elif (self.controller.rfsend) and (cmd[0] != 'PL'):
            self.controller.rfsend(cmd[1],msgq)
        else:
            logger.error("No controller detected for sending command %s", repr(cmd))

    def allRXTX(self):
        self.dispatch(
            target=broadcast(self.features)
            )

    def allrxtx(self):
        self.tt = threading.Thread(target=self.allRXTX)
        self.startt()

    def dropdupRX(self):
        self.dispatch(
            target=unduplicate(broadcast(self.features),ellapse=3)
            )

    def dropduprx(self):
        self.tt = threading.Thread(target=self.dropdupRX)
        self.startt()

    def startt(self):
        self.tt.name = 'CONTROLLEROf_' + self.namedev
        self.tt.start()
        self.ttname = self.tt.name
        print('%s starting...' % self.tt.name)

    def stops(self):
        self.stop = True
        print('%s stopping...' % self.ttname)
