# ----------------------------------------------------------------------------
#       Copyright (C) 2013-2014 Huynh Vi Lam  <domovilam@gmail.com>
#
#       This file is part of pimucha.
#
#	This program is free software: you can redistribute it and/or modify
#	it under the terms of the GNU General Public License as published by
#	the Free Software Foundation, either version 3 of the License, or
#	(at your option) any later version.
#	
#	This program is distributed in the hope that it will be useful,
#	but WITHOUT ANY WARRANTY; without even the implied warranty of
#	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#	GNU General Public License for more details.
#	
#	You should have received a copy of the GNU General Public License
#	along with this program.  If not, see <http://www.gnu.org/licenses/>.
# ----------------------------------------------------------------------------

import logging,sys,time

from .corelibs import (adstr2addr,chksetup,chkcmdname,serverthreads,infothreads)
from .coredecoders import (addparser,adddecoder)
from .controller import CONTROLLER
from .netcontroller import NETCONTROLLER
from .multictrx import MULTICTRX
from .rpccmds import RPCFunctions
from .rpclcmds import RPCLocalFunctions
from .servers import (PortInUse,Servers,SetupDaemons,SetupControllers)

logger = logging.getLogger()


def mctrx(param):
    """
    param = INIT|name	First step : Name to identify MCTRX
    param = PARSER|daemon	Second steps : add already setup RX controllers to MCTRX and parser
    daemon = namedaemon:controller
    param = DECODER|decoder	Third steps : add decoder with args to parser
    param = STARTM		Last step : start MCTRX controller
    """
    if param.startswith('INIT|'):
        name = param.split('|')[1]
        namedev = name + ':MCTRX'
        if 'MCTRX' in SetupDaemons:
            print ('A  MULTICTRX controller is already setup, only one MCTRX can be setup for process')
            return False
        else:
            #Init 
            try:
                dev = MULTICTRX()
                dev.name = namedev
                SetupDaemons.setdefault('MCTRX')
                SetupDaemons['MCTRX'] = dev
                return True
            except Exception as e:
                logger.critical("%s : %s", repr(e.__class__), str(e))
                print ('Error in init daemon for %s' % namedev)
                return False
    else:
        if 'MCTRX' in SetupDaemons:
            dev = SetupDaemons['MCTRX']
        else:
            logger.error('MULTICTRX controller is not yet setup')
            return False
    #Add an already setup RX controller
    if param.startswith('PARSER|'):
        namectrl = param.split('|')[1]
        if namectrl in dev.rxctrls:
            logger.error('Controller %s is already affected to MULTICTRX controller',namectrl)
            return False
        if namectrl in SetupControllers:
            dev.rxctrls.setdefault(namectrl)
            dev.rxctrls[namectrl] = SetupControllers[namectrl]
            logger.debug('Controller %s affected to MULTICTRX',repr(dev.rxctrls))
            rxname,rxctrl = namectrl.split(':')
            res = addparser(namectrl,dev)
            if res :
                logger.debug('Parser %s added to MULTICTRX',repr(dev.features))
                return True
            else:
                logger.error('Error in adding parser %s to MULTICTRX',namectrl)
                return False
        else:
            logger.error('An RX controller already setup is needed')
            return False
    #Add decoder to parser
    if param.startswith('DECODER|'):
        p = param.split('|')[1]
        namectrl,dargs = p.split('-')
        if namectrl == p:
            logger.error('Syntax error with decoder')
            return False
        if namectrl not in dev.rxctrls:
            logger.error('%s controller and parser not yet affected to MULTICTRX controller',namectrl)
            return False
        res = adddecoder(namectrl + '|' + dargs,dev)
        if res:
            logger.debug('Decoder %s added to MULTICTRX',repr(dargs))
            return True
        else:
            logger.error('Error in adding decoder %s to MULTICTRX',repr(dargs))
            return False
    #Start MULTICTRX
    if param == 'STARTM':
        dev.tsetup()
        if dev.setup:
            dev.startm()
            print ('Daemon of MULTICTRX correctly started : %s' % repr(dev))
            return True
        else:
            print ('Error in setup Daemon of MULTICTRX %s' % param)
            return False
    #Stop MULTICTRX
    if param == 'STOPS':
        if dev.setup:
            dev.stops()
            print ('Daemon of MULTICTRX correctly stopped')
            return True
        else:
            print ('Daemon of MULTICTRX is not yet setup and started %s' % repr(dev))
            return False
    else:
        print ('Syntax error %s' % param)
        return False


def setupctrl(param):
    """
    param = name|daemonthread
    daemonthread = controller-port
             port = needed for serial
                    needed for USB if many controllers of same type
    name  = identify daemonthread
    """
    p = chksetup(param,SetupDaemons)	#p = name,idctrl,port
    if not p:
        return False
    name,idctrl,port,rxtx = p
    if port == 'Defaults' : 
        port = None
    if port in PortInUse:
        if idctrl in PortInUse[port]:
            logger.error('The same port %s is already setup with %s ',port,PortInUse[port])
            return False
    namedev = name + ':' + idctrl
    if namedev in SetupControllers:
        logger.error('A controller is already setup with %s',namedev)
        return False
    #Init class object
    try:
        dev = CONTROLLER(idctrl,port)
        dev.name = name
    except Exception as e:
        logger.critical("%s : %s", repr(e.__class__), str(e))
        logger.error('Error in init class object for %s',param)
        return False
    #Setup daemon
    try:
        dev.tsetup()
    except Exception as e:
        logger.critical("%s : %s", repr(e.__class__), str(e))
        logger.error('Error in setup daemon %s',param)
        return False
    if dev.setup:
        logger.warning('Daemon of Controller correctly setup : %s',repr(dev))
        return True
    else:
        logger.error('Error in setup Daemon of Controller %s',param)
        return False


def setupnetctrl(param):
    """
    param = name|daemonthread
    daemonthread = netcontroller-addr-RX
            addr = host:port
              RX = RXcap
              TX = TXcap
    name  = identify daemonthread
    """
    p = chksetup(param,SetupDaemons)	#p = name,idctrl,port
    if not p:
        return False
    name,idctrl,port, rxtx = p
    if (rxtx is None) or (port == 'Defaults'):
        logger.error('Syntax error with the parameter')
        return False
    addr = adstr2addr(port)
    if addr in PortInUse:
        logger.error('The same address:port %s is already setup with %s ',port,PortInUse[addr])
        return False
    namedev = name + ':' + idctrl
    if namedev in SetupControllers:
        logger.error('A netcontroller is already setup with %s',namedev)
        return False
    if rxtx == 'RX':
        rx = True
    else:
        rx = False
    #Init class object
    try:
        dev = NETCONTROLLER(idctrl,addr=addr,RX=rx)
        dev.name = name
    except Exception as e:
        logger.critical("%s : %s", repr(e.__class__), str(e))
        logger.error('Error in init class object for %s',param)
        return False
    #Setup daemon
    try:
        dev.tsetup()
    except Exception as e:
        logger.critical("%s : %s", repr(e.__class__), str(e))
        logger.error('Error in setup daemon %s',param)
        return False
    if dev.setup:
        logger.warning('Daemon of NETController correctly setup : %s',repr(dev))
        return True
    else:
        logger.error('Error in setup Daemon of NETController %s',param)
        return False


def setupsrv(param):
    """
    param = name|daemonthread
    daemonthread = server-port
            port = address:port
    name  = identify daemonthread
    """
    p = chksetup(param,SetupDaemons)
    if not p:
        return False
    name,server,port,rxtx = p
    if port != 'Defaults' : 
        if port in PortInUse:
            logger.error('The same address:port %s is already setup with %s ',port,PortInUse[port])
            return False
    if server not in ['RPCS','TCPRX','TCPS']:
        logger.error('Syntax error for the type of server to setup')
        return False
    namesrv = name + ':' + server
    if port == 'Defaults':
        addr = None
    else:
        addr = adstr2addr(port)
    #Init class object
    try:
        srv = Servers[server](addr)
        srv.name = namesrv
    except Exception as e:
        logger.critical("%s : %s", repr(e.__class__), str(e))
        logger.error('Error in init class object for %s',param)
        return False
    if server == 'RPCS':
        srv.Functions += RPCFunctions
        if port == 'Defaults': srv.Functions += RPCLocalFunctions
    #Setup daemon
    try:
        srv.tsetup()
    except Exception as e:
        logger.critical("%s : %s", repr(e.__class__), str(e))
        logger.error('Error in setup daemon %s',param)
        return False
    if srv.setup:
        PortInUse.setdefault(port)
        PortInUse[port] = namesrv
        SetupDaemons.setdefault(namesrv)
        SetupDaemons[namesrv] = srv
        logger.warning('Daemon of server correctly setup : %s',repr(srv))
        return True
    else:
        logger.error('Error in setup Daemon of server %s',param)
        return False


def startsrv(param):
    p = chkcmdname(param,SetupDaemons)	#p = [cmd,name,dev]
    if not p:
        return False
    cmd, name, dev = p
    tt = serverthreads(name)
    if tt:
        logger.warning('An instance %s of %s is already active' , tt.name,name)
        return False
    else:
        try:
            cmd()
        except Exception as e:
            logger.critical("%s : %s", repr(e.__class__), str(e))
            logger.error('Unable to execute %s',repr(cmd))
            return False
    logger.info('Command started for %s',param)
    return True


def stopsrv(namectrl):
    if namectrl == 'SYSLOGF':
        SetupDaemons['INParser'].stops()
        SetupDaemons['SYSLOGF'].stops()
        del SetupDaemons['SYSLOGF']
        del SetupDaemons['INParser']
        return True
    tt = serverthreads(namectrl)
    if tt:
        SetupDaemons[namectrl].stops()
        tt.join(5)
        if not tt.isAlive():
            logger.info('The thread instance %s is stopped.',tt.name)
            del SetupDaemons[namectrl]
            if namectrl.split(':')[1] in ['RPCS','TCPRX','TCPS']: del PortInUse[namectrl]
        return True
    else:
        logger.info('No thread instance for %s', namectrl)
        return False


def stopallctrls():
    SC = SetupControllers.keys()
    for ctrl in SC:
        stopsrv(SetupControllers[ctrl].namedev)


def stopall_rpc(params):
    stopallctrls()
    stopsrv('SYSLOGF')
    return 'STOPALL daemons and threads'
