#! /usr/bin/env pythoN
import argparse
import sys

from .server import run_server
from .certs import gen_certs
from .util import user_mod, do_init

__version__ = '0.1.0'


_conf_file = 'pipa.cfg'


def main(args=None):
    root = argparse.ArgumentParser(
        'pipa',
        description='The simple PyPI server'
        )
    commands = root.add_subparsers()

    # pipa init
    init = commands.add_parser('init', description='Initialize pipa')
    init.add_argument(
        '-c', '--config',
        dest='conf_file', default=_conf_file, nargs=1,
        metavar='CONFIG', help='(default: pipa.cfg)'
        )
    init.add_argument(
        '-s', '--salt',
        dest='salt', default=None, nargs=1,
        help='Randomly generated by default'
        )
    init.add_argument(
        '-p', '--packages',
        dest='packages', nargs=1,
        help="Folder to store pipa's packages"
        )
    init.add_argument(
        '-n', '--no-certs',
        dest='no_certs', action='store_true',
        help='Use flag to *not* (re)generate certs'
        )
    init.set_defaults(func=do_init)

    # pipa serve
    serve = commands.add_parser('serve', description='Run the pipa server')
    serve.set_defaults(func=run_server)
    serve.add_argument(
        '-c', '--config',
        dest='conf_file', default=_conf_file
        )
    serve.add_argument(
        '-p', '--port',
        dest='port', default=None, type=int,
        help='(default: 5351)'
        )
    serve.add_argument(
        '--host',
        dest='host', default=None,
        help='(default: localhost)'
        )
    serve.add_argument(
        '-l', '--cert',
        dest='cert', default=None,
        metavar='CERT_FILE'
        )
    serve.add_argument(
        '-k', '--key',
        dest='key', default=None,
        metavar='KEY_FILE'
        )
    serve.add_argument(
        '--no-ssl',
        dest='no_ssl', action='store_true'
        )

    # pipa gen-certs
    cert = commands.add_parser(
        'gen-certs',
        description='Generate CA & server certs'
        )
    cert.set_defaults(func=gen_certs)
    cert.add_argument(
        'host',
        default='localhost', nargs='?',
        help='Hostname of the server (default: localhost)'
        )
    cert.add_argument(
        '--dir',
        dest='dir', default='',
        help='Directory to save certificates'
        )
    cert.add_argument(
        '-s', '--save-all',
        dest='save_all', action='store_true',
        help='Store all associated certs & keys'
        )
    cert.add_argument(
        '-d', '--days',
        dest='days', default=365, type=int,
        help='How many days certs are valid (default: 365)')

    # pipa user
    user = commands.add_parser(
        'user',
        description='Configure users with access to upload to pipa'
        )
    user.set_defaults(func=user_mod)
    user.add_argument(
        '-c', '--config',
        dest='conf_file', default=_conf_file,
        metavar='CONFIG', nargs=1,
        help='(default: pipa.cfg)'
        )
    group = user.add_mutually_exclusive_group(required=True)
    group.add_argument(
        '-l', '--list',
        dest='list', default=False,
        action='store_true',
        help='List current users'
        )
    group.add_argument(
        '-a', '--add',
        dest='add', default=False,
        metavar=('USER', 'PASS'), nargs=2,
        help='add a user (-a username password)'
        )
    group.add_argument(
        '-d', '--delete',
        dest='delete', default=False,
        metavar='USER', nargs=1,
        help='Delete a user (-d username)'
        )

    if args is None:
        args = sys.argv[1:]
    parsed = vars(root.parse_args(args))
    func = parsed.pop('func', root.print_help)
    func(**parsed)


if __name__ == '__main__':
    main()
