import os
import tarfile
import logging
import regex
import sqlite3
from datetime import datetime, timedelta
from parser import parse_lines, LogParser
from korg.korg import PatternRepo

here = lambda x: os.path.abspath(os.path.join(os.path.dirname(__file__), x))

class Singleton(type):
    _instances = {}
    def __call__(cls, *args, **kwargs):
        if cls not in cls._instances:
            cls._instances[cls] = super(Singleton, cls).__call__(*args, **kwargs)
        return cls._instances[cls]


class Datastore(object):
    __metaclass__ = Singleton
    testruns = {}

    def __init__(self, maxsize_mb=1000):
        self.maxsize_mb = maxsize_mb
        self.pr = PatternRepo([here('../patterns')])

    def parse_oscounters(self, infile, db, tablename):
        def timestamp(logstart, interval):
            # helper to create a generator for adding timestamps to
            # parsed loglines
            # workaround to implement missing closure
            nonlocal = {
                'logstart': logstart,
                'interval': int(interval)
                }

            def gen(**kw_args):
                nonlocal['logstart'] += timedelta(
                    seconds=nonlocal['interval'])
                return nonlocal['logstart'].strftime('%s%f')[:13]
            return gen

        # read logstart and interval from file
        match = self.pr.compile_regex('%{COUNTER_START}').match(
            infile.readline()).groupdict()

        callbacks = {'timestamp': timestamp(
            datetime.strptime(match['logstart'], '%Y%m%d %H%M%S'),
            match['interval'])
        }

        # parse the data from the logfiles
        vmstat = LogParser(self.pr.compile_regex('%{LINUX_VMSTAT}'), callbacks, db, tablename)
        discard = LogParser(self.pr.compile_regex('%{LINUX_VMSTAT_DISCARD}'))
        parse_lines([vmstat, discard], infile)


    def populate(self, testrun):
        self.testruns[testrun] = sqlite3.connect(':memory:',
        detect_types=sqlite3.PARSE_COLNAMES)

        # TODO: read in all files in folder and parse according to type
        # suffix = os.path.basename(sys.argv[2]).split('.', 1)[0].split('-', 2)[2]
        with tarfile.open('../testruns/%s/vm-ubuntu-oscounters-%s.tgz' % (testrun, testrun),
                'r:gz') as tar:
            infile = tar.extractfile('./vmstat.txt.%s' % testrun)
            self.parse_oscounters(infile, self.testruns[testrun], 'vmubuntu_oscounters') # no - allowed here


    def list_hosts(self, testrun):
        if testrun not in self.testruns:
            self.populate(testrun)
        try:
            c = self.testruns[testrun].cursor()
            c.execute('select name from sqlite_master where type = "table"')

            rows = c.fetchall()
            c.close()
            return {'hosts': [h[0].rpartition('_')[0] for h in rows]}
        except sqlite3.OperationalError as e:
            raise KeyError(e)


    def list_metrics(self, testrun, host):
        if testrun not in self.testruns:
            self.populate(testrun)
        try:
            part = self._assemble_tablename(host, '')
            c = self.testruns[testrun].cursor()
            c.execute('select name from sqlite_master where type = "table"')

            rows = c.fetchall()
            c.close()
            return {'metrics': [h[0][len(part):] for h in rows if h[0].startswith(part)]}
        except sqlite3.OperationalError as e:
            raise KeyError(e)


    def read(self, testrun, host, metric, arguments=[]):
        if testrun not in self.testruns:
            self.populate(testrun)
        tablename = self._assemble_tablename(host, metric)
        try:
            c = self.testruns[testrun].cursor()
            c.execute('select * from %s order by timestamp' % tablename)

            rows = c.fetchall()
            headers = tuple([h[0] for h in c.description])
            c.close()
            # return {'headers': headers, 'rows': rows}
            # now we use csv
            return [headers] + rows
        except sqlite3.OperationalError as e:
            raise KeyError(e)

    def _assemble_tablename(self, host, metric):
        return regex.sub('[^\w_]+', '', '%s_%s' % (host, metric))
