import re

from pkg.fig.project import Project
from pkg.fig.service import VALID_NAME_CHARS, ConfigError, NAME_RE
from pkg.fig.cli.docker_client import docker_client
from service import Service

CONTAINER_NAME_REGEX = '\s+\_\d+\_\s+\_\d+'
DEFAULT_IMAGE = '10.66.128.203:49153/coreqa/splunk:fig'


# import os
# os.environ["FIG_FILE"] = "/Users/weiwang/work/github/rider/rider/pkg/fig.yml"
# os.environ["DOCKER_HOST"] = 'tcp://192.168.59.103:2376'
# os.environ["DOCKER_TLS_VERIFY"] = '1'
# os.environ["DOCKER_CERT_PATH"] = '/Users/weiwang/.boot2docker/certs/boot2docker-vm'

class Environment(object):
    """
    Represents a rider environment of splunk instances
    Currently support: cluster, dst(distribute search), shc (search head clustering)

    """
    client = docker_client()

    def __init__(self, env_name, image_name):
        self.env_name = env_name
        self._image_name = image_name
        self._service_dict = self._get_service_dict(image_name)
        self._project = Project.from_dicts(self.env_name, self._service_dict, self.client)
        for service in self._project.get_services():
            setattr(self, service.name, Service(service.name, service, self))

    @property
    def service_list(self):
        """
        return a service list of current environment

        :return: service list of current environment

        >>> from rider.environment import Environment
        >>> myEnv = Environment.create_environment("cluster","jacktestapi")
        >>> server_list = myEnv.service_list
        """
        return [Service(service.name, service, self) for service in self._project.get_services()]

    def get_scaleable_services_list(self):
        """
        Return all scale-able services as a list of current environment

        :return: all scale-able services as a list of current environment

        >>> from rider.environment import Environment
        >>> myEnv = Environment.create_environment("cluster","jacktestapi")
        >>> server_list = myEnv.get_scaleable_services_list()
        """
        return [service for service in self.service_list if service.is_scaleable() == True]

        pass

    @classmethod
    def create_environment(cls, env_type, env_name, image_name=None):
        """
        Create an environment object

        :param env_type: string cluster|dst|shc
        :param env_name: string identifier of an environment, the environment name will be [env_name]_[env_type]
        :param image_name: string   if not provided will use default which is '10.66.128.203:49153/coreqa/splunk:fig'

        :return:    environment object


        >>> from rider.environment import Environment
        >>> myEnv = Environment.create_environment("cluster", "jacktestapi")
        """
        if image_name is None:
            image_name = DEFAULT_IMAGE

        if not re.match('^%s+$' % VALID_NAME_CHARS, env_name):
            raise ConfigError('Invalid service name "%s" - only %s are allowed' % (env_name, VALID_NAME_CHARS))

        Env_class = ENV_CLASS_MAPPING[env_type]
        new_env_name = "%s_%s" % (env_name, env_type)
        return Env_class(new_env_name, image_name)


    @classmethod
    def get_environments(cls):
        """
        Return all environments object

        :return:    list of environment objects


        >>> from rider.environment import Environment
        >>> env_list = Environment.get_environments()
        """

        all_environment = []
        all_environment_name_image = cls.get_all_environments_name_image_dict()
        if len(all_environment_name_image.keys()) != 0:
            for env_identifier in all_environment_name_image.keys():
                env_name, env_type = env_identifier.split('_')
                Env_class = ENV_CLASS_MAPPING[env_type]
                env_instance = Env_class(env_identifier, all_environment_name_image[env_identifier])
                all_environment.append(env_instance)

        return all_environment


    @classmethod
    def get_environment_by_name(cls, name):
        """
        Return the environment object specified by name ([env_name]_[env_type])
        :param name: string name = env_name+"_"+env_type

        :return:    environment object

        >>> from rider.environment import Environment
        >>> env = Environment.get_environment_by_name('jacktestapi_cluster')
        """

        for env in cls.get_environments():
            if env.env_name == name:
                return env


    @classmethod
    def get_all_environments_name_image_dict(cls, stopped=False):

        all_environments_name_image = {}
        for container in cls.client.containers(all=stopped):
            match = NAME_RE.match(container['Names'][0])
            if match:
                env_identifier = match.group(1).replace("/", "") + "_" + match.group(2)
                if not env_identifier in all_environments_name_image.keys():
                    all_environments_name_image[env_identifier] = container['Image']

        return all_environments_name_image


    def get_service_names(self):
        """
        Get list of service name belong to environment
        :return: list of service name

        >>> from rider.environment import  Environment
        >>> env = Environment.get_environment_by_name('jacktestapi_cluster')
        >>> services_name_list = env.get_service_names()
        """
        service_name = []
        for service in self.service_list:
            if service.name not in service_name:
                service_name.append(service.name)

        return service_name


    def clean(self):
        """
        Clean the entire environment by stop and remove the running containers in current environment
        :return:

        >>> from rider.environment import  Environment
        >>> env = Environment.get_environment_by_name('jacktestapi_cluster')
        >>> env.bootstrap()
        >>> env.clean()
        """
        for service in self.service_list:
            service._service.stop()
            service._service.remove_stopped(**{"v": True})


    def up(self):
        """
        start environment by start only one container for each services
        :return:

        >>> from rider.environment import  Environment
        >>> env = Environment.get_environment_by_name('jacktestapi_cluster')
        >>> env.up()
        """

        self._project.up()
        pass


    def bootstrap(self):
        """
        Bootstrap a standard environment

        :return:

        >>> from rider.environment import  Environment
        >>> env = Environment.get_environment_by_name('jacktestapi_cluster')
        >>> env.bootstrap()

        """

        if self.env_type == "cluster":
            self.up()
            self.indexer.scale(3)

        elif self.env_type == "dst":
            self.up()
            self.dsp.scale(3)

        pass


    def _get_service_dict(self, image_name):
        pre_service_dict = ENV_DICT_MAPPING[self.env_type]
        aft_service_dict = []
        for service in pre_service_dict:
            service['image'] = image_name
            aft_service_dict.append(service)
        return aft_service_dict


class ClusterEnvironment(Environment):
    env_type = "cluster"

    def __init__(self, env_name, image_name):
        super(ClusterEnvironment, self).__init__(env_name, image_name)


class DistributedEnvironment(Environment):
    env_type = "dst"

    def __init__(self, env_name, image_name):
        super(DistributedEnvironment, self).__init__(env_name, image_name)


# class ShcEnvironment(Environment):
# env_type = "shc"
#
# def __init__(self, env_name):
#         super(ShcEnvironment, self).__init__(env_name)


# CONSTANTS VARIABLES
ENV_CLASS_MAPPING = {
    ClusterEnvironment.env_type: ClusterEnvironment,
    DistributedEnvironment.env_type: DistributedEnvironment,
    # ShcEnvironment.env_type: ShcEnvironment
}

ENV_DICT_MAPPING = {
    ClusterEnvironment.env_type: [
        {'image': '10.66.128.203:49153/coreqa/splunk:fig', 'command': 'sh', 'ports': ['8000', '8089', '22'],
         'links': ['master'], u'name': 'sh'},
        {'image': '10.66.128.203:49153/coreqa/splunk:fig', 'command': 'master', 'ports': ['8000', '8089', '22'],
         u'name': 'master'},
        {'image': '10.66.128.203:49153/coreqa/splunk:fig', 'command': 'indexer', 'ports': ['8000', '8089', '22'],
         'links': ['master'], u'name': 'indexer'}],
    DistributedEnvironment.env_type: [
        {'image': '10.66.128.203:49153/coreqa/splunk:fig', 'command': 'dsh', 'ports': ['8000', '8089', '22'],
         'links': ['dsp'], u'name': 'dsh'},
        {'image': '10.66.128.203:49153/coreqa/splunk:fig', 'command': 'dsp', 'ports': ['8000', '8089', '22'],
         u'name': 'dsp'}],
    # ShcEnvironment.env_type: [
    # {'image': '10.66.128.203:49153/coreqa/splunk:fig', 'command': 'sh', 'ports': ['8000', '8089', '22'],
    #      'links': ['master'], u'name': 'sh'},
    #     {'image': '10.66.128.203:49153/coreqa/splunk:fig', 'command': 'master', 'ports': ['8000', '8089', '22'],
    #      u'name': 'master'},
    #     {'image': '10.66.128.203:49153/coreqa/splunk:fig', 'command': 'indexer', 'ports': ['8000', '8089', '22'],
    #      'links': ['master'], u'name': 'indexer'}]
}

# if __name__ == "__main__":
#
#
# # testenv = Environment.get_environment_by_name('jacktestapi2')
# # testenv = Environment.get_environments()
# # print testenv[0].indexer.machines
#
#     myEnv = Environment.create_environment("dst", "dsttest")
#     myEnv.bootstrap()
#
# # myEnv.get_service_names()
# #
# # myEnv.up()
# # myEnv.indexer.scale(2)
# #
# # print myEnv.indexer.machines[0].name
# # Environment.get_environments()
# # env = Environment.get_environment_by_name('jacktestapi_cluster')
# # print env.service_list
# # env.clean()
#     print 1
#
