import time

from machine import Machine


class Service(object):
    def __init__(self, name, _service, _environment):
        self.name = name
        self._environment = _environment
        self._service = _service


    def scale(self, desire_number):
        """
        scale a service to desire_number

        :param desire_number: int
        :return:

        >>> from rider.environment import Environment
        >>> env = Environment.get_environments('jacktestapi2_cluster')
        >>> env.indexer.scale(5)
        """

        if self._can_be_scaled():

            before_scale = self.machines
            before_scale_machine_name = [machine.name for machine in self.machines]
            self._service.scale(desire_number)
            self.ssh_check()
            after_scale = self.machines
            need_to_deploy = [item for item in after_scale if item.name not in before_scale_machine_name]

            if self.name == "dsp":
                self.deploy_dsp(need_to_deploy)


    def deploy_dsp(self, need_to_deploy_machines):
        service_dsh = self.get_links_for_dsp()
        machine_dsh = service_dsh.machines[0]

        for machine in need_to_deploy_machines:
            if machine.islinked == False:
                ip = machine.container.get("NetworkSettings.IPAddress")
                cmd = '/tmp/splunk/bin/splunk add search-server -host https://%s:8089 -auth admin:notchangeme -remoteUsername admin -remotePassword notchangeme' % ip
                machine_dsh.execute_command(cmd)
                machine.islinked = True
        pass


    def get_links_for_dsp(self):

        if self.is_scaleable() and self.name == "dsp":
            for service in self._environment.service_list:
                if service.name == "dsh":
                    return service

        pass


    def get_env(self):
        return self._service.project


    def _can_be_scaled(self):

        if self._service.name == 'master' or self._service.name == 'multimaster':
            raise CouldNotScale(self._service.name)
        else:
            return True

    def is_scaleable(self):
        if self._service.name == 'master' or self._service.name == 'multimaster':
            return False
        else:
            return True


    @property
    def machines(self):
        """
        Return machine objects belong to this service

        :return: list of machine objects

        >>> from rider.environment import Environment
        >>> env = Environment.get_environments('jacktestapi2_cluster')
        >>> env.indexer.machines
        >>> env.indexer.machines[0]
        """
        machines = []
        for _container in self._service.containers():
            machines.append(Machine(_container, self))
        return machines

    def ssh_check(self):
        retry = 0
        while retry < 10:
            for machine in self.machines:
                if not machine.is_ssh_accessible:
                    break
            else:
                return True
            time.sleep(10)
            retry += 1
        else:
            # FIXME catagory the exception
            raise Exception("The ssh service is not accessible")


class CouldNotScale(Exception):
    def __init__(self, name):
        self.name = name
        self.msg = "%s is not able to scale" % self.name

    def __str__(self):
        return self.msg
