import sys
import optparse
from optparse import Option, BadOptionError

from rider.cli import cmdoptions
from rider.cli.cmdparser import ConfigOptionParser, UpdatingDefaultsHelpFormatter
from rider.cli.utils import get_prog
from rider.log import Logger
from rider.environment import Environment


class Command(object):
    name = None
    usage = None
    hidden = None
    summary = ""

    def __init__(self):
        parser_kw = {
            'usage': self.usage,
            'prog': '%s %s' % (get_prog(), self.name),
            'formatter': UpdatingDefaultsHelpFormatter(),
            'add_help_option': False,
            'name': self.name,
            'description': self.__doc__,
        }

        self.parser = ConfigOptionParser(**parser_kw)

        # Commands should add options to this option group
        optgroup_name = '%s Options' % self.name.capitalize()
        self.cmd_opts = optparse.OptionGroup(self.parser, optgroup_name)

        # Add the general options
        gen_opts = cmdoptions.make_option_group(cmdoptions.general_group, self.parser)
        self.parser.add_option_group(gen_opts)

        # set logger
        self.logger = Logger()


    def setup_logging(self):
        pass

    def parse_args(self, args):
        # factored out for testability
        return self.parser.parse_args(args)

    def run(self, args):
        """
            The sub command class should overide this method
        """
        NotImplemented

    def execute(self, args=None):
        """
            The main interface for exectute the command
        """
        try:
            self.run(args)
        except Exception:
            self.logger.error("ERROR: %s" % str(sys.exc_info()[1]))
            sys.exit(1)
        except KeyboardInterrupt:
            sys.exit(1)


class ScaleCommand(Command):
    name = "scale"
    usage = """%prog """
    summary = "scale out the indexer and search head according to the current envrionment"

    def __init__(self):
        super(ScaleCommand, self).__init__()

        self.parser.add_option(Option(
            '--env_name',
            dest='env_name',
            acticon='store',
            default=None,
            help="the name of environment"
        ))

        self.parser.add_option(Option(
            '--indexer-num',
            dest='indexer_num',
            action='store',
            default=None,
            help="the cluster indexer number"))

        self.parser.add_option(Option(
            '--sh-num',
            dest='sh_num',
            action='store',
            default=None,
            help="the cluster sh number"
        ))


    def run(self, args):
        try:
            options, arg_else = self.parse_args(args)
        except BadOptionError:
            self.logger.error("ERROR: %s" % str(sys.exc_info()[1]))
            return

        # scale services
        if not options.env_name:
            self.logger.error("ERROR: %s" % "please provide environment name")
        else:
            riderEnv = Environment.get_environment_by_name(options.env_name)
            riderEnv.indexer.scale(int(options.indexer_num))


class CleanCommand(Command):
    name = "clean"
    usage = """%prog """
    summary = "build the new image"

    def __init__(self):
        super(Command, self).__init__()


class BuildCommand(Command):
    name = "build"
    usage = """%prog """
    summary = "build the new image"

    def __init__(self):
        super(Command, self).__init__()
        self.parser.add_option(Option(
            '--splunk-pkg',
            dest='splunk_pkg',
            action='store',
            default='splunk_tgz',
            help="Splunk release build package"
        ))

        self.parser.add_option(Option(
            '--image-name',
            dest='image_name',
            action='store',
            default='10.66.128.203:49153/coreqa/splunk:latest',
            help="Image Name"
        ))

    def run(self, args):
        pass


COMMANDS = {
    BuildCommand.name: BuildCommand,
    CleanCommand.name: CleanCommand,
    ScaleCommand.name: ScaleCommand,
}


def get_summaries(ignore_hidden=True, ordered=True):
    """Yields sorted (command name, command summary) tuples."""
    cmd_items = COMMANDS.items()
    for name, command_class in cmd_items:
        if ignore_hidden and command_class.hidden:
            continue
        yield (name, command_class.summary)

