import tempfile
import shutil
import sys

import os
import re
from rider.pkg.fig.cli.docker_client import docker_client
from rider.pkg.fig.progress_stream import stream_output, StreamOutputError


class TagValueError(Exception):
    pass


class BuildError(Exception):
    def __init__(self, msg):
        self.msg


REPO_NAME = "rider_splunk"  # the image name is fixed
DOCKERFILE_FOLDER = "dockerbuild"
SPLUNK_PKG_NAME = "splunk.tgz"
docker_connection = docker_client()


def image_name_to_tuple(repo_tag):
    registry, ns, repo, tag = None, None, None, None
    prefix_name_tuple = repo_tag.split("/")
    if len(prefix_name_tuple) == 2:
        ns = prefix_name_tuple[0]
        repo, tag = prefix_name_tuple[1].split(":")
    elif len(prefix_name_tuple) == 3:
        registry, ns = prefix_name_tuple[0], prefix_name_tuple[1]
        repo, tag = prefix_name_tuple[2].split(":")
    return registry, ns, repo, tag


def dict_to_image_name(**kwargs):
    if kwargs["registry"]:
        return "{registry}/{ns}/{repo_name}:{tag}".format(registry=kwargs["registry"], ns=kwargs["ns"],
                                                          repo_name=REPO_NAME,
                                                          tag=kwargs["tag"])
    else:
        return "{ns}/{repo_name}:{tag}".format(ns=kwargs["ns"],
                                               repo_name=REPO_NAME,
                                               tag=kwargs["tag"])


class Image(object):
    """
        Image class is the abstraction for the Docker::Image
    """

    def __init__(self, tag, registry=None, ns="coreqa", docker_image=None):
        self.tag = tag
        self.registry = registry
        self.ns = ns
        self.docker_image = docker_image
        self.created = True if docker_image else None

    @property
    def image_id(self):
        return self.docker_image["Id"]

    @property
    def image_name(self):
        return dict_to_image_name(registry=self.registry, ns=self.ns, tag=self.tag)

    @classmethod
    def get_all_images(cls):
        all_docker_images = docker_connection.images()
        image_dict = {}
        for docker_image in all_docker_images:
            convert_result = cls.docker_image_convert(docker_image)
            image_dict[convert_result[1]] = convert_result[0]
        return image_dict

    @classmethod
    def docker_image_convert(cls, docker_image):
        registry, ns, repo, tag = image_name_to_tuple(docker_image["RepoTags"][0])
        return Image(tag=tag, registry=registry, ns=ns, docker_image=docker_image), tag

    @classmethod
    def get_image_by_image_id(cls, image_id):
        for tag, image in cls.get_all_images().items():
            if image.image_id.startswith(image_id):
                return image
        return None

    @classmethod
    def build_image(cls, registry, ns, splunk_build_id, splunk_pkg):
        temp_build_path = tempfile.mkdtemp()
        image_id = None
        try:
            temp_dockerbuild_path = os.path.join(temp_build_path, DOCKERFILE_FOLDER)
            docker_build_path = os.path.abspath(os.path.join(os.path.dirname(__file__), DOCKERFILE_FOLDER))

            # prepare the build environment
            shutil.copytree(os.path.join(docker_build_path), os.path.join(temp_build_path, DOCKERFILE_FOLDER))
            shutil.copyfile(splunk_pkg, os.path.join(temp_build_path, DOCKERFILE_FOLDER, SPLUNK_PKG_NAME))

            build_output = docker_connection.build(
                path=temp_dockerbuild_path,
                tag=dict_to_image_name(registry=registry, ns=ns, tag=splunk_build_id),
                stream=True,
                rm=True,
                nocache=True,
            )

            try:
                all_events = stream_output(build_output, sys.stdout)
            except StreamOutputError, e:
                raise BuildError(unicode(e))

            for event in all_events:
                if 'stream' in event:
                    match = re.search(r'Successfully built ([0-9a-f]+)', event.get('stream', ''))
                    if match:
                        image_id = match.group(1)

            if image_id is None:
                raise BuildError("image build fail")

            return cls.get_image_by_image_id(image_id)
        finally:
            if os.path.exists(temp_build_path):
                shutil.rmtree(temp_build_path)


if __name__ == "__main__":
    image = Image.build_image(registry="10.66.128.128:49154", ns="markshao", splunk_build_id="12341234",
                              splunk_pkg="/Users/cesc/Work/Releases/splunk-6.2-235774-darwin-64.tgz")
    print image
