import sys
import optparse
from optparse import Option, BadOptionError

from rider.cli import cmdoptions
from rider.cli.cmdparser import ConfigOptionParser, UpdatingDefaultsHelpFormatter, parse_opts
from rider.cli.utils import get_prog, get_userinput, is_true
from rider.log import Logger
from rider.environment import Environment


class Command(object):
    name = None
    usage = None
    hidden = None
    summary = ""

    def __init__(self):
        self.parser_kw = {
            'usage': self.usage,
            'prog': '%s %s' % (get_prog(), self.name),
            'formatter': UpdatingDefaultsHelpFormatter(),
            'add_help_option': False,
            'name': self.name,
            'description': self.__doc__,
        }

        self.parser = ConfigOptionParser(**self.parser_kw)

        # Commands should add options to this option group
        optgroup_name = '%s Options' % self.name.capitalize()
        self.cmd_opts = optparse.OptionGroup(self.parser, optgroup_name)

        # Add the general options
        gen_opts = cmdoptions.make_option_group(cmdoptions.general_group, self.parser)
        self.parser.add_option_group(gen_opts)

        # set logger
        self.logger = Logger()

    def parse_args(self, args):
        # factored out for testability
        return self.parser.parse_args(args)

    def run(self, args):
        """
            The sub command class should overide this method
        """
        NotImplemented

    def execute(self, args=None):
        """
            The main interface for exectute the command
        """
        try:
            self.run(args)
        except Exception:
            self.logger.error("ERROR: %s" % str(sys.exc_info()[1]))
            sys.exit(1)
        except KeyboardInterrupt:
            sys.exit(1)


class ScaleCommand(Command):
    name = "scale"
    usage = """%prog """
    summary = "scale the service of the environment"

    def __init__(self):
        super(ScaleCommand, self).__init__()

        self.parser.add_option(Option(
            '--envname',
            dest='envname',
            action='store',
            default=None,
            help="the name of environment"
        ))

        self.parser.add_option(Option(
            '--sh-num',
            dest='sh_num',
            action='store',
            default=None,
            help="the cluster sh number"
        ))


    def run(self, args):
        try:
            options, arg_else = self.parse_args(args)
        except BadOptionError:
            self.logger.error("ERROR: %s" % str(sys.exc_info()[1]))
            return

        # scale services
        if not options.env_name:
            self.logger.error("ERROR: %s" % "please provide environment name")
        else:
            riderEnv = Environment.get_environment_by_name(options.env_name)
            riderEnv.indexer.scale(int(options.indexer_num))


class CleanCommand(Command):
    name = "clean"
    usage = """%prog """
    summary = "clean the environment"

    def __init__(self):
        super(CleanCommand, self).__init__()
        self.parser.add_option(Option(
            '--envname',
            dest='envname',
            action='store',
            help="the name of environment"
        ))

    def run(self, args):
        try:
            options, arg_else = self.parse_args(args)
            msg = "Do you really want to delete the environment %s [y/n]:" % options.envname
            user_input = get_userinput(msg)
            if is_true(user_input):
                environment = Environment.get_environment_by_name(name=options.envname)
                environment.clean()
                self.logger.info("the environment [%s] has been deleted" % options.envname)
        except BadOptionError:
            self.logger.error("ERROR: %s" % str(sys.exc_info()[1]))
            return


class BuildCommand(Command):
    name = "build"
    usage = """%prog """
    summary = "build the new image with splunk package"

    def __init__(self):
        super(Command, self).__init__()
        self.parser.add_option(Option(
            '--splunk-pkg',
            dest='splunk_pkg',
            action='store',
            default='splunk_tgz',
            help="Splunk release build package"
        ))

        self.parser.add_option(Option(
            '--image-name',
            dest='image_name',
            action='store',
            default='10.66.128.203:49153/coreqa/splunk:latest',
            help="Image Name"
        ))

    def run(self, args):
        pass


class BootStrapCommand(Command):
    name = "bootstrap"
    usage = """%prog """
    summary = "first bootstrap the environment"

    def __init__(self):
        super(BootStrapCommand, self).__init__()
        self.parser.add_option(Option(
            '--envname',
            dest='envname',
            action='store',
            help="the name of environment"
        ))
        self.parser.add_option(Option(
            '--envtype',
            dest='envtype',
            action='store',
            help="the type of environment"
        ))

    def run(self, args):
        pass


class ShowCommand(Command):
    name = "show"
    usage = """%prog """
    summary = "show the information for the matrix"

    def __init__(self):
        super(ShowCommand, self).__init__()

    def run(self, args):
        # define the subclass here
        class EnvCommand(self.__class__):
            name = "env"

            def __init__(self):
                super(EnvCommand, self).__init__()
                self.parser.add_option(Option(
                    '--envname',
                    dest='envname',
                    action='store',
                    default=None,
                    help="the name of environment"
                ))

            def run(self, args):
                try:
                    options, arg_else = self.parse_args(args)
                    if options.envname:
                        pass
                    else:
                        pass
                except BadOptionError:
                    self.logger.error("ERROR: %s" % str(sys.exc_info()[1]))
                    return

        class EnvTypeCommand(self.__class__):
            name = "envtype"

            def __init__(self):
                super(EnvTypeCommand, self).__init__()
                self.parser.add_option(Option(
                    '--envtype',
                    dest='envtype',
                    action='store',
                    default=None,
                    help="the type of environment"
                ))

            def run(self, args):
                try:
                    options, arg_else = self.parse_args(args)
                    if options.envtype:
                        pass
                    else:
                        pass
                except BadOptionError:
                    self.logger.error("ERROR: %s" % str(sys.exc_info()[1]))
                    return


        sub_commands = {
            EnvCommand.name: EnvCommand,
            EnvTypeCommand.name: EnvTypeCommand
        }

        parser_kw = {
            'usage': self.usage,
            'prog': '%s %s <subcommand> [options]' % (get_prog(), self.name),
            'formatter': UpdatingDefaultsHelpFormatter(),
            'add_help_option': False,
            'name': self.name,
            'description': self.__doc__,
        }
        sub_command, args_else = parse_opts(args, get_commands_summary(sub_commands), parser_kw,
                                            cmdoptions.general_group)

        sub_commands[sub_command]().run(args_else)


COMMANDS = {
    ShowCommand.name: ShowCommand,
    BuildCommand.name: BuildCommand,
    CleanCommand.name: CleanCommand,
    ScaleCommand.name: ScaleCommand,
    BootStrapCommand.name: BootStrapCommand
}


def get_commands_summary(commands):
    """Yields sorted (command name, command summary) tuples."""
    cmd_items = commands.items()
    for name, command_class in cmd_items:
        yield (name, command_class.summary)

