import sys
import optparse
from optparse import Option, BadOptionError

import re
from texttable import Texttable
import os
from rider.cli import cmdoptions
from rider.cli.cmdparser import ConfigOptionParser, UpdatingDefaultsHelpFormatter, parse_opts
from rider.cli.utils import get_prog, get_userinput_boolean, get_userinput_str, get_userinput_choice, get_userinput_int, \
    is_true
from rider.log import Logger
from rider.environment import Environment, ENV_CLASS_MAPPING
from rider.image import Image
from rider.pkg.fig.cli.docker_client import docker_client


class Command(object):
    name = None
    usage = None
    hidden = None
    summary = ""

    def __init__(self):
        self.parser_kw = {
            'usage': self.usage,
            'prog': '%s %s' % (get_prog(), self.name),
            'formatter': UpdatingDefaultsHelpFormatter(),
            'add_help_option': False,
            'name': self.name,
            'description': self.__doc__,
        }

        self.parser = ConfigOptionParser(**self.parser_kw)

        # Commands should add options to this option group
        optgroup_name = '%s Options' % self.name.capitalize()
        self.cmd_opts = optparse.OptionGroup(self.parser, optgroup_name)

        # Add the general options
        gen_opts = cmdoptions.make_option_group(cmdoptions.general_group, self.parser)
        self.parser.add_option_group(gen_opts)

        # set logger
        self.logger = Logger()
        self.logger.add_consumers(
            (Logger.VERBOSE_DEBUG, sys.stdout),
        )

    def parse_args(self, args):
        # factored out for testability
        return self.parser.parse_args(args)

    def run(self, args):
        """
            The sub command class should overide this method
        """
        NotImplemented

    def execute(self, args=None):
        """
            The main interface for exectute the command
        """
        try:
            self.run(args)
        except Exception:
            sys.stderr.write("ERROR: %s \n" % str(sys.exc_info()[1]))
            sys.exit(1)
        except KeyboardInterrupt:
            sys.exit(1)


class ScaleCommand(Command):
    name = "scale"
    usage = """%prog """
    summary = "scale the service of the environment"

    def __init__(self):
        super(ScaleCommand, self).__init__()

        self.parser.add_option(Option(
            '--envname',
            dest='envname',
            action='store',
            default=None,
            help="the name of environment"
        ))

    def run(self, args):
        try:
            options, arg_else = self.parse_args(args)
        except BadOptionError:
            sys.stderr.write("ERROR: %s \n" % str(sys.exc_info()[1]))
            return

        # scale services
        if not options.envname:
            sys.stderr.write("ERROR: %s \s" % "please provide environment name")
        else:
            env = Environment.get_environment_by_name(options.envname)
            scaleable_service_list = env.get_scaleable_services_list()
            service_scale_dict = {}
            for service in scaleable_service_list:
                scaled_number = get_userinput_int(
                    "Input the expected instance number for the service [%s] after scaled : " % service.name)
                if scaled_number <= len(service.machines):
                    sys.stdout.write("the scaled number is even smaller than the current scope , escape\n")
                else:
                    service_scale_dict[service] = scaled_number

            for service in service_scale_dict:
                sys.stdout.write(
                    "scale the [%s] from %s to %s \n" % (
                        service.name, len(service.machines), service_scale_dict[service]))
                service.scale(service_scale_dict[service])


class CleanCommand(Command):
    name = "clean"
    usage = """%prog """
    summary = "clean the environment"

    def __init__(self):
        super(CleanCommand, self).__init__()
        self.parser.add_option(Option(
            '--envname',
            dest='envname',
            action='store',
            help="the name of environment"
        ))

    def run(self, args):
        try:
            options, arg_else = self.parse_args(args)
            msg = "Do you really want to delete the environment %s [y/n]: " % options.envname
            user_input = get_userinput_boolean(msg)
            if is_true(user_input):
                environment = Environment.get_environment_by_name(name=options.envname)
                environment.clean()
                sys.stdout.write("the environment [%s] has been deleted \n" % options.envname)
        except BadOptionError:
            sys.stderr.write("ERROR: %s" % str(sys.exc_info()[1]))
            return


class BuildCommand(Command):
    name = "build"
    usage = """%prog """
    summary = "build the new image with splunk package"

    def __init__(self):
        super(BuildCommand, self).__init__()

    def run(self, args):
        ns = get_userinput_str("Please input the namespace : ").strip()
        splunk_pkg_path = get_userinput_str("Please input the splunk build package path : ").strip()
        if not os.path.exists(splunk_pkg_path):
            sys.stderr.write("the splunk pkg path [%s] not existed \n" % splunk_pkg_path)
        build_number = get_userinput_int("Please input the splunk build number: ")
        existed_image = Image.get_image_by_build_id(build_number)
        if existed_image:
            sys.stderr.write("the build [%s] of splunk package already existed \n" % build_number)
        new_image = Image.build_image("", ns, build_number, splunk_pkg_path)
        sys.stdout.write("successfully build the new image [%s] \n" % new_image.image_name)


class BootStrapCommand(Command):
    name = "bootstrap"
    usage = """%prog """
    summary = "first bootstrap the environment"

    def __init__(self):
        super(BootStrapCommand, self).__init__()

    def run(self, args):
        env_name = get_userinput_str("Input the env name: ")
        envtype_choice_list = [key for key in ENV_CLASS_MAPPING.keys()]
        env_type = get_userinput_choice(envtype_choice_list, "Choose the envirnoment type ")

        # check the env whether existed
        rider_env_name = "%s_%s" % (env_name, env_type)
        old_env = Environment.get_environment_by_name(rider_env_name)
        if old_env:
            sys.stderr.write("the environment %s already exites , please choose another name")
            sys.exit(1)

        image_dict = Image.get_all_images()
        if not image_dict:
            sys.stderr.write("no available image exists , please use `rider build` to build the image first \n")

        image_choice_list = [image_dict[key].image_name for key in image_dict.keys()]
        image_name = get_userinput_choice(image_choice_list, "Choose the image")

        new_env = Environment.create_environment(env_type=env_type, env_name=env_name, image_name=image_name)
        new_env.bootstrap()

        sys.stdout.write("successfully create the environment [%s] \n" % env_name)


class ShowCommand(Command):
    name = "show"
    usage = """%prog """
    summary = "show the information for the matrix"

    def __init__(self):
        super(ShowCommand, self).__init__()

    def get_prog(self):
        return "%s %s" % (get_prog(), self.name)

    def run(self, args):
        # define the subclass here
        class EnvCommand(self.__class__):
            name = "show env"
            subcommand = "env"
            usage = """%prog"""
            summary = "show the information for the matrix"

            def __init__(self):
                super(EnvCommand, self).__init__()
                self.parser.add_option(Option(
                    '--envname',
                    dest='envname',
                    action='store',
                    default=None,
                    help="the name of environment"
                ))

            def run(self, args):
                try:
                    options, arg_else = self.parse_args(args)
                    if options.envname:
                        tb = Texttable()
                        tb.header(["container_name", "port mapping", "splunk authentication",
                                   "ssh authentication"])
                        env = Environment.get_environment_by_name(options.envname)
                        if not env:
                            sys.stderr.write("the environment [%s] does not exist " % options.envname)
                        for service in env.service_list:
                            for machine in service.machines:
                                table = []
                                table.append(machine.name)
                                table.append(machine.container.human_readable_ports)
                                table.append("admin/notchangeme")
                                table.append("root/password")
                                tb.add_row(table)
                        sys.stdout.write(tb.draw())
                        sys.stdout.write("\n")
                    else:
                        tb = Texttable()
                        tb.header(["Environments"])
                        for env in Environment.get_environments():
                            tb.add_row([env.env_name])
                        sys.stdout.write(tb.draw())
                        sys.stdout.write("\n")
                except BadOptionError:
                    sys.stderr.write("ERROR: %s" % str(sys.exc_info()[1]))
                    return

        class ImageCommand(self.__class__):
            name = "show image"
            subcommand = "image"
            usage = """%prog"""
            summary = "show the information for the matrix"

            def __init__(self):
                super(ImageCommand, self).__init__()

            def run(self, args):
                images = Image.get_all_images()
                tb = Texttable()
                tb.header(["rider support images"])
                for key in images.keys():
                    tb.add_row([images[key].image_name])
                sys.stdout.write(tb.draw())
                sys.stdout.write("\n")

        sub_commands = {
            EnvCommand.subcommand: EnvCommand,
            ImageCommand.subcommand: ImageCommand
        }

        parser_kw = {
            'usage': '\n%prog <command> [options]',
            'add_help_option': False,
            'formatter': UpdatingDefaultsHelpFormatter(),
            'name': 'global',
            'prog': self.get_prog(),
        }
        sub_command, args_else = parse_opts(args, get_commands_summary(sub_commands), parser_kw,
                                            cmdoptions.general_group)

        sub_commands[sub_command]().run(args_else)

class WRiderCommand(Command):
    name = "wrider"
    usage = """%prog """
    summary = "build wrider image with splunk package"

    def __init__(self):
        super(WRiderCommand, self).__init__()
        self.wrider_container_name = u'wrider_webserver_container'
        self.client = docker_client()

    def get_prog(self):
        return "%s %s" % (get_prog(), self.name)

    def run(self, args):
        #sub command class
        class WRiderStartCommand(self.__class__):
            name = "wrider start"
            subcommand = "start"
            usage = """%prog"""
            summary = "start wrider container"

            def __init__(self):
                super(WRiderStartCommand, self).__init__()

            def run(self, args):

                if self.is_wrider_container_exist():
                    if self.is_wrider_container_up():
                        h_port = self.get_wrider_container_port()
                        msg = '\nWRider web link is: http://boot2docker:{p}\n'.format(p=h_port)
                        sys.stdout.write(msg)
                        sys.stdout.write('\n')
                    else:
                        self.remove_wrider_container()
                        self.create_wrider_container()
                else:
                    # Create container with the name
                    self.create_wrider_container()

        class WRiderStopCommand(self.__class__):
            name = "wrider stop"
            subcommand = "stop"
            usage = """%prog"""
            summary = "stop wrider container"

            def __init__(self):
                super(WRiderStopCommand, self).__init__()

            def run(self, args):
                if self.is_wrider_container_exist():
                    self.stop_wrider_container()

        sub_commands = {
            WRiderStartCommand.subcommand: WRiderStartCommand,
            WRiderStopCommand.subcommand: WRiderStopCommand
        }

        parser_kw = {
            'usage': '\n%prog <command> [options]',
            'add_help_option': False,
            'formatter': UpdatingDefaultsHelpFormatter(),
            'name': 'global',
            'prog': self.get_prog(),
        }
        sub_command, args_else = parse_opts(args, get_commands_summary(sub_commands), parser_kw,
                                            cmdoptions.general_group)

        sub_commands[sub_command]().run(args_else)

    def is_wrider_container_exist(self):
        containers = self.client.containers(all=True)
        for container in containers:
            name = self.get_container_name(container)
            if name == self.wrider_container_name:
                return True
        return False

    def is_wrider_container_up(self):
        containers = self.client.containers(all=True)
        for container in containers:
            name = self.get_container_name(container)
            if name == self.wrider_container_name:
                if 'Up' in str(container['Status']):
                    return True
        return False

    def get_container_name(self, container):
        if container:
            name_list = container['Names']
            return name_list[0][1:]
        return None

    def stop_wrider_container(self):
        containers = self.client.containers(all=True)
        for container in containers:
            if self.get_container_name(container) == self.wrider_container_name:
                sys.stdout.write('Stopping...')
                self.client.stop(container["Id"])

        msg = '\nWRider container was stopped successfully.\n\n'
        sys.stdout.write(msg)

    def remove_wrider_container(self):
        containers = self.client.containers(all=True)
        for container in containers:
            if self.get_container_name(container) == self.wrider_container_name:
                self.client.remove_container(container["Id"])

    def create_wrider_container(self):

        cert_path = os.environ.get('DOCKER_CERT_PATH', '')
        docker_host = "tcp://boot2docker:2376"
        docker_cert_path = "/code/boot2docker-vm"
        image_name = "10.66.128.203:49153/coreqa/wrider"
        environment = ['DOCKER_TLS_VERIFY="1"',
                       'DOCKER_CERT_PATH='+docker_cert_path,
                       'DOCKER_HOST='+docker_host]

        container_id = self.client.create_container(
            image=image_name, environment=environment, name=self.wrider_container_name)

        sys.stdout.write('Starting...\n')
        self.client.start(container=container_id, publish_all_ports=True,
                     binds={cert_path: {'bind': docker_cert_path, 'ro': False}})

        docker_host_ip = self.get_docker_host_ip()
        msg = '\n\
        WRider containder started successfully\n \
        You need to do below two steps to use your WRider container ready:\n \
        1. sudo echo "{i}   boot2docker" >> /etc/hosts\n \
        2. Open your browser and access http://boot2docker:{p}\n \
        '.format(i=docker_host_ip, p=self.get_wrider_container_port())
        sys.stdout.write(msg)
        sys.stdout.write('\n')

    def get_wrider_container_port(self):
        containers = self.client.containers(all=True)
        for container in containers:
            if self.get_container_name(container) == self.wrider_container_name:
                h_port = self.client.port(container["Id"], 5000)
                return h_port[0].get('HostPort')


    def get_docker_host_ip(self):
        '''
        Read DOCKER_HOST environment variable and get the IP address of docker host.
        :return: Docker host IP address
        '''
        hostname = '192.168.59.103'
        docker_host = os.environ.get('DOCKER_HOST', '')
        if docker_host:
            match = re.search("tcp://(.*):[0-9]+", docker_host)
            hostname = match.group(1)
        return hostname

COMMANDS = {
    ShowCommand.name: ShowCommand,
    BuildCommand.name: BuildCommand,
    CleanCommand.name: CleanCommand,
    ScaleCommand.name: ScaleCommand,
    BootStrapCommand.name: BootStrapCommand,
    WRiderCommand.name: WRiderCommand
}


def get_commands_summary(commands):
    """Yields sorted (command name, command summary) tuples."""
    cmd_items = commands.items()
    for name, command_class in cmd_items:
        yield (name, command_class.summary)

