"""
A prettier printer
introdued by Philip Wadler
 http://homepages.inf.ed.ac.uk/wadler/papers/prettier/prettier.pdf
implemented by NISHIO Hirokazu
licensed same as Python 3.0

tests: 
>>> pretty(10, Text("a"))
'a'

>>> doc = fillwords("a b c d e")
>>> print(pretty(10, doc))
a b c d e

>>> print(pretty(5, doc))
a b c
d e

>>> print(pretty(3, doc))
a b
c d
e

>>> doc = bracket("(", fillwords("a b c d e"), ")")
>>> print(pretty(10, doc))
(
  a b c d e
)

>>> print(pretty(5, doc))
(
  a b
  c d
  e
)

>>> docs = [Text("1. abc"), Text("2. abc"), Text("3. abc")]
>>> print(pretty(40, fill(docs)))
1. abc 2. abc 3. abc

>>> print(pretty(13, fill(docs)))
1. abc 2. abc
3. abc

>>> print(pretty(8, fill(docs)))
1. abc
2. abc
3. abc

>>> print(pretty(1, spread(docs))) # horizontal only
1. abc 2. abc 3. abc

>>> print(pretty(100, stack(docs))) # vertical only
1. abc
2. abc
3. abc

# when there isn't enough space to put horizontal, put vertical
>>> print(pretty(20, stack_or_spread(docs)))
1. abc 2. abc 3. abc

>>> print(pretty(13, stack_or_spread(docs)))
1. abc
2. abc
3. abc

# when there isn't enough space to put horizontal, use indent
>>> print(pretty(10, can_break_with_nest(docs)))
1. abc
  2. abc
  3. abc

>>> pretty(5, NIL)
''

>>> fill([])
(Nil)

>>> spread([])
(Nil)

>>> flatten(NIL)
(Nil)

samples:
>>> print(pretty(40, Concat(Text("a"), Text("b")).pretty()))
(Concat (Text 'a') (Text 'b'))

>>> print(pretty(80, Concat(Text("a"), Text("b")).pretty().pretty()))
(Union
  (Text "(Concat (Text 'a') (Text 'b'))")
  (Concat
    (Text '(Concat')
    (Nest 1
      (Concat
        (Line)
        (Concat
          (Concat (Text "(Text 'a')") (Concat (Line) (Text "(Text 'b')")))
          (Text ')'))))))
"""
try:
    from cStringIO import StringIO
except: # pragma: no cover
    from io import StringIO # for py3


from collections import deque
from copy import copy

# === settings
SEPARATOR = " "
NEWLINE = "\n"
INDENT = "  "
length = len # take string s and return length of it

# === Document with Alternative layout
# method "pretty" is sample
class ADoc:
    def pretty(self):
        return Text(repr(self))


class Nil(ADoc):
    def __repr__(self):
        return "(Nil)"

NIL = Nil()


class Concat(ADoc): # operator <>
    def __init__(self, x, y):
        if not isinstance(x, ADoc):
            raise AssertionError("Concat arg 1 must be ADoc: %r" % x)
        if not isinstance(y, ADoc):
            raise AssertionError("Concat arg 2 must be ADoc: %r" % y)
        self.x = x
        self.y = y

    def __repr__(self):
        return "(Concat %r %r)" % (self.x, self.y)

    def pretty(self):
        return sample_impl("Concat", self.x, self.y)

class Nest(ADoc):
    def __init__(self, i, x):
        if not isinstance(i, int):
            raise AssertionError("Nest arg 1 must be int: %r" % i)
        if not isinstance(x, ADoc):
            raise AssertionError("Nest arg 2 must be ADoc: %r" % x)
        self.i = i
        self.x = x

    def __repr__(self):
        return "(Nest %r %r)" % (self.i, self.x)

    def pretty(self):
        return Union(
            Text("(Nest %d %r)" % (self.i, self.x)),
            concat(
                Text("(Nest %d" % self.i), 
                Nest(1, concat(
                        LINE,
                        self.x.pretty(),
                        Text(")")))))


class Text(ADoc):
    def __init__(self, s):
        if not isinstance(s, str):
            raise AssertionError("Nest arg 1 must be str: %r" % s)
        self.s = s
    def __repr__(self):
        return "(Text %r)" % (self.s)


class Line(ADoc):
    def __repr__(self):
        return "(Line)"

LINE = Line()


class Union(ADoc):
    def __init__(self, x, y):
        if not isinstance(x, ADoc):
            raise AssertionError("Union arg 1 must be ADoc: %r" % x)
        if not isinstance(y, ADoc):
            raise AssertionError("Union arg 2 must be ADoc: %r" % y)

        self.x = x
        self.y = y

    def __repr__(self):
        return "(Union %r %r)" % (self.x, self.y)

    def pretty(self):
        return sample_impl("Union", self.x, self.y)

# === Docment with single layout, used internally
class Doc: pass 


class Nil_(Doc): pass
NIL_ = Nil_()


class Text_(Doc):
    def __init__(self, s, x):
        if not isinstance(s, str):
            raise AssertionError("Text_ arg 1 must be sre: %r" % s)
        if not isinstance(x, Doc):
            raise AssertionError("Text_ arg 2 must be Doc: %r" % x)
        self.s = s
        self.x = x


class Line_(Doc):
    def __init__(self, i, x):
        if not isinstance(i, int):
            raise AssertionError("Line_ arg 1 must be int: %r" % i)
        if not isinstance(x, Doc):
            raise AssertionError("Line_ arg 2 must be Doc: %r" % x)
        self.i = i
        self.x = x


# === core functions    

def group(x):
    return Union(flatten(x), x)


def flatten(x):
    if x is NIL:
        return NIL
    if isinstance(x, Concat):
        return Concat(flatten(x.x), flatten(x.y))
    if isinstance(x, Nest):
        return Nest(x.i, flatten(x.x))
    if isinstance(x, Text):
        return x
    if isinstance(x, Line):
        return Text(SEPARATOR)
    if isinstance(x, Union):
        return flatten(x.x)
    raise NotImplementedError(x)


def layout(x, sio=None):
    if x is NIL_:
        return sio
    if isinstance(x, Text_):
        sio.write(x.s)
        layout(x.x, sio)
        return sio
    if isinstance(x, Line_):
        sio.write(NEWLINE + INDENT * x.i)
        layout(x.x, sio)
        return sio
    
    raise NotImplementedError

    
def best(width, already, x):
    return be(width, already, deque([(0, x)]))


def _build_result(start, result):
    cur = start
    for (f, a) in reversed(result):
        cur = f(a, cur)
    return cur


def be(width, already, x):
    result = deque()

    while True:
        if not x:
            return _build_result(NIL_, result)

        head = x.popleft()
        i, y = head
        if y is NIL:
            continue
        if isinstance(y, Concat):
            x.extendleft([(i, y.y), (i, y.x)])
            continue
        if isinstance(y, Nest):
            x.appendleft((i + y.i, y.x))
            continue
        if isinstance(y, Text):
            result.append((Text_, y.s))
            already += length(y.s)
            continue
        if isinstance(y, Line):
            result.append((Line_, i))
            already = i
            continue
        if isinstance(y, Union):
            lhs = copy(x)
            lhs.appendleft((i, y.x))
            lhs = be(width, already, lhs)
            if fits(width - already, lhs):
                return _build_result(lhs, result)

            x.appendleft((i, y.y))
            continue
        raise NotImplementedError(x)


def fits(width, x):
    if width < 0: return False
    if isinstance(x, Nil_): return True
    if isinstance(x, Text_):
        return fits(width - length(x.s), x.x)
    if isinstance(x, Line_):
        return True
    raise NotImplementedError(x)


def pretty(width, x):
    sio = StringIO()
    sio = layout(best(width, 0, x), sio)
    return sio.getvalue()


# --- utility functions ---

def folddoc(f, xs):
    try:
        n = len(xs)
    except TypeError:
        xs = list(xs)
        n = len(xs) # for py3

    if n == 0:
        return NIL
    if n == 1:
        if isinstance(xs[0], list) or isinstance(xs[0], tuple):
            # TODO: it should be misuse. warn
            return folddoc(f, xs[0])
        return xs[0]
    return f(xs[0], folddoc(f, xs[1:]))


def concat(*xs):
    return folddoc(Concat, xs)


def union(*xs):
    return folddoc(Union, xs)


# overload (+)?
def join_with_space(x, y): # operator <+>
    return concat(x, Text(" "), y)


# overload (/)?
def join_with_line(x, y): # operator </>
    return concat(x, LINE, y)

def spread(*xs):
    return folddoc(join_with_space, xs)


def stack(*xs):
    return folddoc(join_with_line, xs)


def stack_or_spread(xs):
    return union(spread(xs), stack(xs))


def bracket(l, x, r):
    return group(
        concat(
            Text(l),
            Nest(1, Concat(LINE, x)),
            LINE,
            Text(r)))


# overload (*)?
def join_with_can_break(x, y): # <+/>
    return concat(x, Union(Text(" "), LINE), y)


def can_break(*xs):
    return folddoc(join_with_can_break, xs)


def can_break_with_nest(*xs):
    if len(xs) == 1 and isinstance(xs[0], list): xs = xs[0] # TODO generalize and put on each *xs function
    return Union(
        spread(*xs),
        Concat(xs[0], Nest(1, Concat(LINE, stack(*xs[1:])))))



def fillwords(s):
    "split string and join it with 'can_break'"
    return folddoc(can_break, (Text(word) for word in s.split()))


def fill(xs):
    if xs == []:
        return NIL
    if len(xs) == 1:
        return xs[0]
    x = xs[0]
    y = xs[1]
    zs = xs[2:]
    return Union(
        join_with_space(flatten(x), fill([flatten(y)] + zs)),
        join_with_line(x, fill([y] + zs)))


def sample_impl(head, x, y):
    "see sample in doc of this module"
    return Union(
        Text("(%s %r %r)" % (head, x ,y)),
        concat(
            Text("(%s" % head), 
            Nest(1, concat(
                    LINE,
                    stack(x.pretty(),
                          y.pretty()),
                    Text(")")))))


def _test():
    import doctest
    doctest.testmod()

def _speed_test(N, width): # pragma: no cover
    print(pretty(width, reduce(can_break, (Text(str(word)) for word in range(N)))))

if __name__ == "__main__": # pragma: no cover
    from optparse import OptionParser
    parser = OptionParser()
    parser.add_option("-s", "--speedtest", dest="n", type=int,
                      help="call _speed_test(N)", metavar="N")
    parser.add_option("-w", "--width", dest="width", type=int, default=80,
                      help="set width for _speed_test", metavar="WIDTH")

    (options, args) = parser.parse_args()
    if options.n:
        import cProfile
        cProfile.run("_speed_test(options.n, options.width)")
    else:
        _test()
