"""Contains the code for building the new package structure from a set
of templates."""
import os
import os.path
import re
import shutil
import imp
import uuid

import jinja2

import utils

class BuilderError(Exception):
    """A generic error occuring when the builder is running."""
    pass

class Builder(object):
    """Builds a fresh package structure from a given template directory."""

    STANDARD_IGNORES = [
        r'^\.pm$', r'^\.config\.py$', r'^\.svn$', r'^\.git$', r'^\.hg$'
        ]
    """The set of filenames that you almost certainly don't want to be copied
    from the template to the new package. This list consists of the main
    version control directories. You can pass in any set of ignores to this
    class's constructor, but it is more common to add this list to whatever
    else you want to ignore. For example:

    >>> ignore = Builder.STANDARD_IGNORES + [r'\.build']
    >>> b = Builder('foo', 'bar', ignore_filenames=ignore)
    """

    def __init__(
        self, original_root, new_root,
        parameters={},
        ignore_filenames=STANDARD_IGNORES,
        verbatim_filenames=[]
        ):

        self.original_root = os.path.abspath(original_root)
        self.new_root = os.path.abspath(new_root)
        self.parameters = parameters
        self._extra_config = []

        # Compile the given regular expressions
        self._ignore_filenames = [
            re.compile(regex) for regex in ignore_filenames
            ]
        self._verbatim_filenames = [
            re.compile(regex) for regex in verbatim_filenames
            ]

        self._jinja2_environment = None

    def run(self):
        """Runs the building process."""
        # Start with no templating environment (create one lazily).
        self._jinja2_environment = None

        # Start the recursive build at the root directories
        assert not self._extra_config
        self._handle_directory('.', '.')

    def _get_abs_paths(self, source_path, target_path):
        """Returns the absolute path for the given file or directory,
        relative to both the original and the new root.

        Returns a 2-tuple of (absolute original path, absolute new path).
        """
        abs_original = os.path.abspath(
            os.path.join(self.original_root, source_path)
            )

        abs_new = os.path.abspath(
            os.path.join(self.new_root, target_path)
            )

        return abs_original, abs_new

    def _get_explicit_contents(self, source_path, files):
        """Loads and templates a '.pm' file to get a dictionary of
        files to copy in this directory."""

        pm_path = os.path.join(source_path, '.pm')

        # Load and render the template
        content = self._render_template(pm_path, files=files)

        # Clean up each line
        lines = {}
        all_files = False
        for line in content.splitlines():
            line = line.strip()

            # Omit comments
            if not line or line[0] == '#': continue

            # Check for all contents
            if line == '*':
                all_files = True
                continue

            # Check for a translation
            if ' -> ' in line:
                source, target = line.split(' -> ')
                lines[source.strip()] = target.strip()
            else:
                lines[line] = line

        # Add any additional files from the directory
        if all_files:
            for fn in files:
                if fn not in lines:
                    lines[fn] = fn

        return lines

    def _get_directory_mapping(self, abs_path, source_path, files):
        """Returns a mapping from source filenames to target filenames
        for the source files in the given directory. This method will
        honor ``.pm`` files, if it finds them."""

        # If we have a '.pm' file, then it should contain a template
        # that, when rendered, will provide a list of the files to contain
        pm_fn = os.path.join(abs_path, '.pm')
        if os.path.isfile(pm_fn):
            return self._get_explicit_contents(source_path, files)
        else:
            return dict([(fn, fn) for fn in files])

    def _path_exists(self, abs_path):
        """Warns the user that the given path exists. Eventually
        this may also undo the builder's actions so far."""
        raise BuilderError('The path: %s already exists' % abs_path)

    def _handle_directory(self, source_path, target_path):
        """Performs the building algorithm on the given path, relative
        to the root directory. If the ``target_path`` argument is given,
        then the resulting directory will be named accordingly."""

        abs_original, abs_new = self._get_abs_paths(source_path, target_path)

        # Find any additional config.
        self._load_additional_config(abs_original)

        # Make sure we don't exist already
        if os.path.exists(abs_new):
            self._path_exists(abs_new)

        # Create the new directory.
        os.mkdir(abs_new)

        # Remove any ignore files from the list.
        files = [
            fn
            for fn in os.listdir(abs_original)
            if not self._check_ignore(fn)
            ]

        # Find how we'll map source to destination filenames.
        mapping = self._get_directory_mapping(abs_original, source_path, files)


        # Recurse through each of its contents.
        for source, target in mapping.items():
            abs_path = os.path.join(abs_original, source)
            if os.path.isdir(abs_path):
                self._handle_directory(
                    os.path.join(source_path, source),
                    os.path.join(target_path, target)
                    )
            else:
                self._handle_file(
                    os.path.join(source_path, source),
                    os.path.join(target_path, target)
                    )

        # Pop down a level on the config stack.
        self._extra_config.pop()

    def _load_additional_config(self, path):
        """
        Loads additional config data from the ``.config.py`` file in
        the given path.
        """
        fn = os.path.join(path, ".config.py")

        config_dict = dict()
        if os.path.isfile(fn):
            module_name = "module_%s" % str(uuid.uuid4()).replace("-", "")
            try:
                module = imp.load_source(module_name, fn)
                ignore = [
                    re.compile(rex) for rex in getattr(module, 'ignore', [])
                    ]
                verbatim = [
                    re.compile(rex) for rex in getattr(module, 'verbatim', [])
                    ]
                config_dict = dict(
                    context = getattr(module, 'context', dict()),
                    ignore = ignore,
                    verbatim = verbatim
                    )
                # Remove the byte-compiled version.
                os.unlink(fn+"c")
            except ValueError:
                pass

        self._extra_config.append(config_dict)

    def _handle_file(self, source_path, target_path):
        """Copies across the file at the given relative path."""


        abs_original, abs_new = self._get_abs_paths(source_path, target_path)

        # Make sure we don't exist already.
        if os.path.exists(abs_new):
            self._path_exists(abs_new)

        # Check if we're looking at a text file that isn't exempt from
        # conversion.
        file_name = os.path.basename(source_path)
        if utils.is_text_file(abs_original) and \
                not self._check_verbatim(file_name):
            # We need to run the file through Jinja before outputting.
            content = self._render_template(source_path)

            # Save the file to its new location.
            file = open(abs_new, 'w')
            try:
                file.write(content)
            finally:
                file.close()

            # Copy across permission bits
            shutil.copymode(abs_original, abs_new)

        else:
            # We can do a straight copy
            shutil.copy(abs_original, abs_new)

    def _get_jinja2_environment(self):
        """Returns the templating environment, creating it if it isn't
        there already."""
        if self._jinja2_environment is None:
            self._jinja2_environment = jinja2.Environment(
                loader = jinja2.FileSystemLoader(self.original_root)
                )
        return self._jinja2_environment

    def _render_template(self, source_path, **additional_context):
        """Renders the template at the given path, and returns the
        rendered content."""
        environment = self._get_jinja2_environment()
        template = environment.get_template(source_path)

        context = {}
        # Add directory specific context.
        for config in self._extra_config:
            context.update(config.get('context', dict()))
        # Add context passed in on the command line.
        context.update(self.parameters)
        # Add context explicitly required by this class.
        context.update(additional_context)

        return template.render(**context)

    def _check_ignore(self, source_path):
        """Checks to see if this path matches any of the regular
        expressions we're ignoring."""

        ignores = self._ignore_filenames
        for config in self._extra_config:
            ignores.extend(config.get('ignore', []))

        return self._check_regexes(source_path, ignores)

    def _check_verbatim(self, source_path):
        """Checks to see if this path is one of those we shouldn't run
        through the tempating system."""

        verbatims = self._verbatim_filenames
        for config in self._extra_config:
            verbatims.extend(config.get('verbatim', []))

        return self._check_regexes(source_path, verbatims)

    def _check_regexes(self, source_path, regexes):
        """Checks to see if the given path matches any of the given
        regular expressions."""
        for regex in regexes:
            if regex.search(source_path):
                return True
        return False

