#! /usr/bin/env python
# coding: utf-8

import os.path
from optparse import OptionParser
import re
import shutil


__version__ = '0.1.0'


class Tokenizer(object):
    """Iterator for splitting a string in Tokens, to be read by `Parser`.

    Tokens are (typestr, args)-tuples. The args are the groups of the token's
    regexp match (see `tokens`)
    """

    tokens = [
        ('value', re.compile(r'- (.+?): (.*)\n')),
        ('section', re.compile(r'--- (.+?):\n((.|\n)*?)\n---\n')),
        ('underline', re.compile(r'=+\n')),
        ('blankline', re.compile(r'(([ \t]|)+\n)')),
        ('text', re.compile(r'(.*\n)')),
    ]

    def __init__(self, text):
        self.text = text

    def __iter__(self):
        return self

    def __next__(self):
        if not self.text:
            raise StopIteration()

        for name, pattern in self.tokens:
            match = pattern.match(self.text)
            if match:
                self.text = self.text[match.end():]
                return name, match.groups()

        raise RuntimeError('could not tokenize: %r' % self.text[:20])

    next = __next__  # python 2.x compatibility


class Parser(object):
    """Parser for parsing Protoplasma-file."""

    def __init__(self):
        self.book = Book()

    def pop(self):
        """gives the next token and removes it from the stream.
        A Token is a (type, args)-tuple
        """
        return self.tokens.pop(0)

    def pushback(self, token):
        """adds a token back at the stream.

        Works so that self.pushback(self.pop()) is a noop.
        """
        self.tokens.insert(0, token)

    def peek(self):
        """returns the type of the next token."""
        try:
            type_, args = self.tokens[0]
            return type_
        except IndexError:
            return 'eof'

    def skipall(self, type_):
        """removes all the next tokens of type from the stream."""
        while self.peek() == type_:
            self.pop()

    def expect(self, type_, msg):
        if self.peek() != type_:
            raise RuntimeError('%s %s' % (msg, self.tokens[:3]))

    def parse(self, tokens):
        self.tokens = list(tokens)
        while self.tokens:
            self.skipall('blankline')
            self.parse_chapter(self.book)

    def parse_chapter(self, book):
        self.skipall('blankline')

        chapter = Chapter()

        self.parse_headline(chapter)

        go_on = True

        while go_on:
            if self.peek() == 'value':
                self.parse_value(chapter)
            elif self.peek() == 'section':
                self.parse_section(chapter)
            else:
                go_on = self.parse_text()

        book.add_chapter(chapter)

    def parse_headline(self, chapter):
        self.expect('text', 'Expected Headline')

        _, (text,) = self.pop()

        self.expect('underline', 'Expected Headline to be underlined')
        self.pop()

        self.skipall('blankline')

        chapter.headline = text.strip()

    def parse_text(self):
        lines = []

        while self.peek() != 'eof':
            type_, args = token = self.pop()
            if type_ in ('value', 'section') or self.peek() == 'underline':
                self.pushback(token)
                break
            else:
                lines.append(args[0])

        return any(lines)

    def parse_value(self, chapter):
        _, (key, value) = self.pop()
        chapter.set_value(key, value.strip())

    def parse_section(self, chapter):
        _, (key, content, _) = self.pop()
        chapter.set_value(key, content)


class Book(object):
    def __init__(self):
        self.chapters = []

    def __repr__(self):
        return '<Book %r>' % self.chapter

    def add_chapter(self, chapter):
        self.chapters.append(chapter)

    def get_chapter_by_filename(self, filename):
        for c in self.chapters:
            if c.filename == filename:
                return c
        raise KeyError(filename)

    def generate(self):
        for chapter in self.chapters:
            print(chapter)
            if not chapter.filename:
                continue

            extension = chapter.filename.split('.')[-1]
            if extension.lower() not in ['html', 'css']:
                continue

            print('generating file: %s' % chapter.filename)
            with open('results/%s' % chapter.filename, 'w') as fp:
                fp.write(chapter.render(self, {}))


class Chapter(object):

    file_content_key = 'Datei'
    template_key = 'Vorlage'

    def __init__(self):
        self.headline = None
        self.values = {}

    def set_value(self, key, value):
        self.values[key] = value

    @property
    def filename(self):
        m = re.match(r'^.*?\((.*?)\)$', self.headline.strip())
        if m:
            return m.groups()[0]
        else:
            return None

    def __repr__(self):
        return '<Chapter headline=%r, filename=%r, values=%r>' % (
                              self.headline, self.filename, self.values.keys())

    def _replace_all_placeholders(self, source, values):
        """ replaces all placeholders in source with the values given in value
        """
        for key, value in values.items():
            source = re.sub(r'\{\{\s*%s\s*\}\}' % key, value, source)
        return source

    def render(self, book, values):
        result_values = {}

        for key, value in self.values.items():
            result_values[key] = self._replace_all_placeholders(value, values)

        if self.file_content_key in result_values:
            return result_values[self.file_content_key]

        elif self.template_key in result_values:
            extends = book.get_chapter_by_filename(
                                              result_values[self.template_key])
            return extends.render(book, result_values)

        else:
            raise RuntimeError('should have Datei oder Vorlage')


def generator_main(options, book):
    main_chapter = book.get_chapter_by_filename('main')
    plugin_codes = [value for key, value in main_chapter.values.items()
                                                  if key.startswith('plugin.')]

    for code in plugin_codes:
        plugin = eval(code)
        print('Using Plugin %r' % plugin)
        plugin(options, book)


def _write_file(filename, data):
    with open(filename, 'w') as fp:
        fp.write(data)


class TemplateRenderer(object):
    def __init__(self, *extensions):
        self.extensions = extensions

    def __call__(self, options, book):
        target_directory = options['target_directory']

        for chapter in book.chapters:
            if not chapter.filename or not self.rendered_by_me(chapter):
                continue

            filename = os.path.join(target_directory, chapter.filename)
            print('generating file: %s' % filename)
            _write_file(filename, chapter.render(book, {}))

    def rendered_by_me(self, chapter):
        return any(chapter.filename.lower().endswith(ext)
                                                    for ext in self.extensions)

    def __repr__(self):
        exts = ', '.join(repr(ext) for ext in self.extensions)
        return 'TemplateRenderer(%s)' % exts


class StaticFiles(object):
    def __init__(self, path):
        self.path = path

    def __call__(self, options, book):
        target = os.path.join(options['target_directory'],
                                                   os.path.basename(self.path))
        shutil.copytree(self.path, target)

    def __repr__(self):
        return 'StaticFiles(%r)' % self.path


def main():
    optionparser = OptionParser(usage="%prog [SOURCEFILE] [...]")
    optionparser.add_option('-t', '--target-directory', default='results',
                                         help='Directory to store the results')

    opts, args = optionparser.parse_args()
    option_dict = dict(opts.__dict__)

    if not args:
        optionparser.error('please provide pp-files')

    parser = Parser()

    for filename in args:
        with open(filename) as fp:
            parser.parse(Tokenizer(fp.read()))

    generator_main(option_dict, parser.book)


if __name__ == '__main__':
    main()
