"""Pulse WSGI Middleware"""
import logging
import threading
import time
import StringIO
#multiprocessing support on python 2.6
try:
    import multiprocessing
except ImportError, e:
    multiprocessing = None

from paste.deploy.converters import asbool
from webob import Request, Response

log = logging.getLogger('pulse.middleware')

defaultenviron = {
    'REQUEST_METHOD':'GET',
    'SCRIPT_NAME':'',
    'PATH_INFO':'',
    'QUERY_STRING':'',
    'CONTENT_TYPE':'',
    'CONTENT_LENGTH':'',
    'SERVER_NAME':'',
    'SERVER_PORT':'',
    'SERVER_PROTOCOL':'HTTP/1.0',
    'wsgi.version': (1,0),
    'wsgi.url_scheme': 'http',
    'wsgi.input': None,
    'wsgi.errors': None,
    'wsgi.multithread': True,
    'wsgi.multiprocess': True,
    'wsgi.run_once': False,
}

defaulttask = dict(
    interval=300,
    action='action',
)

defaultoption = dict(
    context='pulse',
    mode='auto',
    guard='true',
)

class PulseTask(object):
    """ 
        A Base Task for pulse tasks.  Each task periodically creates
        a WSGI request with the correct context (PATH_INFO) and 
        calls app.  Threading and Processing and handled by sub-classes.
    """
    def __init__(self, name='task', context='/pulse/action', interval=60, 
        app=None):
        """ 
            name - The name of the task.
            context - The context this task should request.
            interval - The time interval to pulse.
            app - The wsgi app to call, with the scheduled task.
        """       
        self.name = name 
        self.interval = int(interval)
        self.app = app
        self.context = context

    def sendrequest(self):
        """
            Create a new request to send to app.
        """
        environ = defaultenviron.copy()
        environ['PATH_INFO'] = self.context
        #simulate file like objects required by spec
        environ['wsgi.errors'] = StringIO.StringIO()
        environ['wsgi.input'] = StringIO.StringIO()
        environ['pulse.info'] = {}
        environ['pulse.info']['context'] = self.context
        environ['pulse.info']['interval'] = self.interval
        req = Request(environ)
        res = req.get_response(self.app)
        if res.status_int != 200:
            log.warn('Task %s returned status code: %s' % (self.context, res.status))

class ThreadedTask(PulseTask, threading.Thread):
    """
        A PulseTask subclass that maintains it's interval
        using the python threading module.
    """
    def __init__(self, **kwargs):
        threading.Thread.__init__(self)
        PulseTask.__init__(self, **kwargs)
        log.debug('Init Task: name: %s, action: %s' % (self.name, self.context))        
    def run(self):
        while True:
            self.sendrequest()
            time.sleep(self.interval)

class ProcessedTask(PulseTask, multiprocessing.Process):
    """
        A PulseTask subclass that maintains it's interval
        using the python multiprocessing module.
    """
    def __init__(self, **kwargs):
        multiprocessing.Process.__init__(self)
        PulseTask.__init__(self, **kwargs)
        log.debug('Init Task: name: %s, action: %s' % (self.name, self.context))        
    def run(self):
        while True:
            self.sendrequest()
            time.sleep(self.interval)
        
#mapping dictionary for pulse.config.mode, to load the 
#correct task subclass.
tasktypes = dict(
    threading = ThreadedTask,
    auto = ThreadedTask,
)
if multiprocessing:
    tasktypes['processing'] = ProcessedTask

class PulseMiddleware(object):
    """ 
        WSGI middleware that is configured to fire periodic requests,
        acting like a scheduler.
    """
    def __init__(self, app, config=None, environ_key='pulse.schedule', 
        tasks=None, **kwargs):
        """
            Initialize the Pulse Middleware, around app.  config is a dictionary
            of configuration options such as those generated by paste, to 
            programmatically configure PulseMiddleware use **kwargs.
            **kwargs take precendence over config
            
        """
        self.app = app
        #create an empty dict if no config is supplied
        config = config or {}      

        #extract configuration and tasks
        #create default options
        self.options = defaultoption.copy()
        self.tasks = tasks or {}
        for k,v in config.iteritems():
            if k.startswith('pulse.config.'):
                self.options[k[13:]] = v
            if k.startswith('pulse.task.'):
                rule = k[11:]
                (taskname, taskconfig) = rule.split('.')
                task = self.tasks.setdefault(taskname, defaulttask.copy())
                task[taskconfig] = v
        for k,v in kwargs.iteritems():
            self.options[k] = v
        #guard list of web contexts to guard
        self.contexts = []
        #start tasks by creating PulseTasks and .start()
        for k,v in self.tasks.iteritems():
            if type(v) is dict:
                #we're dealing with a dictionary of values from paste here

                #work out path_info that will be sent            
                action = v['action']
                context = self.options['context']
                if action.startswith('/'):
                    path_info = action
                else:
                    path_info = '/%s/%s' % (context, action)
                self.contexts.append(path_info)
                task = tasktypes[self.options['mode']](app=self.app, 
                    context=path_info, interval=v['interval'], name=k)
            else:
                #we're dealing with a programmatically configured task here.
                task =v
            task.start()

    def __call__(self, environ, start_response):
        if asbool(self.options['guard']) == True and environ['PATH_INFO'] in self.contexts:
            res = Response()
            res.status = 404
            return res(environ, start_response)
        return self.app(environ, start_response)

def pulse_filter_app_factory(app, global_conf, **kwargs):
    """ Paste filter_app_factory entry point """
    return PulseMiddleware(app, config=global_conf, **kwargs)
    
