# -*- mode: python; coding: utf-8 -*-
# This work is dedicated to the public domain.

"""unicode_to_latex - what it says

Provides unicode_to_latex(u) and unicode_to_latex_string(u).

unicode_to_latex returns ASCII bytes that can be fed to LaTeX to
reproduce the Unicode string 'u' as closely as possible.

unicode_to_latex_string returns a Unicode string rather than bytes.
That is,
  unicode_to_latex(u) = unicode_to_latex_string(u).encode('ascii').
"""

from __future__ import absolute_import, division, print_function, unicode_literals

__all__ = (b'unicode_to_latex unicode_to_latex_string').split ()

from . import text_type

# Based on https://gist.github.com/798549 (owned by github user piquadrat),
# but modified to make a table usable with unicode.translate(). I had to
# comment out a few things and also tweak various of the conversions to make
# the LaTeX more natural. There were also some surprising missing conversions
# (e.g. u2010 -> -).

# The original table comes from http://www.w3.org/Math/characters/unicode.xml,
# converted via this XSL script: https://gist.github.com/798546 . Based on my
# experience so far, the source table is far from perfect.

unicode_to_latex_table_base = {
    u"\u0023": br"\#",
    u"\u0024": br"\textdollar{}",
    u"\u0025": br"\%",
    u"\u0026": br"\&",
    u"\u0027": br"'",
    u"\u002A": br"\ast{}",
    u"\u005C": br"\textbackslash{}",
    u"\u005E": br"\^{}",
    u"\u005F": br"\_",
    u"\u0060": br"\textasciigrave{}",
    u"\u007B": br"\lbrace{}",
    u"\u007C": br"\vert{}",
    u"\u007D": br"\rbrace{}",
    u"\u007E": br"\textasciitilde{}",
    u"\u00A0": br"~",
    u"\u00A1": br"\textexclamdown{}",
    u"\u00A2": br"\textcent{}",
    u"\u00A3": br"\textsterling{}",
    u"\u00A4": br"\textcurrency{}",
    u"\u00A5": br"\textyen{}",
    u"\u00A6": br"\textbrokenbar{}",
    u"\u00A7": br"\textsection{}",
    u"\u00A8": br"\textasciidieresis{}",
    u"\u00A9": br"\textcopyright{}",
    u"\u00AA": br"\textordfeminine{}",
    u"\u00AB": br"\guillemotleft{}",
    u"\u00AC": br"\lnot{}",
    u"\u00AD": br"\-",
    u"\u00AE": br"\textregistered{}",
    u"\u00AF": br"\textasciimacron{}",
    u"\u00B0": br"\textdegree{}",
    u"\u00B1": br"\pm{}",
    u"\u00B2": br"\ensuremath{^2}",
    u"\u00B3": br"\ensuremath{^3}",
    u"\u00B4": br"\textasciiacute{}",
    u"\u00B5": br"\mathrm{\mu}",
    u"\u00B6": br"\textparagraph{}",
    u"\u00B7": br"\cdot{}",
    u"\u00B8": br"\c{}",
    u"\u00B9": br"\ensuremath{^1}",
    u"\u00BA": br"\textordmasculine{}",
    u"\u00BB": br"\guillemotright{}",
    u"\u00BC": br"\textonequarter{}",
    u"\u00BD": br"\textonehalf{}",
    u"\u00BE": br"\textthreequarters{}",
    u"\u00BF": br"\textquestiondown{}",
    u"\u00C0": br"\`{A}",
    u"\u00C1": br"\'{A}",
    u"\u00C2": br"\^{A}",
    u"\u00C3": br"\~{A}",
    u"\u00C4": br'\"{A}', # note single-quoted string
    u"\u00C5": br"\AA{}",
    u"\u00C6": br"\AE{}",
    u"\u00C7": br"\c{C}",
    u"\u00C8": br"\`{E}",
    u"\u00C9": br"\'{E}",
    u"\u00CA": br"\^{E}",
    u"\u00CB": br'\"{E}', # note single-quoted string
    u"\u00CC": br"\`{I}",
    u"\u00CD": br"\'{I}",
    u"\u00CE": br"\^{I}",
    u"\u00CF": br'\"{I}', # note single-quoted string
    u"\u00D0": br"\DH{}",
    u"\u00D1": br"\~{N}",
    u"\u00D2": br"\`{O}",
    u"\u00D3": br"\'{O}",
    u"\u00D4": br"\^{O}",
    u"\u00D5": br"\~{O}",
    u"\u00D6": br'\"{O}', # note single-quoted string
    u"\u00D7": br"\texttimes{}",
    u"\u00D8": br"\O{}",
    u"\u00D9": br"\`{U}",
    u"\u00DA": br"\'{U}",
    u"\u00DB": br"\^{U}",
    u"\u00DC": br'\"{U}', # note single-quoted string
    u"\u00DD": br"\'{Y}",
    u"\u00DE": br"\TH{}",
    u"\u00DF": br"\ss{}",
    u"\u00E0": br"\`{a}",
    u"\u00E1": br"\'{a}",
    u"\u00E2": br"\^{a}",
    u"\u00E3": br"\~{a}",
    u"\u00E4": br'\"{a}', # note single-quoted string
    u"\u00E5": br"\aa{}",
    u"\u00E6": br"\ae{}",
    u"\u00E7": br"\c{c}",
    u"\u00E8": br"\`{e}",
    u"\u00E9": br"\'{e}",
    u"\u00EA": br"\^{e}",
    u"\u00EB": br'\"{e}', # note single-quoted string
    u"\u00EC": br"\`{\i}",
    u"\u00ED": br"\'{\i}",
    u"\u00EE": br"\^{\i}",
    u"\u00EF": br'\"{\i}', # note single-quoted string
    u"\u00F0": br"\dh{}",
    u"\u00F1": br"\~{n}",
    u"\u00F2": br"\`{o}",
    u"\u00F3": br"\'{o}",
    u"\u00F4": br"\^{o}",
    u"\u00F5": br"\~{o}",
    u"\u00F6": br'\"{o}', # note single-quoted string
    u"\u00F7": br"\div{}",
    u"\u00F8": br"\o{}",
    u"\u00F9": br"\`{u}",
    u"\u00FA": br"\'{u}",
    u"\u00FB": br"\^{u}",
    u"\u00FC": br'\"{u}', # note single-quoted string
    u"\u00FD": br"\'{y}",
    u"\u00FE": br"\th{}",
    u"\u00FF": br'\"{y}', # note single-quoted string
    u"\u0100": br"\={A}",
    u"\u0101": br"\={a}",
    u"\u0102": br"\u{A}",
    u"\u0103": br"\u{a}",
    u"\u0104": br"\k{A}",
    u"\u0105": br"\k{a}",
    u"\u0106": br"\'{C}",
    u"\u0107": br"\'{c}",
    u"\u0108": br"\^{C}",
    u"\u0109": br"\^{c}",
    u"\u010A": br"\.{C}",
    u"\u010B": br"\.{c}",
    u"\u010C": br"\v{C}",
    u"\u010D": br"\v{c}",
    u"\u010E": br"\v{D}",
    u"\u010F": br"\v{d}",
    u"\u0110": br"\DJ{}",
    u"\u0111": br"\dj{}",
    u"\u0112": br"\={E}",
    u"\u0113": br"\={e}",
    u"\u0114": br"\u{E}",
    u"\u0115": br"\u{e}",
    u"\u0116": br"\.{E}",
    u"\u0117": br"\.{e}",
    u"\u0118": br"\k{E}",
    u"\u0119": br"\k{e}",
    u"\u011A": br"\v{E}",
    u"\u011B": br"\v{e}",
    u"\u011C": br"\^{G}",
    u"\u011D": br"\^{g}",
    u"\u011E": br"\u{G}",
    u"\u011F": br"\u{g}",
    u"\u0120": br"\.{G}",
    u"\u0121": br"\.{g}",
    u"\u0122": br"\c{G}",
    u"\u0123": br"\c{g}",
    u"\u0124": br"\^{H}",
    u"\u0125": br"\^{h}",
    u"\u0126": br"{\fontencoding{LELA}\selectfont\char40}",
    u"\u0127": br"\Elzxh{}",
    u"\u0128": br"\~{I}",
    u"\u0129": br"\~{\i}",
    u"\u012A": br"\={I}",
    u"\u012B": br"\={\i}",
    u"\u012C": br"\u{I}",
    u"\u012D": br"\u{\i}",
    u"\u012E": br"\k{I}",
    u"\u012F": br"\k{i}",
    u"\u0130": br"\.{I}",
    u"\u0131": br"\i{}",
    u"\u0132": br"IJ",
    u"\u0133": br"ij",
    u"\u0134": br"\^{J}",
    u"\u0135": br"\^{\j}",
    u"\u0136": br"\c{K}",
    u"\u0137": br"\c{k}",
    u"\u0138": br"{\fontencoding{LELA}\selectfont\char91}",
    u"\u0139": br"\'{L}",
    u"\u013A": br"\'{l}",
    u"\u013B": br"\c{L}",
    u"\u013C": br"\c{l}",
    u"\u013D": br"\v{L}",
    u"\u013E": br"\v{l}",
    u"\u013F": br"{\fontencoding{LELA}\selectfont\char201}",
    u"\u0140": br"{\fontencoding{LELA}\selectfont\char202}",
    u"\u0141": br"\L{}",
    u"\u0142": br"\l{}",
    u"\u0143": br"\'{N}",
    u"\u0144": br"\'{n}",
    u"\u0145": br"\c{N}",
    u"\u0146": br"\c{n}",
    u"\u0147": br"\v{N}",
    u"\u0148": br"\v{n}",
    u"\u0149": br"'n",
    u"\u014A": br"\NG{}",
    u"\u014B": br"\ng{}",
    u"\u014C": br"\={O}",
    u"\u014D": br"\={o}",
    u"\u014E": br"\u{O}",
    u"\u014F": br"\u{o}",
    u"\u0150": br"\H{O}",
    u"\u0151": br"\H{o}",
    u"\u0152": br"\OE{}",
    u"\u0153": br"\oe{}",
    u"\u0154": br"\'{R}",
    u"\u0155": br"\'{r}",
    u"\u0156": br"\c{R}",
    u"\u0157": br"\c{r}",
    u"\u0158": br"\v{R}",
    u"\u0159": br"\v{r}",
    u"\u015A": br"\'{S}",
    u"\u015B": br"\'{s}",
    u"\u015C": br"\^{S}",
    u"\u015D": br"\^{s}",
    u"\u015E": br"\c{S}",
    u"\u015F": br"\c{s}",
    u"\u0160": br"\v{S}",
    u"\u0161": br"\v{s}",
    u"\u0162": br"\c{T}",
    u"\u0163": br"\c{t}",
    u"\u0164": br"\v{T}",
    u"\u0165": br"\v{t}",
    u"\u0166": br"{\fontencoding{LELA}\selectfont\char47}",
    u"\u0167": br"{\fontencoding{LELA}\selectfont\char63}",
    u"\u0168": br"\~{U}",
    u"\u0169": br"\~{u}",
    u"\u016A": br"\={U}",
    u"\u016B": br"\={u}",
    u"\u016C": br"\u{U}",
    u"\u016D": br"\u{u}",
    u"\u016E": br"\r{U}",
    u"\u016F": br"\r{u}",
    u"\u0170": br"\H{U}",
    u"\u0171": br"\H{u}",
    u"\u0172": br"\k{U}",
    u"\u0173": br"\k{u}",
    u"\u0174": br"\^{W}",
    u"\u0175": br"\^{w}",
    u"\u0176": br"\^{Y}",
    u"\u0177": br"\^{y}",
    u"\u0178": br'\"{Y}', # note single-quoted string
    u"\u0179": br"\'{Z}",
    u"\u017A": br"\'{z}",
    u"\u017B": br"\.{Z}",
    u"\u017C": br"\.{z}",
    u"\u017D": br"\v{Z}",
    u"\u017E": br"\v{z}",
    u"\u0195": br"\texthvlig{}",
    u"\u019E": br"\textnrleg{}",
    u"\u01AA": br"\eth{}",
    u"\u01BA": br"{\fontencoding{LELA}\selectfont\char195}",
    u"\u01C2": br"\textdoublepipe{}",
    u"\u01F5": br"\'{g}",
    u"\u0250": br"\Elztrna{}",
    u"\u0252": br"\Elztrnsa{}",
    u"\u0254": br"\Elzopeno{}",
    u"\u0256": br"\Elzrtld{}",
    u"\u0258": br"{\fontencoding{LEIP}\selectfont\char61}",
    u"\u0259": br"\Elzschwa{}",
    u"\u025B": br"\varepsilon{}",
    u"\u0263": br"\Elzpgamma{}",
    u"\u0264": br"\Elzpbgam{}",
    u"\u0265": br"\Elztrnh{}",
    u"\u026C": br"\Elzbtdl{}",
    u"\u026D": br"\Elzrtll{}",
    u"\u026F": br"\Elztrnm{}",
    u"\u0270": br"\Elztrnmlr{}",
    u"\u0271": br"\Elzltlmr{}",
    u"\u0272": br"\Elzltln{}",
    u"\u0273": br"\Elzrtln{}",
    u"\u0277": br"\Elzclomeg{}",
    u"\u0278": br"\textphi{}",
    u"\u0279": br"\Elztrnr{}",
    u"\u027A": br"\Elztrnrl{}",
    u"\u027B": br"\Elzrttrnr{}",
    u"\u027C": br"\Elzrl{}",
    u"\u027D": br"\Elzrtlr{}",
    u"\u027E": br"\Elzfhr{}",
    u"\u027F": br"{\fontencoding{LEIP}\selectfont\char202}",
    u"\u0282": br"\Elzrtls{}",
    u"\u0283": br"\Elzesh{}",
    u"\u0287": br"\Elztrnt{}",
    u"\u0288": br"\Elzrtlt{}",
    u"\u028A": br"\Elzpupsil{}",
    u"\u028B": br"\Elzpscrv{}",
    u"\u028C": br"\Elzinvv{}",
    u"\u028D": br"\Elzinvw{}",
    u"\u028E": br"\Elztrny{}",
    u"\u0290": br"\Elzrtlz{}",
    u"\u0292": br"\Elzyogh{}",
    u"\u0294": br"\Elzglst{}",
    u"\u0295": br"\Elzreglst{}",
    u"\u0296": br"\Elzinglst{}",
    u"\u029E": br"\textturnk{}",
    u"\u02A4": br"\Elzdyogh{}",
    u"\u02A7": br"\Elztesh{}",
    u"\u02C7": br"\textasciicaron{}",
    u"\u02C8": br"\Elzverts{}",
    u"\u02CC": br"\Elzverti{}",
    u"\u02D0": br"\Elzlmrk{}",
    u"\u02D1": br"\Elzhlmrk{}",
    u"\u02D2": br"\Elzsbrhr{}",
    u"\u02D3": br"\Elzsblhr{}",
    u"\u02D4": br"\Elzrais{}",
    u"\u02D5": br"\Elzlow{}",
    u"\u02D8": br"\textasciibreve{}",
    u"\u02D9": br"\textperiodcentered{}",
    u"\u02DA": br"\r{}",
    u"\u02DB": br"\k{}",
    u"\u02DC": br"\texttildelow{}",
    u"\u02DD": br"\H{}",
    u"\u02E5": br"\tone{55}",
    u"\u02E6": br"\tone{44}",
    u"\u02E7": br"\tone{33}",
    u"\u02E8": br"\tone{22}",
    u"\u02E9": br"\tone{11}",
    u"\u0300": br"\`",
    u"\u0301": br"\'",
    u"\u0302": br"\^",
    u"\u0303": br"\~",
    u"\u0304": br"\=",
    u"\u0306": br"\u",
    u"\u0307": br"\.",
    u"\u0308": br'\"', # note single-quoted string
    u"\u030A": br"\r",
    u"\u030B": br"\H",
    u"\u030C": br"\v",
    u"\u030F": br"\cyrchar\C",
    u"\u0311": br"{\fontencoding{LECO}\selectfont\char177}",
    u"\u0318": br"{\fontencoding{LECO}\selectfont\char184}",
    u"\u0319": br"{\fontencoding{LECO}\selectfont\char185}",
    u"\u0321": br"\Elzpalh{}",
    u"\u0322": br"\Elzrh{}",
    u"\u0327": br"\c",
    u"\u0328": br"\k",
    u"\u032A": br"\Elzsbbrg{}",
    u"\u032B": br"{\fontencoding{LECO}\selectfont\char203}",
    u"\u032F": br"{\fontencoding{LECO}\selectfont\char207}",
    u"\u0335": br"\Elzxl{}",
    u"\u0336": br"\Elzbar{}",
    u"\u0337": br"{\fontencoding{LECO}\selectfont\char215}",
    u"\u0338": br"{\fontencoding{LECO}\selectfont\char216}",
    u"\u033A": br"{\fontencoding{LECO}\selectfont\char218}",
    u"\u033B": br"{\fontencoding{LECO}\selectfont\char219}",
    u"\u033C": br"{\fontencoding{LECO}\selectfont\char220}",
    u"\u033D": br"{\fontencoding{LECO}\selectfont\char221}",
    u"\u0361": br"{\fontencoding{LECO}\selectfont\char225}",
    u"\u0386": br"\'{A}",
    u"\u0388": br"\'{E}",
    u"\u0389": br"\'{H}",
    u"\u038A": br"\'{}{I}",
    u"\u038C": br"\'{}O",
    u"\u038E": br"\mathrm{'Y}",
    u"\u038F": br"\mathrm{'\Omega}",
    u"\u0390": br"\acute{\ddot{\iota}}",
    u"\u0391": br"\ensuremath{\Alpha}",
    u"\u0392": br"\ensuremath{\Beta}",
    u"\u0393": br"\ensuremath{\Gamma}",
    u"\u0394": br"\ensuremath{\Delta}",
    u"\u0395": br"\ensuremath{\Epsilon}",
    u"\u0396": br"\ensuremath{\Zeta}",
    u"\u0397": br"\ensuremath{\Eta}",
    u"\u0398": br"\ensuremath{\Theta}",
    u"\u0399": br"\ensuremath{\Iota}",
    u"\u039A": br"\ensuremath{\Kappa}",
    u"\u039B": br"\ensuremath{\Lambda}",
    u"\u039E": br"\ensuremath{\Xi}",
    u"\u03A0": br"\ensuremath{\Pi}",
    u"\u03A1": br"\ensuremath{\Rho}",
    u"\u03A3": br"\ensuremath{\Sigma}",
    u"\u03A4": br"\ensuremath{\Tau}",
    u"\u03A5": br"\ensuremath{\Upsilon}",
    u"\u03A6": br"\ensuremath{\Phi}",
    u"\u03A7": br"\ensuremath{\Chi}",
    u"\u03A8": br"\ensuremath{\Psi}",
    u"\u03A9": br"\ensuremath{\Omega}",
    u"\u03AA": br"\mathrm{\ddot{I}}",
    u"\u03AB": br"\mathrm{\ddot{Y}}",
    u"\u03AC": br"\'{$\alpha$}",
    u"\u03AD": br"\acute{\epsilon}",
    u"\u03AE": br"\acute{\eta}",
    u"\u03AF": br"\acute{\iota}",
    u"\u03B0": br"\acute{\ddot{\upsilon}}",
    u"\u03B1": br"\ensuremath{\alpha}",
    u"\u03B2": br"\ensuremath{\beta}",
    u"\u03B3": br"\ensuremath{\gamma}",
    u"\u03B4": br"\ensuremath{\delta}",
    u"\u03B5": br"\ensuremath{\epsilon}",
    u"\u03B6": br"\ensuremath{\zeta}",
    u"\u03B7": br"\ensuremath{\eta}",
    u"\u03B8": br"\texttheta{}",
    u"\u03B9": br"\ensuremath{\iota}",
    u"\u03BA": br"\ensuremath{\kappa}",
    u"\u03BB": br"\ensuremath{\lambda}",
    u"\u03BC": br"\ensuremath{\mu}",
    u"\u03BD": br"\ensuremath{\nu}",
    u"\u03BE": br"\ensuremath{\xi}",
    u"\u03C0": br"\ensuremath{\pi}",
    u"\u03C1": br"\ensuremath{\rho}",
    u"\u03C2": br"\ensuremath{\varsigma}",
    u"\u03C3": br"\ensuremath{\sigma}",
    u"\u03C4": br"\ensuremath{\tau}",
    u"\u03C5": br"\ensuremath{\upsilon}",
    u"\u03C6": br"\ensuremath{\varphi}",
    u"\u03C7": br"\ensuremath{\chi}",
    u"\u03C8": br"\ensuremath{\psi}",
    u"\u03C9": br"\ensuremath{\omega}",
    u"\u03CA": br"\ensuremath{\ddot{\iota}}",
    u"\u03CB": br"\ensuremath{\ddot{\upsilon}}",
    u"\u03CC": br"\'{o}",
    u"\u03CD": br"\acute{\upsilon}",
    u"\u03CE": br"\acute{\omega}",
    u"\u03D0": br"\Pisymbol{ppi022}{87}",
    u"\u03D1": br"\textvartheta{}",
    u"\u03D2": br"\ensuremath{\Upsilon}",
    u"\u03D5": br"\ensuremath{\phi}",
    u"\u03D6": br"\ensuremath{\varpi}",
    u"\u03DA": br"\Stigma{}",
    u"\u03DC": br"\Digamma{}",
    u"\u03DD": br"\digamma{}",
    u"\u03DE": br"\Koppa{}",
    u"\u03E0": br"\Sampi{}",
    u"\u03F0": br"\varkappa{}",
    u"\u03F1": br"\varrho{}",
    u"\u03F4": br"\textTheta{}",
    u"\u03F6": br"\backepsilon{}",
    u"\u0401": br"\cyrchar\CYRYO{}",
    u"\u0402": br"\cyrchar\CYRDJE{}",
    u"\u0403": br"\cyrchar{\'\CYRG}",
    u"\u0404": br"\cyrchar\CYRIE{}",
    u"\u0405": br"\cyrchar\CYRDZE{}",
    u"\u0406": br"\cyrchar\CYRII{}",
    u"\u0407": br"\cyrchar\CYRYI{}",
    u"\u0408": br"\cyrchar\CYRJE{}",
    u"\u0409": br"\cyrchar\CYRLJE{}",
    u"\u040A": br"\cyrchar\CYRNJE{}",
    u"\u040B": br"\cyrchar\CYRTSHE{}",
    u"\u040C": br"\cyrchar{\'\CYRK}",
    u"\u040E": br"\cyrchar\CYRUSHRT{}",
    u"\u040F": br"\cyrchar\CYRDZHE{}",
    u"\u0410": br"\cyrchar\CYRA{}",
    u"\u0411": br"\cyrchar\CYRB{}",
    u"\u0412": br"\cyrchar\CYRV{}",
    u"\u0413": br"\cyrchar\CYRG{}",
    u"\u0414": br"\cyrchar\CYRD{}",
    u"\u0415": br"\cyrchar\CYRE{}",
    u"\u0416": br"\cyrchar\CYRZH{}",
    u"\u0417": br"\cyrchar\CYRZ{}",
    u"\u0418": br"\cyrchar\CYRI{}",
    u"\u0419": br"\cyrchar\CYRISHRT{}",
    u"\u041A": br"\cyrchar\CYRK{}",
    u"\u041B": br"\cyrchar\CYRL{}",
    u"\u041C": br"\cyrchar\CYRM{}",
    u"\u041D": br"\cyrchar\CYRN{}",
    u"\u041E": br"\cyrchar\CYRO{}",
    u"\u041F": br"\cyrchar\CYRP{}",
    u"\u0420": br"\cyrchar\CYRR{}",
    u"\u0421": br"\cyrchar\CYRS{}",
    u"\u0422": br"\cyrchar\CYRT{}",
    u"\u0423": br"\cyrchar\CYRU{}",
    u"\u0424": br"\cyrchar\CYRF{}",
    u"\u0425": br"\cyrchar\CYRH{}",
    u"\u0426": br"\cyrchar\CYRC{}",
    u"\u0427": br"\cyrchar\CYRCH{}",
    u"\u0428": br"\cyrchar\CYRSH{}",
    u"\u0429": br"\cyrchar\CYRSHCH{}",
    u"\u042A": br"\cyrchar\CYRHRDSN{}",
    u"\u042B": br"\cyrchar\CYRERY{}",
    u"\u042C": br"\cyrchar\CYRSFTSN{}",
    u"\u042D": br"\cyrchar\CYREREV{}",
    u"\u042E": br"\cyrchar\CYRYU{}",
    u"\u042F": br"\cyrchar\CYRYA{}",
    u"\u0430": br"\cyrchar\cyra{}",
    u"\u0431": br"\cyrchar\cyrb{}",
    u"\u0432": br"\cyrchar\cyrv{}",
    u"\u0433": br"\cyrchar\cyrg{}",
    u"\u0434": br"\cyrchar\cyrd{}",
    u"\u0435": br"\cyrchar\cyre{}",
    u"\u0436": br"\cyrchar\cyrzh{}",
    u"\u0437": br"\cyrchar\cyrz{}",
    u"\u0438": br"\cyrchar\cyri{}",
    u"\u0439": br"\cyrchar\cyrishrt{}",
    u"\u043A": br"\cyrchar\cyrk{}",
    u"\u043B": br"\cyrchar\cyrl{}",
    u"\u043C": br"\cyrchar\cyrm{}",
    u"\u043D": br"\cyrchar\cyrn{}",
    u"\u043E": br"\cyrchar\cyro{}",
    u"\u043F": br"\cyrchar\cyrp{}",
    u"\u0440": br"\cyrchar\cyrr{}",
    u"\u0441": br"\cyrchar\cyrs{}",
    u"\u0442": br"\cyrchar\cyrt{}",
    u"\u0443": br"\cyrchar\cyru{}",
    u"\u0444": br"\cyrchar\cyrf{}",
    u"\u0445": br"\cyrchar\cyrh{}",
    u"\u0446": br"\cyrchar\cyrc{}",
    u"\u0447": br"\cyrchar\cyrch{}",
    u"\u0448": br"\cyrchar\cyrsh{}",
    u"\u0449": br"\cyrchar\cyrshch{}",
    u"\u044A": br"\cyrchar\cyrhrdsn{}",
    u"\u044B": br"\cyrchar\cyrery{}",
    u"\u044C": br"\cyrchar\cyrsftsn{}",
    u"\u044D": br"\cyrchar\cyrerev{}",
    u"\u044E": br"\cyrchar\cyryu{}",
    u"\u044F": br"\cyrchar\cyrya{}",
    u"\u0451": br"\cyrchar\cyryo{}",
    u"\u0452": br"\cyrchar\cyrdje{}",
    u"\u0453": br"\cyrchar{\'\cyrg}",
    u"\u0454": br"\cyrchar\cyrie{}",
    u"\u0455": br"\cyrchar\cyrdze{}",
    u"\u0456": br"\cyrchar\cyrii{}",
    u"\u0457": br"\cyrchar\cyryi{}",
    u"\u0458": br"\cyrchar\cyrje{}",
    u"\u0459": br"\cyrchar\cyrlje{}",
    u"\u045A": br"\cyrchar\cyrnje{}",
    u"\u045B": br"\cyrchar\cyrtshe{}",
    u"\u045C": br"\cyrchar{\'\cyrk}",
    u"\u045E": br"\cyrchar\cyrushrt{}",
    u"\u045F": br"\cyrchar\cyrdzhe{}",
    u"\u0460": br"\cyrchar\CYROMEGA{}",
    u"\u0461": br"\cyrchar\cyromega{}",
    u"\u0462": br"\cyrchar\CYRYAT{}",
    u"\u0464": br"\cyrchar\CYRIOTE{}",
    u"\u0465": br"\cyrchar\cyriote{}",
    u"\u0466": br"\cyrchar\CYRLYUS{}",
    u"\u0467": br"\cyrchar\cyrlyus{}",
    u"\u0468": br"\cyrchar\CYRIOTLYUS{}",
    u"\u0469": br"\cyrchar\cyriotlyus{}",
    u"\u046A": br"\cyrchar\CYRBYUS{}",
    u"\u046C": br"\cyrchar\CYRIOTBYUS{}",
    u"\u046D": br"\cyrchar\cyriotbyus{}",
    u"\u046E": br"\cyrchar\CYRKSI{}",
    u"\u046F": br"\cyrchar\cyrksi{}",
    u"\u0470": br"\cyrchar\CYRPSI{}",
    u"\u0471": br"\cyrchar\cyrpsi{}",
    u"\u0472": br"\cyrchar\CYRFITA{}",
    u"\u0474": br"\cyrchar\CYRIZH{}",
    u"\u0478": br"\cyrchar\CYRUK{}",
    u"\u0479": br"\cyrchar\cyruk{}",
    u"\u047A": br"\cyrchar\CYROMEGARND{}",
    u"\u047B": br"\cyrchar\cyromegarnd{}",
    u"\u047C": br"\cyrchar\CYROMEGATITLO{}",
    u"\u047D": br"\cyrchar\cyromegatitlo{}",
    u"\u047E": br"\cyrchar\CYROT{}",
    u"\u047F": br"\cyrchar\cyrot{}",
    u"\u0480": br"\cyrchar\CYRKOPPA{}",
    u"\u0481": br"\cyrchar\cyrkoppa{}",
    u"\u0482": br"\cyrchar\cyrthousands{}",
    u"\u0488": br"\cyrchar\cyrhundredthousands{}",
    u"\u0489": br"\cyrchar\cyrmillions{}",
    u"\u048C": br"\cyrchar\CYRSEMISFTSN{}",
    u"\u048D": br"\cyrchar\cyrsemisftsn{}",
    u"\u048E": br"\cyrchar\CYRRTICK{}",
    u"\u048F": br"\cyrchar\cyrrtick{}",
    u"\u0490": br"\cyrchar\CYRGUP{}",
    u"\u0491": br"\cyrchar\cyrgup{}",
    u"\u0492": br"\cyrchar\CYRGHCRS{}",
    u"\u0493": br"\cyrchar\cyrghcrs{}",
    u"\u0494": br"\cyrchar\CYRGHK{}",
    u"\u0495": br"\cyrchar\cyrghk{}",
    u"\u0496": br"\cyrchar\CYRZHDSC{}",
    u"\u0497": br"\cyrchar\cyrzhdsc{}",
    u"\u0498": br"\cyrchar\CYRZDSC{}",
    u"\u0499": br"\cyrchar\cyrzdsc{}",
    u"\u049A": br"\cyrchar\CYRKDSC{}",
    u"\u049B": br"\cyrchar\cyrkdsc{}",
    u"\u049C": br"\cyrchar\CYRKVCRS{}",
    u"\u049D": br"\cyrchar\cyrkvcrs{}",
    u"\u049E": br"\cyrchar\CYRKHCRS{}",
    u"\u049F": br"\cyrchar\cyrkhcrs{}",
    u"\u04A0": br"\cyrchar\CYRKBEAK{}",
    u"\u04A1": br"\cyrchar\cyrkbeak{}",
    u"\u04A2": br"\cyrchar\CYRNDSC{}",
    u"\u04A3": br"\cyrchar\cyrndsc{}",
    u"\u04A4": br"\cyrchar\CYRNG{}",
    u"\u04A5": br"\cyrchar\cyrng{}",
    u"\u04A6": br"\cyrchar\CYRPHK{}",
    u"\u04A7": br"\cyrchar\cyrphk{}",
    u"\u04A8": br"\cyrchar\CYRABHHA{}",
    u"\u04A9": br"\cyrchar\cyrabhha{}",
    u"\u04AA": br"\cyrchar\CYRSDSC{}",
    u"\u04AB": br"\cyrchar\cyrsdsc{}",
    u"\u04AC": br"\cyrchar\CYRTDSC{}",
    u"\u04AD": br"\cyrchar\cyrtdsc{}",
    u"\u04AE": br"\cyrchar\CYRY{}",
    u"\u04AF": br"\cyrchar\cyry{}",
    u"\u04B0": br"\cyrchar\CYRYHCRS{}",
    u"\u04B1": br"\cyrchar\cyryhcrs{}",
    u"\u04B2": br"\cyrchar\CYRHDSC{}",
    u"\u04B3": br"\cyrchar\cyrhdsc{}",
    u"\u04B4": br"\cyrchar\CYRTETSE{}",
    u"\u04B5": br"\cyrchar\cyrtetse{}",
    u"\u04B6": br"\cyrchar\CYRCHRDSC{}",
    u"\u04B7": br"\cyrchar\cyrchrdsc{}",
    u"\u04B8": br"\cyrchar\CYRCHVCRS{}",
    u"\u04B9": br"\cyrchar\cyrchvcrs{}",
    u"\u04BA": br"\cyrchar\CYRSHHA{}",
    u"\u04BB": br"\cyrchar\cyrshha{}",
    u"\u04BC": br"\cyrchar\CYRABHCH{}",
    u"\u04BD": br"\cyrchar\cyrabhch{}",
    u"\u04BE": br"\cyrchar\CYRABHCHDSC{}",
    u"\u04BF": br"\cyrchar\cyrabhchdsc{}",
    u"\u04C0": br"\cyrchar\CYRpalochka{}",
    u"\u04C3": br"\cyrchar\CYRKHK{}",
    u"\u04C4": br"\cyrchar\cyrkhk{}",
    u"\u04C7": br"\cyrchar\CYRNHK{}",
    u"\u04C8": br"\cyrchar\cyrnhk{}",
    u"\u04CB": br"\cyrchar\CYRCHLDSC{}",
    u"\u04CC": br"\cyrchar\cyrchldsc{}",
    u"\u04D4": br"\cyrchar\CYRAE{}",
    u"\u04D5": br"\cyrchar\cyrae{}",
    u"\u04D8": br"\cyrchar\CYRSCHWA{}",
    u"\u04D9": br"\cyrchar\cyrschwa{}",
    u"\u04E0": br"\cyrchar\CYRABHDZE{}",
    u"\u04E1": br"\cyrchar\cyrabhdze{}",
    u"\u04E8": br"\cyrchar\CYROTLD{}",
    u"\u04E9": br"\cyrchar\cyrotld{}",
    u"\u2002": br"\hspace{0.6em}",
    u"\u2003": br"\hspace{1em}",
    u"\u2004": br"\hspace{0.33em}",
    u"\u2005": br"\hspace{0.25em}",
    u"\u2006": br"\hspace{0.166em}",
    u"\u2007": br"\hphantom{0}",
    u"\u2008": br"\hphantom{,}",
    u"\u2009": br"\hspace{0.167em}",
    u"\u200A": br"\mkern1mu{}",
    u"\u2010": br"-",
    u"\u2013": br"\textendash{}",
    u"\u2014": br"\textemdash{}",
    u"\u2015": br"\rule{1em}{1pt}",
    u"\u2016": br"\Vert{}",
    u"\u2018": br"`",
    u"\u2019": br"'",
    u"\u201B": br"\Elzreapos{}",
    u"\u201C": br"\textquotedblleft{}",
    u"\u201D": br"\textquotedblright{}",
    u"\u201E": br",,",
    u"\u2020": br"\textdagger{}",
    u"\u2021": br"\textdaggerdbl{}",
    u"\u2022": br"\textbullet{}",
    u"\u2025": br"..",
    u"\u2026": br"\ldots{}",
    u"\u2030": br"\textperthousand{}",
    u"\u2031": br"\textpertenthousand{}",
    u"\u2032": br"\ensuremath{'}",
    u"\u2033": br"\ensuremath{''}",
    u"\u2034": br"\ensuremath{'''}",
    u"\u2035": br"\backprime{}",
    u"\u2039": br"\guilsinglleft{}",
    u"\u203A": br"\guilsinglright{}",
    u"\u2057": br"''''",
    u"\u205F": br"\mkern4mu{}",
    u"\u2060": br"\nolinebreak{}",
    u"\u20A7": br"\ensuremath{\Elzpes}",
    u"\u20AC": br"\mbox{\texteuro}{}",
    u"\u20DB": br"\dddot{}",
    u"\u20DC": br"\ddddot{}",
    u"\u2102": br"\mathbb{C}",
    u"\u210A": br"\mathscr{g}",
    u"\u210B": br"\mathscr{H}",
    u"\u210C": br"\mathfrak{H}",
    u"\u210D": br"\mathbb{H}",
    u"\u210F": br"\hslash{}",
    u"\u2110": br"\mathscr{I}",
    u"\u2111": br"\mathfrak{I}",
    u"\u2112": br"\mathscr{L}",
    u"\u2113": br"\mathscr{l}",
    u"\u2115": br"\mathbb{N}",
    u"\u2116": br"\cyrchar\textnumero{}",
    u"\u2118": br"\wp{}",
    u"\u2119": br"\mathbb{P}",
    u"\u211A": br"\mathbb{Q}",
    u"\u211B": br"\mathscr{R}",
    u"\u211C": br"\mathfrak{R}",
    u"\u211D": br"\mathbb{R}",
    u"\u211E": br"\Elzxrat{}",
    u"\u2122": br"\texttrademark{}",
    u"\u2124": br"\mathbb{Z}",
    u"\u2126": br"\Omega{}",
    u"\u2127": br"\mho{}",
    u"\u2128": br"\mathfrak{Z}",
    u"\u2129": br"\ElsevierGlyph{2129}",
    u"\u212B": br"\AA{}",
    u"\u212C": br"\mathscr{B}",
    u"\u212D": br"\mathfrak{C}",
    u"\u212F": br"\mathscr{e}",
    u"\u2130": br"\mathscr{E}",
    u"\u2131": br"\mathscr{F}",
    u"\u2133": br"\mathscr{M}",
    u"\u2134": br"\mathscr{o}",
    u"\u2135": br"\aleph{}",
    u"\u2136": br"\beth{}",
    u"\u2137": br"\gimel{}",
    u"\u2138": br"\daleth{}",
    u"\u2153": br"\textfrac{1}{3}",
    u"\u2154": br"\textfrac{2}{3}",
    u"\u2155": br"\textfrac{1}{5}",
    u"\u2156": br"\textfrac{2}{5}",
    u"\u2157": br"\textfrac{3}{5}",
    u"\u2158": br"\textfrac{4}{5}",
    u"\u2159": br"\textfrac{1}{6}",
    u"\u215A": br"\textfrac{5}{6}",
    u"\u215B": br"\textfrac{1}{8}",
    u"\u215C": br"\textfrac{3}{8}",
    u"\u215D": br"\textfrac{5}{8}",
    u"\u215E": br"\textfrac{7}{8}",
    u"\u2190": br"\leftarrow{}",
    u"\u2191": br"\uparrow{}",
    u"\u2192": br"\rightarrow{}",
    u"\u2193": br"\downarrow{}",
    u"\u2194": br"\leftrightarrow{}",
    u"\u2195": br"\updownarrow{}",
    u"\u2196": br"\nwarrow{}",
    u"\u2197": br"\nearrow{}",
    u"\u2198": br"\searrow{}",
    u"\u2199": br"\swarrow{}",
    u"\u219A": br"\nleftarrow{}",
    u"\u219B": br"\nrightarrow{}",
    u"\u219C": br"\arrowwaveright{}",
    u"\u219D": br"\arrowwaveright{}",
    u"\u219E": br"\twoheadleftarrow{}",
    u"\u21A0": br"\twoheadrightarrow{}",
    u"\u21A2": br"\leftarrowtail{}",
    u"\u21A3": br"\rightarrowtail{}",
    u"\u21A6": br"\mapsto{}",
    u"\u21A9": br"\hookleftarrow{}",
    u"\u21AA": br"\hookrightarrow{}",
    u"\u21AB": br"\looparrowleft{}",
    u"\u21AC": br"\looparrowright{}",
    u"\u21AD": br"\leftrightsquigarrow{}",
    u"\u21AE": br"\nleftrightarrow{}",
    u"\u21B0": br"\Lsh{}",
    u"\u21B1": br"\Rsh{}",
    u"\u21B3": br"\ElsevierGlyph{21B3}",
    u"\u21B6": br"\curvearrowleft{}",
    u"\u21B7": br"\curvearrowright{}",
    u"\u21BA": br"\circlearrowleft{}",
    u"\u21BB": br"\circlearrowright{}",
    u"\u21BC": br"\leftharpoonup{}",
    u"\u21BD": br"\leftharpoondown{}",
    u"\u21BE": br"\upharpoonright{}",
    u"\u21BF": br"\upharpoonleft{}",
    u"\u21C0": br"\rightharpoonup{}",
    u"\u21C1": br"\rightharpoondown{}",
    u"\u21C2": br"\downharpoonright{}",
    u"\u21C3": br"\downharpoonleft{}",
    u"\u21C4": br"\rightleftarrows{}",
    u"\u21C5": br"\dblarrowupdown{}",
    u"\u21C6": br"\leftrightarrows{}",
    u"\u21C7": br"\leftleftarrows{}",
    u"\u21C8": br"\upuparrows{}",
    u"\u21C9": br"\rightrightarrows{}",
    u"\u21CA": br"\downdownarrows{}",
    u"\u21CB": br"\leftrightharpoons{}",
    u"\u21CC": br"\rightleftharpoons{}",
    u"\u21CD": br"\nLeftarrow{}",
    u"\u21CE": br"\nLeftrightarrow{}",
    u"\u21CF": br"\nRightarrow{}",
    u"\u21D0": br"\Leftarrow{}",
    u"\u21D1": br"\Uparrow{}",
    u"\u21D2": br"\Rightarrow{}",
    u"\u21D3": br"\Downarrow{}",
    u"\u21D4": br"\Leftrightarrow{}",
    u"\u21D5": br"\Updownarrow{}",
    u"\u21DA": br"\Lleftarrow{}",
    u"\u21DB": br"\Rrightarrow{}",
    u"\u21DD": br"\rightsquigarrow{}",
    u"\u21F5": br"\DownArrowUpArrow{}",
    u"\u2200": br"\forall{}",
    u"\u2201": br"\complement{}",
    u"\u2202": br"\partial{}",
    u"\u2203": br"\exists{}",
    u"\u2204": br"\nexists{}",
    u"\u2205": br"\varnothing{}",
    u"\u2207": br"\nabla{}",
    u"\u2208": br"\in{}",
    u"\u2209": br"\not\in{}",
    u"\u220B": br"\ni{}",
    u"\u220C": br"\not\ni{}",
    u"\u220F": br"\prod{}",
    u"\u2210": br"\coprod{}",
    u"\u2211": br"\sum{}",
    u"\u2213": br"\mp{}",
    u"\u2214": br"\dotplus{}",
    u"\u2216": br"\setminus{}",
    u"\u2217": br"{_\ast}",
    u"\u2218": br"\circ{}",
    u"\u2219": br"\bullet{}",
    u"\u221A": br"\surd{}",
    u"\u221D": br"\propto{}",
    u"\u221E": br"\infty{}",
    u"\u221F": br"\rightangle{}",
    u"\u2220": br"\angle{}",
    u"\u2221": br"\measuredangle{}",
    u"\u2222": br"\sphericalangle{}",
    u"\u2223": br"\mid{}",
    u"\u2224": br"\nmid{}",
    u"\u2225": br"\parallel{}",
    u"\u2226": br"\nparallel{}",
    u"\u2227": br"\wedge{}",
    u"\u2228": br"\vee{}",
    u"\u2229": br"\cap{}",
    u"\u222A": br"\cup{}",
    u"\u222B": br"\int{}",
    u"\u222C": br"\int\!\int{}",
    u"\u222D": br"\int\!\int\!\int{}",
    u"\u222E": br"\oint{}",
    u"\u222F": br"\surfintegral{}",
    u"\u2230": br"\volintegral{}",
    u"\u2231": br"\clwintegral{}",
    u"\u2232": br"\ElsevierGlyph{2232}",
    u"\u2233": br"\ElsevierGlyph{2233}",
    u"\u2234": br"\therefore{}",
    u"\u2235": br"\because{}",
    u"\u2237": br"\Colon{}",
    u"\u2238": br"\ElsevierGlyph{2238}",
    u"\u223A": br"\mathbin{{:}\!\!{-}\!\!{:}}",
    u"\u223B": br"\homothetic{}",
    u"\u223C": br"\sim{}",
    u"\u223D": br"\backsim{}",
    u"\u223E": br"\lazysinv{}",
    u"\u2240": br"\wr{}",
    u"\u2241": br"\not\sim{}",
    u"\u2242": br"\ElsevierGlyph{2242}",
    u"\u2243": br"\simeq{}",
    u"\u2244": br"\not\simeq{}",
    u"\u2245": br"\cong{}",
    u"\u2246": br"\approxnotequal{}",
    u"\u2247": br"\not\cong{}",
    u"\u2248": br"\approx{}",
    u"\u2249": br"\not\approx{}",
    u"\u224A": br"\approxeq{}",
    u"\u224B": br"\tildetrpl{}",
    u"\u224C": br"\allequal{}",
    u"\u224D": br"\asymp{}",
    u"\u224E": br"\Bumpeq{}",
    u"\u224F": br"\bumpeq{}",
    u"\u2250": br"\doteq{}",
    u"\u2251": br"\doteqdot{}",
    u"\u2252": br"\fallingdotseq{}",
    u"\u2253": br"\risingdotseq{}",
    u"\u2254": br":=",
    u"\u2255": br"=:",
    u"\u2256": br"\eqcirc{}",
    u"\u2257": br"\circeq{}",
    u"\u2259": br"\estimates{}",
    u"\u225A": br"\ElsevierGlyph{225A}",
    u"\u225B": br"\starequal{}",
    u"\u225C": br"\triangleq{}",
    u"\u225F": br"\ElsevierGlyph{225F}",
    u"\u2260": br"\not =",
    u"\u2261": br"\equiv{}",
    u"\u2262": br"\not\equiv{}",
    u"\u2264": br"\leq{}",
    u"\u2265": br"\geq{}",
    u"\u2266": br"\leqq{}",
    u"\u2267": br"\geqq{}",
    u"\u2268": br"\lneqq{}",
    u"\u2269": br"\gneqq{}",
    u"\u226A": br"\ll{}",
    u"\u226B": br"\gg{}",
    u"\u226C": br"\between{}",
    u"\u226D": br"\not\kern-0.3em\times{}",
    u"\u226E": br"\not&lt;",
    u"\u226F": br"\not&gt;",
    u"\u2270": br"\not\leq{}",
    u"\u2271": br"\not\geq{}",
    u"\u2272": br"\lessequivlnt{}",
    u"\u2273": br"\greaterequivlnt{}",
    u"\u2274": br"\ElsevierGlyph{2274}",
    u"\u2275": br"\ElsevierGlyph{2275}",
    u"\u2276": br"\lessgtr{}",
    u"\u2277": br"\gtrless{}",
    u"\u2278": br"\notlessgreater{}",
    u"\u2279": br"\notgreaterless{}",
    u"\u227A": br"\prec{}",
    u"\u227B": br"\succ{}",
    u"\u227C": br"\preccurlyeq{}",
    u"\u227D": br"\succcurlyeq{}",
    u"\u227E": br"\precapprox{}",
    u"\u227F": br"\succapprox{}",
    u"\u2280": br"\not\prec{}",
    u"\u2281": br"\not\succ{}",
    u"\u2282": br"\subset{}",
    u"\u2283": br"\supset{}",
    u"\u2284": br"\not\subset{}",
    u"\u2285": br"\not\supset{}",
    u"\u2286": br"\subseteq{}",
    u"\u2287": br"\supseteq{}",
    u"\u2288": br"\not\subseteq{}",
    u"\u2289": br"\not\supseteq{}",
    u"\u228A": br"\subsetneq{}",
    u"\u228B": br"\supsetneq{}",
    u"\u228E": br"\uplus{}",
    u"\u228F": br"\sqsubset{}",
    u"\u2290": br"\sqsupset{}",
    u"\u2291": br"\sqsubseteq{}",
    u"\u2292": br"\sqsupseteq{}",
    u"\u2293": br"\sqcap{}",
    u"\u2294": br"\sqcup{}",
    u"\u2295": br"\oplus{}",
    u"\u2296": br"\ominus{}",
    u"\u2297": br"\otimes{}",
    u"\u2298": br"\oslash{}",
    u"\u2299": br"\odot{}",
    u"\u229A": br"\circledcirc{}",
    u"\u229B": br"\circledast{}",
    u"\u229D": br"\circleddash{}",
    u"\u229E": br"\boxplus{}",
    u"\u229F": br"\boxminus{}",
    u"\u22A0": br"\boxtimes{}",
    u"\u22A1": br"\boxdot{}",
    u"\u22A2": br"\vdash{}",
    u"\u22A3": br"\dashv{}",
    u"\u22A4": br"\top{}",
    u"\u22A5": br"\perp{}",
    u"\u22A7": br"\truestate{}",
    u"\u22A8": br"\forcesextra{}",
    u"\u22A9": br"\Vdash{}",
    u"\u22AA": br"\Vvdash{}",
    u"\u22AB": br"\VDash{}",
    u"\u22AC": br"\nvdash{}",
    u"\u22AD": br"\nvDash{}",
    u"\u22AE": br"\nVdash{}",
    u"\u22AF": br"\nVDash{}",
    u"\u22B2": br"\vartriangleleft{}",
    u"\u22B3": br"\vartriangleright{}",
    u"\u22B4": br"\trianglelefteq{}",
    u"\u22B5": br"\trianglerighteq{}",
    u"\u22B6": br"\original{}",
    u"\u22B7": br"\image{}",
    u"\u22B8": br"\multimap{}",
    u"\u22B9": br"\hermitconjmatrix{}",
    u"\u22BA": br"\intercal{}",
    u"\u22BB": br"\veebar{}",
    u"\u22BE": br"\rightanglearc{}",
    u"\u22C0": br"\ElsevierGlyph{22C0}",
    u"\u22C1": br"\ElsevierGlyph{22C1}",
    u"\u22C2": br"\bigcap{}",
    u"\u22C3": br"\bigcup{}",
    u"\u22C4": br"\diamond{}",
    u"\u22C5": br"\cdot{}",
    u"\u22C6": br"\star{}",
    u"\u22C7": br"\divideontimes{}",
    u"\u22C8": br"\bowtie{}",
    u"\u22C9": br"\ltimes{}",
    u"\u22CA": br"\rtimes{}",
    u"\u22CB": br"\leftthreetimes{}",
    u"\u22CC": br"\rightthreetimes{}",
    u"\u22CD": br"\backsimeq{}",
    u"\u22CE": br"\curlyvee{}",
    u"\u22CF": br"\curlywedge{}",
    u"\u22D0": br"\Subset{}",
    u"\u22D1": br"\Supset{}",
    u"\u22D2": br"\Cap{}",
    u"\u22D3": br"\Cup{}",
    u"\u22D4": br"\pitchfork{}",
    u"\u22D6": br"\lessdot{}",
    u"\u22D7": br"\gtrdot{}",
    u"\u22D8": br"\verymuchless{}",
    u"\u22D9": br"\verymuchgreater{}",
    u"\u22DA": br"\lesseqgtr{}",
    u"\u22DB": br"\gtreqless{}",
    u"\u22DE": br"\curlyeqprec{}",
    u"\u22DF": br"\curlyeqsucc{}",
    u"\u22E2": br"\not\sqsubseteq{}",
    u"\u22E3": br"\not\sqsupseteq{}",
    u"\u22E5": br"\Elzsqspne{}",
    u"\u22E6": br"\lnsim{}",
    u"\u22E7": br"\gnsim{}",
    u"\u22E8": br"\precedesnotsimilar{}",
    u"\u22E9": br"\succnsim{}",
    u"\u22EA": br"\ntriangleleft{}",
    u"\u22EB": br"\ntriangleright{}",
    u"\u22EC": br"\ntrianglelefteq{}",
    u"\u22ED": br"\ntrianglerighteq{}",
    u"\u22EE": br"\vdots{}",
    u"\u22EF": br"\cdots{}",
    u"\u22F0": br"\upslopeellipsis{}",
    u"\u22F1": br"\downslopeellipsis{}",
    u"\u2305": br"\barwedge{}",
    u"\u2306": br"\perspcorrespond{}",
    u"\u2308": br"\lceil{}",
    u"\u2309": br"\rceil{}",
    u"\u230A": br"\lfloor{}",
    u"\u230B": br"\rfloor{}",
    u"\u2315": br"\recorder{}",
    u"\u2316": br'\mathchar"2208', # note single-quoted string
    u"\u231C": br"\ulcorner{}",
    u"\u231D": br"\urcorner{}",
    u"\u231E": br"\llcorner{}",
    u"\u231F": br"\lrcorner{}",
    u"\u2322": br"\frown{}",
    u"\u2323": br"\smile{}",
    u"\u2329": br"\langle{}",
    u"\u232A": br"\rangle{}",
    u"\u233D": br"\ElsevierGlyph{E838}",
    u"\u23A3": br"\Elzdlcorn{}",
    u"\u23B0": br"\lmoustache{}",
    u"\u23B1": br"\rmoustache{}",
    u"\u2423": br"\textvisiblespace{}",
    u"\u2460": br"\ding{172}",
    u"\u2461": br"\ding{173}",
    u"\u2462": br"\ding{174}",
    u"\u2463": br"\ding{175}",
    u"\u2464": br"\ding{176}",
    u"\u2465": br"\ding{177}",
    u"\u2466": br"\ding{178}",
    u"\u2467": br"\ding{179}",
    u"\u2468": br"\ding{180}",
    u"\u2469": br"\ding{181}",
    u"\u24C8": br"\circledS{}",
    u"\u2506": br"\Elzdshfnc{}",
    u"\u2519": br"\Elzsqfnw{}",
    u"\u2571": br"\diagup{}",
    u"\u25A0": br"\ding{110}",
    u"\u25A1": br"\square{}",
    u"\u25AA": br"\blacksquare{}",
    u"\u25AD": br"\fbox{~~}",
    u"\u25AF": br"\Elzvrecto{}",
    u"\u25B1": br"\ElsevierGlyph{E381}",
    u"\u25B2": br"\ding{115}",
    u"\u25B3": br"\bigtriangleup{}",
    u"\u25B4": br"\blacktriangle{}",
    u"\u25B5": br"\vartriangle{}",
    u"\u25B8": br"\blacktriangleright{}",
    u"\u25B9": br"\triangleright{}",
    u"\u25BC": br"\ding{116}",
    u"\u25BD": br"\bigtriangledown{}",
    u"\u25BE": br"\blacktriangledown{}",
    u"\u25BF": br"\triangledown{}",
    u"\u25C2": br"\blacktriangleleft{}",
    u"\u25C3": br"\triangleleft{}",
    u"\u25C6": br"\ding{117}",
    u"\u25CA": br"\lozenge{}",
    u"\u25CB": br"\bigcirc{}",
    u"\u25CF": br"\ding{108}",
    u"\u25D0": br"\Elzcirfl{}",
    u"\u25D1": br"\Elzcirfr{}",
    u"\u25D2": br"\Elzcirfb{}",
    u"\u25D7": br"\ding{119}",
    u"\u25D8": br"\Elzrvbull{}",
    u"\u25E7": br"\Elzsqfl{}",
    u"\u25E8": br"\Elzsqfr{}",
    u"\u25EA": br"\Elzsqfse{}",
    u"\u25EF": br"\bigcirc{}",
    u"\u2605": br"\ding{72}",
    u"\u2606": br"\ding{73}",
    u"\u260E": br"\ding{37}",
    u"\u261B": br"\ding{42}",
    u"\u261E": br"\ding{43}",
    u"\u263E": br"\rightmoon{}",
    u"\u263F": br"\mercury{}",
    u"\u2640": br"\venus{}",
    u"\u2642": br"\male{}",
    u"\u2643": br"\jupiter{}",
    u"\u2644": br"\saturn{}",
    u"\u2645": br"\uranus{}",
    u"\u2646": br"\neptune{}",
    u"\u2647": br"\pluto{}",
    u"\u2648": br"\aries{}",
    u"\u2649": br"\taurus{}",
    u"\u264A": br"\gemini{}",
    u"\u264B": br"\cancer{}",
    u"\u264C": br"\leo{}",
    u"\u264D": br"\virgo{}",
    u"\u264E": br"\libra{}",
    u"\u264F": br"\scorpio{}",
    u"\u2650": br"\sagittarius{}",
    u"\u2651": br"\capricornus{}",
    u"\u2652": br"\aquarius{}",
    u"\u2653": br"\pisces{}",
    u"\u2660": br"\ding{171}",
    u"\u2662": br"\diamond{}",
    u"\u2663": br"\ding{168}",
    u"\u2665": br"\ding{170}",
    u"\u2666": br"\ding{169}",
    u"\u2669": br"\quarternote{}",
    u"\u266A": br"\eighthnote{}",
    u"\u266D": br"\flat{}",
    u"\u266E": br"\natural{}",
    u"\u266F": br"\sharp{}",
    u"\u2701": br"\ding{33}",
    u"\u2702": br"\ding{34}",
    u"\u2703": br"\ding{35}",
    u"\u2704": br"\ding{36}",
    u"\u2706": br"\ding{38}",
    u"\u2707": br"\ding{39}",
    u"\u2708": br"\ding{40}",
    u"\u2709": br"\ding{41}",
    u"\u270C": br"\ding{44}",
    u"\u270D": br"\ding{45}",
    u"\u270E": br"\ding{46}",
    u"\u270F": br"\ding{47}",
    u"\u2710": br"\ding{48}",
    u"\u2711": br"\ding{49}",
    u"\u2712": br"\ding{50}",
    u"\u2713": br"\ding{51}",
    u"\u2714": br"\ding{52}",
    u"\u2715": br"\ding{53}",
    u"\u2716": br"\ding{54}",
    u"\u2717": br"\ding{55}",
    u"\u2718": br"\ding{56}",
    u"\u2719": br"\ding{57}",
    u"\u271A": br"\ding{58}",
    u"\u271B": br"\ding{59}",
    u"\u271C": br"\ding{60}",
    u"\u271D": br"\ding{61}",
    u"\u271E": br"\ding{62}",
    u"\u271F": br"\ding{63}",
    u"\u2720": br"\ding{64}",
    u"\u2721": br"\ding{65}",
    u"\u2722": br"\ding{66}",
    u"\u2723": br"\ding{67}",
    u"\u2724": br"\ding{68}",
    u"\u2725": br"\ding{69}",
    u"\u2726": br"\ding{70}",
    u"\u2727": br"\ding{71}",
    u"\u2729": br"\ding{73}",
    u"\u272A": br"\ding{74}",
    u"\u272B": br"\ding{75}",
    u"\u272C": br"\ding{76}",
    u"\u272D": br"\ding{77}",
    u"\u272E": br"\ding{78}",
    u"\u272F": br"\ding{79}",
    u"\u2730": br"\ding{80}",
    u"\u2731": br"\ding{81}",
    u"\u2732": br"\ding{82}",
    u"\u2733": br"\ding{83}",
    u"\u2734": br"\ding{84}",
    u"\u2735": br"\ding{85}",
    u"\u2736": br"\ding{86}",
    u"\u2737": br"\ding{87}",
    u"\u2738": br"\ding{88}",
    u"\u2739": br"\ding{89}",
    u"\u273A": br"\ding{90}",
    u"\u273B": br"\ding{91}",
    u"\u273C": br"\ding{92}",
    u"\u273D": br"\ding{93}",
    u"\u273E": br"\ding{94}",
    u"\u273F": br"\ding{95}",
    u"\u2740": br"\ding{96}",
    u"\u2741": br"\ding{97}",
    u"\u2742": br"\ding{98}",
    u"\u2743": br"\ding{99}",
    u"\u2744": br"\ding{100}",
    u"\u2745": br"\ding{101}",
    u"\u2746": br"\ding{102}",
    u"\u2747": br"\ding{103}",
    u"\u2748": br"\ding{104}",
    u"\u2749": br"\ding{105}",
    u"\u274A": br"\ding{106}",
    u"\u274B": br"\ding{107}",
    u"\u274D": br"\ding{109}",
    u"\u274F": br"\ding{111}",
    u"\u2750": br"\ding{112}",
    u"\u2751": br"\ding{113}",
    u"\u2752": br"\ding{114}",
    u"\u2756": br"\ding{118}",
    u"\u2758": br"\ding{120}",
    u"\u2759": br"\ding{121}",
    u"\u275A": br"\ding{122}",
    u"\u275B": br"\ding{123}",
    u"\u275C": br"\ding{124}",
    u"\u275D": br"\ding{125}",
    u"\u275E": br"\ding{126}",
    u"\u2761": br"\ding{161}",
    u"\u2762": br"\ding{162}",
    u"\u2763": br"\ding{163}",
    u"\u2764": br"\ding{164}",
    u"\u2765": br"\ding{165}",
    u"\u2766": br"\ding{166}",
    u"\u2767": br"\ding{167}",
    u"\u2776": br"\ding{182}",
    u"\u2777": br"\ding{183}",
    u"\u2778": br"\ding{184}",
    u"\u2779": br"\ding{185}",
    u"\u277A": br"\ding{186}",
    u"\u277B": br"\ding{187}",
    u"\u277C": br"\ding{188}",
    u"\u277D": br"\ding{189}",
    u"\u277E": br"\ding{190}",
    u"\u277F": br"\ding{191}",
    u"\u2780": br"\ding{192}",
    u"\u2781": br"\ding{193}",
    u"\u2782": br"\ding{194}",
    u"\u2783": br"\ding{195}",
    u"\u2784": br"\ding{196}",
    u"\u2785": br"\ding{197}",
    u"\u2786": br"\ding{198}",
    u"\u2787": br"\ding{199}",
    u"\u2788": br"\ding{200}",
    u"\u2789": br"\ding{201}",
    u"\u278A": br"\ding{202}",
    u"\u278B": br"\ding{203}",
    u"\u278C": br"\ding{204}",
    u"\u278D": br"\ding{205}",
    u"\u278E": br"\ding{206}",
    u"\u278F": br"\ding{207}",
    u"\u2790": br"\ding{208}",
    u"\u2791": br"\ding{209}",
    u"\u2792": br"\ding{210}",
    u"\u2793": br"\ding{211}",
    u"\u2794": br"\ding{212}",
    u"\u2798": br"\ding{216}",
    u"\u2799": br"\ding{217}",
    u"\u279A": br"\ding{218}",
    u"\u279B": br"\ding{219}",
    u"\u279C": br"\ding{220}",
    u"\u279D": br"\ding{221}",
    u"\u279E": br"\ding{222}",
    u"\u279F": br"\ding{223}",
    u"\u27A0": br"\ding{224}",
    u"\u27A1": br"\ding{225}",
    u"\u27A2": br"\ding{226}",
    u"\u27A3": br"\ding{227}",
    u"\u27A4": br"\ding{228}",
    u"\u27A5": br"\ding{229}",
    u"\u27A6": br"\ding{230}",
    u"\u27A7": br"\ding{231}",
    u"\u27A8": br"\ding{232}",
    u"\u27A9": br"\ding{233}",
    u"\u27AA": br"\ding{234}",
    u"\u27AB": br"\ding{235}",
    u"\u27AC": br"\ding{236}",
    u"\u27AD": br"\ding{237}",
    u"\u27AE": br"\ding{238}",
    u"\u27AF": br"\ding{239}",
    u"\u27B1": br"\ding{241}",
    u"\u27B2": br"\ding{242}",
    u"\u27B3": br"\ding{243}",
    u"\u27B4": br"\ding{244}",
    u"\u27B5": br"\ding{245}",
    u"\u27B6": br"\ding{246}",
    u"\u27B7": br"\ding{247}",
    u"\u27B8": br"\ding{248}",
    u"\u27B9": br"\ding{249}",
    u"\u27BA": br"\ding{250}",
    u"\u27BB": br"\ding{251}",
    u"\u27BC": br"\ding{252}",
    u"\u27BD": br"\ding{253}",
    u"\u27BE": br"\ding{254}",
    u"\u27F5": br"\longleftarrow{}",
    u"\u27F6": br"\longrightarrow{}",
    u"\u27F7": br"\longleftrightarrow{}",
    u"\u27F8": br"\Longleftarrow{}",
    u"\u27F9": br"\Longrightarrow{}",
    u"\u27FA": br"\Longleftrightarrow{}",
    u"\u27FC": br"\longmapsto{}",
    u"\u27FF": br"\sim\joinrel\leadsto",
    u"\u2905": br"\ElsevierGlyph{E212}",
    u"\u2912": br"\UpArrowBar{}",
    u"\u2913": br"\DownArrowBar{}",
    u"\u2923": br"\ElsevierGlyph{E20C}",
    u"\u2924": br"\ElsevierGlyph{E20D}",
    u"\u2925": br"\ElsevierGlyph{E20B}",
    u"\u2926": br"\ElsevierGlyph{E20A}",
    u"\u2927": br"\ElsevierGlyph{E211}",
    u"\u2928": br"\ElsevierGlyph{E20E}",
    u"\u2929": br"\ElsevierGlyph{E20F}",
    u"\u292A": br"\ElsevierGlyph{E210}",
    u"\u2933": br"\ElsevierGlyph{E21C}",
    u"\u2936": br"\ElsevierGlyph{E21A}",
    u"\u2937": br"\ElsevierGlyph{E219}",
    u"\u2940": br"\Elolarr{}",
    u"\u2941": br"\Elorarr{}",
    u"\u2942": br"\ElzRlarr{}",
    u"\u2944": br"\ElzrLarr{}",
    u"\u2947": br"\Elzrarrx{}",
    u"\u294E": br"\LeftRightVector{}",
    u"\u294F": br"\RightUpDownVector{}",
    u"\u2950": br"\DownLeftRightVector{}",
    u"\u2951": br"\LeftUpDownVector{}",
    u"\u2952": br"\LeftVectorBar{}",
    u"\u2953": br"\RightVectorBar{}",
    u"\u2954": br"\RightUpVectorBar{}",
    u"\u2955": br"\RightDownVectorBar{}",
    u"\u2956": br"\DownLeftVectorBar{}",
    u"\u2957": br"\DownRightVectorBar{}",
    u"\u2958": br"\LeftUpVectorBar{}",
    u"\u2959": br"\LeftDownVectorBar{}",
    u"\u295A": br"\LeftTeeVector{}",
    u"\u295B": br"\RightTeeVector{}",
    u"\u295C": br"\RightUpTeeVector{}",
    u"\u295D": br"\RightDownTeeVector{}",
    u"\u295E": br"\DownLeftTeeVector{}",
    u"\u295F": br"\DownRightTeeVector{}",
    u"\u2960": br"\LeftUpTeeVector{}",
    u"\u2961": br"\LeftDownTeeVector{}",
    u"\u296E": br"\UpEquilibrium{}",
    u"\u296F": br"\ReverseUpEquilibrium{}",
    u"\u2970": br"\RoundImplies{}",
    u"\u297C": br"\ElsevierGlyph{E214}",
    u"\u297D": br"\ElsevierGlyph{E215}",
    u"\u2980": br"\Elztfnc{}",
    u"\u2985": br"\ElsevierGlyph{3018}",
    u"\u2986": br"\Elroang{}",
    u"\u2993": br"&lt;\kern-0.58em(",
    u"\u2994": br"\ElsevierGlyph{E291}",
    u"\u2999": br"\Elzddfnc{}",
    u"\u299C": br"\Angle{}",
    u"\u29A0": br"\Elzlpargt{}",
    u"\u29B5": br"\ElsevierGlyph{E260}",
    u"\u29B6": br"\ElsevierGlyph{E61B}",
    u"\u29CA": br"\ElzLap{}",
    u"\u29CB": br"\Elzdefas{}",
    u"\u29CF": br"\LeftTriangleBar{}",
    u"\u29D0": br"\RightTriangleBar{}",
    u"\u29DC": br"\ElsevierGlyph{E372}",
    u"\u29EB": br"\blacklozenge{}",
    u"\u29F4": br"\RuleDelayed{}",
    u"\u2A04": br"\Elxuplus{}",
    u"\u2A05": br"\ElzThr{}",
    u"\u2A06": br"\Elxsqcup{}",
    u"\u2A07": br"\ElzInf{}",
    u"\u2A08": br"\ElzSup{}",
    u"\u2A0D": br"\ElzCint{}",
    u"\u2A0F": br"\clockoint{}",
    u"\u2A10": br"\ElsevierGlyph{E395}",
    u"\u2A16": br"\sqrint{}",
    u"\u2A25": br"\ElsevierGlyph{E25A}",
    u"\u2A2A": br"\ElsevierGlyph{E25B}",
    u"\u2A2D": br"\ElsevierGlyph{E25C}",
    u"\u2A2E": br"\ElsevierGlyph{E25D}",
    u"\u2A2F": br"\ElzTimes{}",
    u"\u2A34": br"\ElsevierGlyph{E25E}",
    u"\u2A35": br"\ElsevierGlyph{E25E}",
    u"\u2A3C": br"\ElsevierGlyph{E259}",
    u"\u2A3F": br"\amalg{}",
    u"\u2A53": br"\ElzAnd{}",
    u"\u2A54": br"\ElzOr{}",
    u"\u2A55": br"\ElsevierGlyph{E36E}",
    u"\u2A56": br"\ElOr{}",
    u"\u2A5E": br"\perspcorrespond{}",
    u"\u2A5F": br"\Elzminhat{}",
    u"\u2A63": br"\ElsevierGlyph{225A}",
    u"\u2A6E": br"\stackrel{*}{=}",
    u"\u2A75": br"\Equal{}",
    u"\u2A7D": br"\leqslant{}",
    u"\u2A7E": br"\geqslant{}",
    u"\u2A85": br"\lessapprox{}",
    u"\u2A86": br"\gtrapprox{}",
    u"\u2A87": br"\lneq{}",
    u"\u2A88": br"\gneq{}",
    u"\u2A89": br"\lnapprox{}",
    u"\u2A8A": br"\gnapprox{}",
    u"\u2A8B": br"\lesseqqgtr{}",
    u"\u2A8C": br"\gtreqqless{}",
    u"\u2A95": br"\eqslantless{}",
    u"\u2A96": br"\eqslantgtr{}",
    u"\u2A9D": br"\Pisymbol{ppi020}{117}",
    u"\u2A9E": br"\Pisymbol{ppi020}{105}",
    u"\u2AA1": br"\NestedLessLess{}",
    u"\u2AA2": br"\NestedGreaterGreater{}",
    u"\u2AAF": br"\preceq{}",
    u"\u2AB0": br"\succeq{}",
    u"\u2AB5": br"\precneqq{}",
    u"\u2AB6": br"\succneqq{}",
    u"\u2AB7": br"\precapprox{}",
    u"\u2AB8": br"\succapprox{}",
    u"\u2AB9": br"\precnapprox{}",
    u"\u2ABA": br"\succnapprox{}",
    u"\u2AC5": br"\subseteqq{}",
    u"\u2AC6": br"\supseteqq{}",
    u"\u2ACB": br"\subsetneqq{}",
    u"\u2ACC": br"\supsetneqq{}",
    u"\u2AEB": br"\ElsevierGlyph{E30D}",
    u"\u2AF6": br"\Elztdcol{}",
    u"\u2AFD": br"{{/}\!\!{/}}",
    u"\u300A": br"\ElsevierGlyph{300A}",
    u"\u300B": br"\ElsevierGlyph{300B}",
    u"\u3018": br"\ElsevierGlyph{3018}",
    u"\u3019": br"\ElsevierGlyph{3019}",
    u"\u301A": br"\openbracketleft{}",
    u"\u301B": br"\openbracketright{}",
    u"\uFB00": br"ff",
    u"\uFB01": br"fi",
    u"\uFB02": br"fl",
    u"\uFB03": br"ffi",
    u"\uFB04": br"ffl",
    u"\uD400": br"\mathbf{A}",
    u"\uD401": br"\mathbf{B}",
    u"\uD402": br"\mathbf{C}",
    u"\uD403": br"\mathbf{D}",
    u"\uD404": br"\mathbf{E}",
    u"\uD405": br"\mathbf{F}",
    u"\uD406": br"\mathbf{G}",
    u"\uD407": br"\mathbf{H}",
    u"\uD408": br"\mathbf{I}",
    u"\uD409": br"\mathbf{J}",
    u"\uD40A": br"\mathbf{K}",
    u"\uD40B": br"\mathbf{L}",
    u"\uD40C": br"\mathbf{M}",
    u"\uD40D": br"\mathbf{N}",
    u"\uD40E": br"\mathbf{O}",
    u"\uD40F": br"\mathbf{P}",
    u"\uD410": br"\mathbf{Q}",
    u"\uD411": br"\mathbf{R}",
    u"\uD412": br"\mathbf{S}",
    u"\uD413": br"\mathbf{T}",
    u"\uD414": br"\mathbf{U}",
    u"\uD415": br"\mathbf{V}",
    u"\uD416": br"\mathbf{W}",
    u"\uD417": br"\mathbf{X}",
    u"\uD418": br"\mathbf{Y}",
    u"\uD419": br"\mathbf{Z}",
    u"\uD41A": br"\mathbf{a}",
    u"\uD41B": br"\mathbf{b}",
    u"\uD41C": br"\mathbf{c}",
    u"\uD41D": br"\mathbf{d}",
    u"\uD41E": br"\mathbf{e}",
    u"\uD41F": br"\mathbf{f}",
    u"\uD420": br"\mathbf{g}",
    u"\uD421": br"\mathbf{h}",
    u"\uD422": br"\mathbf{i}",
    u"\uD423": br"\mathbf{j}",
    u"\uD424": br"\mathbf{k}",
    u"\uD425": br"\mathbf{l}",
    u"\uD426": br"\mathbf{m}",
    u"\uD427": br"\mathbf{n}",
    u"\uD428": br"\mathbf{o}",
    u"\uD429": br"\mathbf{p}",
    u"\uD42A": br"\mathbf{q}",
    u"\uD42B": br"\mathbf{r}",
    u"\uD42C": br"\mathbf{s}",
    u"\uD42D": br"\mathbf{t}",
    u"\uD42E": br"\mathbf{u}",
    u"\uD42F": br"\mathbf{v}",
    u"\uD430": br"\mathbf{w}",
    u"\uD431": br"\mathbf{x}",
    u"\uD432": br"\mathbf{y}",
    u"\uD433": br"\mathbf{z}",
    u"\uD434": br"\mathsl{A}",
    u"\uD435": br"\mathsl{B}",
    u"\uD436": br"\mathsl{C}",
    u"\uD437": br"\mathsl{D}",
    u"\uD438": br"\mathsl{E}",
    u"\uD439": br"\mathsl{F}",
    u"\uD43A": br"\mathsl{G}",
    u"\uD43B": br"\mathsl{H}",
    u"\uD43C": br"\mathsl{I}",
    u"\uD43D": br"\mathsl{J}",
    u"\uD43E": br"\mathsl{K}",
    u"\uD43F": br"\mathsl{L}",
    u"\uD440": br"\mathsl{M}",
    u"\uD441": br"\mathsl{N}",
    u"\uD442": br"\mathsl{O}",
    u"\uD443": br"\mathsl{P}",
    u"\uD444": br"\mathsl{Q}",
    u"\uD445": br"\mathsl{R}",
    u"\uD446": br"\mathsl{S}",
    u"\uD447": br"\mathsl{T}",
    u"\uD448": br"\mathsl{U}",
    u"\uD449": br"\mathsl{V}",
    u"\uD44A": br"\mathsl{W}",
    u"\uD44B": br"\mathsl{X}",
    u"\uD44C": br"\mathsl{Y}",
    u"\uD44D": br"\mathsl{Z}",
    u"\uD44E": br"\mathsl{a}",
    u"\uD44F": br"\mathsl{b}",
    u"\uD450": br"\mathsl{c}",
    u"\uD451": br"\mathsl{d}",
    u"\uD452": br"\mathsl{e}",
    u"\uD453": br"\mathsl{f}",
    u"\uD454": br"\mathsl{g}",
    u"\uD456": br"\mathsl{i}",
    u"\uD457": br"\mathsl{j}",
    u"\uD458": br"\mathsl{k}",
    u"\uD459": br"\mathsl{l}",
    u"\uD45A": br"\mathsl{m}",
    u"\uD45B": br"\mathsl{n}",
    u"\uD45C": br"\mathsl{o}",
    u"\uD45D": br"\mathsl{p}",
    u"\uD45E": br"\mathsl{q}",
    u"\uD45F": br"\mathsl{r}",
    u"\uD460": br"\mathsl{s}",
    u"\uD461": br"\mathsl{t}",
    u"\uD462": br"\mathsl{u}",
    u"\uD463": br"\mathsl{v}",
    u"\uD464": br"\mathsl{w}",
    u"\uD465": br"\mathsl{x}",
    u"\uD466": br"\mathsl{y}",
    u"\uD467": br"\mathsl{z}",
    u"\uD468": br"\mathbit{A}",
    u"\uD469": br"\mathbit{B}",
    u"\uD46A": br"\mathbit{C}",
    u"\uD46B": br"\mathbit{D}",
    u"\uD46C": br"\mathbit{E}",
    u"\uD46D": br"\mathbit{F}",
    u"\uD46E": br"\mathbit{G}",
    u"\uD46F": br"\mathbit{H}",
    u"\uD470": br"\mathbit{I}",
    u"\uD471": br"\mathbit{J}",
    u"\uD472": br"\mathbit{K}",
    u"\uD473": br"\mathbit{L}",
    u"\uD474": br"\mathbit{M}",
    u"\uD475": br"\mathbit{N}",
    u"\uD476": br"\mathbit{O}",
    u"\uD477": br"\mathbit{P}",
    u"\uD478": br"\mathbit{Q}",
    u"\uD479": br"\mathbit{R}",
    u"\uD47A": br"\mathbit{S}",
    u"\uD47B": br"\mathbit{T}",
    u"\uD47C": br"\mathbit{U}",
    u"\uD47D": br"\mathbit{V}",
    u"\uD47E": br"\mathbit{W}",
    u"\uD47F": br"\mathbit{X}",
    u"\uD480": br"\mathbit{Y}",
    u"\uD481": br"\mathbit{Z}",
    u"\uD482": br"\mathbit{a}",
    u"\uD483": br"\mathbit{b}",
    u"\uD484": br"\mathbit{c}",
    u"\uD485": br"\mathbit{d}",
    u"\uD486": br"\mathbit{e}",
    u"\uD487": br"\mathbit{f}",
    u"\uD488": br"\mathbit{g}",
    u"\uD489": br"\mathbit{h}",
    u"\uD48A": br"\mathbit{i}",
    u"\uD48B": br"\mathbit{j}",
    u"\uD48C": br"\mathbit{k}",
    u"\uD48D": br"\mathbit{l}",
    u"\uD48E": br"\mathbit{m}",
    u"\uD48F": br"\mathbit{n}",
    u"\uD490": br"\mathbit{o}",
    u"\uD491": br"\mathbit{p}",
    u"\uD492": br"\mathbit{q}",
    u"\uD493": br"\mathbit{r}",
    u"\uD494": br"\mathbit{s}",
    u"\uD495": br"\mathbit{t}",
    u"\uD496": br"\mathbit{u}",
    u"\uD497": br"\mathbit{v}",
    u"\uD498": br"\mathbit{w}",
    u"\uD499": br"\mathbit{x}",
    u"\uD49A": br"\mathbit{y}",
    u"\uD49B": br"\mathbit{z}",
    u"\uD49C": br"\mathscr{A}",
    u"\uD49E": br"\mathscr{C}",
    u"\uD49F": br"\mathscr{D}",
    u"\uD4A2": br"\mathscr{G}",
    u"\uD4A5": br"\mathscr{J}",
    u"\uD4A6": br"\mathscr{K}",
    u"\uD4A9": br"\mathscr{N}",
    u"\uD4AA": br"\mathscr{O}",
    u"\uD4AB": br"\mathscr{P}",
    u"\uD4AC": br"\mathscr{Q}",
    u"\uD4AE": br"\mathscr{S}",
    u"\uD4AF": br"\mathscr{T}",
    u"\uD4B0": br"\mathscr{U}",
    u"\uD4B1": br"\mathscr{V}",
    u"\uD4B2": br"\mathscr{W}",
    u"\uD4B3": br"\mathscr{X}",
    u"\uD4B4": br"\mathscr{Y}",
    u"\uD4B5": br"\mathscr{Z}",
    u"\uD4B6": br"\mathscr{a}",
    u"\uD4B7": br"\mathscr{b}",
    u"\uD4B8": br"\mathscr{c}",
    u"\uD4B9": br"\mathscr{d}",
    u"\uD4BB": br"\mathscr{f}",
    u"\uD4BD": br"\mathscr{h}",
    u"\uD4BE": br"\mathscr{i}",
    u"\uD4BF": br"\mathscr{j}",
    u"\uD4C0": br"\mathscr{k}",
    u"\uD4C1": br"\mathscr{l}",
    u"\uD4C2": br"\mathscr{m}",
    u"\uD4C3": br"\mathscr{n}",
    u"\uD4C5": br"\mathscr{p}",
    u"\uD4C6": br"\mathscr{q}",
    u"\uD4C7": br"\mathscr{r}",
    u"\uD4C8": br"\mathscr{s}",
    u"\uD4C9": br"\mathscr{t}",
    u"\uD4CA": br"\mathscr{u}",
    u"\uD4CB": br"\mathscr{v}",
    u"\uD4CC": br"\mathscr{w}",
    u"\uD4CD": br"\mathscr{x}",
    u"\uD4CE": br"\mathscr{y}",
    u"\uD4CF": br"\mathscr{z}",
    u"\uD4D0": br"\mathmit{A}",
    u"\uD4D1": br"\mathmit{B}",
    u"\uD4D2": br"\mathmit{C}",
    u"\uD4D3": br"\mathmit{D}",
    u"\uD4D4": br"\mathmit{E}",
    u"\uD4D5": br"\mathmit{F}",
    u"\uD4D6": br"\mathmit{G}",
    u"\uD4D7": br"\mathmit{H}",
    u"\uD4D8": br"\mathmit{I}",
    u"\uD4D9": br"\mathmit{J}",
    u"\uD4DA": br"\mathmit{K}",
    u"\uD4DB": br"\mathmit{L}",
    u"\uD4DC": br"\mathmit{M}",
    u"\uD4DD": br"\mathmit{N}",
    u"\uD4DE": br"\mathmit{O}",
    u"\uD4DF": br"\mathmit{P}",
    u"\uD4E0": br"\mathmit{Q}",
    u"\uD4E1": br"\mathmit{R}",
    u"\uD4E2": br"\mathmit{S}",
    u"\uD4E3": br"\mathmit{T}",
    u"\uD4E4": br"\mathmit{U}",
    u"\uD4E5": br"\mathmit{V}",
    u"\uD4E6": br"\mathmit{W}",
    u"\uD4E7": br"\mathmit{X}",
    u"\uD4E8": br"\mathmit{Y}",
    u"\uD4E9": br"\mathmit{Z}",
    u"\uD4EA": br"\mathmit{a}",
    u"\uD4EB": br"\mathmit{b}",
    u"\uD4EC": br"\mathmit{c}",
    u"\uD4ED": br"\mathmit{d}",
    u"\uD4EE": br"\mathmit{e}",
    u"\uD4EF": br"\mathmit{f}",
    u"\uD4F0": br"\mathmit{g}",
    u"\uD4F1": br"\mathmit{h}",
    u"\uD4F2": br"\mathmit{i}",
    u"\uD4F3": br"\mathmit{j}",
    u"\uD4F4": br"\mathmit{k}",
    u"\uD4F5": br"\mathmit{l}",
    u"\uD4F6": br"\mathmit{m}",
    u"\uD4F7": br"\mathmit{n}",
    u"\uD4F8": br"\mathmit{o}",
    u"\uD4F9": br"\mathmit{p}",
    u"\uD4FA": br"\mathmit{q}",
    u"\uD4FB": br"\mathmit{r}",
    u"\uD4FC": br"\mathmit{s}",
    u"\uD4FD": br"\mathmit{t}",
    u"\uD4FE": br"\mathmit{u}",
    u"\uD4FF": br"\mathmit{v}",
    u"\uD500": br"\mathmit{w}",
    u"\uD501": br"\mathmit{x}",
    u"\uD502": br"\mathmit{y}",
    u"\uD503": br"\mathmit{z}",
    u"\uD504": br"\mathfrak{A}",
    u"\uD505": br"\mathfrak{B}",
    u"\uD507": br"\mathfrak{D}",
    u"\uD508": br"\mathfrak{E}",
    u"\uD509": br"\mathfrak{F}",
    u"\uD50A": br"\mathfrak{G}",
    u"\uD50D": br"\mathfrak{J}",
    u"\uD50E": br"\mathfrak{K}",
    u"\uD50F": br"\mathfrak{L}",
    u"\uD510": br"\mathfrak{M}",
    u"\uD511": br"\mathfrak{N}",
    u"\uD512": br"\mathfrak{O}",
    u"\uD513": br"\mathfrak{P}",
    u"\uD514": br"\mathfrak{Q}",
    u"\uD516": br"\mathfrak{S}",
    u"\uD517": br"\mathfrak{T}",
    u"\uD518": br"\mathfrak{U}",
    u"\uD519": br"\mathfrak{V}",
    u"\uD51A": br"\mathfrak{W}",
    u"\uD51B": br"\mathfrak{X}",
    u"\uD51C": br"\mathfrak{Y}",
    u"\uD51E": br"\mathfrak{a}",
    u"\uD51F": br"\mathfrak{b}",
    u"\uD520": br"\mathfrak{c}",
    u"\uD521": br"\mathfrak{d}",
    u"\uD522": br"\mathfrak{e}",
    u"\uD523": br"\mathfrak{f}",
    u"\uD524": br"\mathfrak{g}",
    u"\uD525": br"\mathfrak{h}",
    u"\uD526": br"\mathfrak{i}",
    u"\uD527": br"\mathfrak{j}",
    u"\uD528": br"\mathfrak{k}",
    u"\uD529": br"\mathfrak{l}",
    u"\uD52A": br"\mathfrak{m}",
    u"\uD52B": br"\mathfrak{n}",
    u"\uD52C": br"\mathfrak{o}",
    u"\uD52D": br"\mathfrak{p}",
    u"\uD52E": br"\mathfrak{q}",
    u"\uD52F": br"\mathfrak{r}",
    u"\uD530": br"\mathfrak{s}",
    u"\uD531": br"\mathfrak{t}",
    u"\uD532": br"\mathfrak{u}",
    u"\uD533": br"\mathfrak{v}",
    u"\uD534": br"\mathfrak{w}",
    u"\uD535": br"\mathfrak{x}",
    u"\uD536": br"\mathfrak{y}",
    u"\uD537": br"\mathfrak{z}",
    u"\uD538": br"\mathbb{A}",
    u"\uD539": br"\mathbb{B}",
    u"\uD53B": br"\mathbb{D}",
    u"\uD53C": br"\mathbb{E}",
    u"\uD53D": br"\mathbb{F}",
    u"\uD53E": br"\mathbb{G}",
    u"\uD540": br"\mathbb{I}",
    u"\uD541": br"\mathbb{J}",
    u"\uD542": br"\mathbb{K}",
    u"\uD543": br"\mathbb{L}",
    u"\uD544": br"\mathbb{M}",
    u"\uD546": br"\mathbb{O}",
    u"\uD54A": br"\mathbb{S}",
    u"\uD54B": br"\mathbb{T}",
    u"\uD54C": br"\mathbb{U}",
    u"\uD54D": br"\mathbb{V}",
    u"\uD54E": br"\mathbb{W}",
    u"\uD54F": br"\mathbb{X}",
    u"\uD550": br"\mathbb{Y}",
    u"\uD552": br"\mathbb{a}",
    u"\uD553": br"\mathbb{b}",
    u"\uD554": br"\mathbb{c}",
    u"\uD555": br"\mathbb{d}",
    u"\uD556": br"\mathbb{e}",
    u"\uD557": br"\mathbb{f}",
    u"\uD558": br"\mathbb{g}",
    u"\uD559": br"\mathbb{h}",
    u"\uD55A": br"\mathbb{i}",
    u"\uD55B": br"\mathbb{j}",
    u"\uD55C": br"\mathbb{k}",
    u"\uD55D": br"\mathbb{l}",
    u"\uD55E": br"\mathbb{m}",
    u"\uD55F": br"\mathbb{n}",
    u"\uD560": br"\mathbb{o}",
    u"\uD561": br"\mathbb{p}",
    u"\uD562": br"\mathbb{q}",
    u"\uD563": br"\mathbb{r}",
    u"\uD564": br"\mathbb{s}",
    u"\uD565": br"\mathbb{t}",
    u"\uD566": br"\mathbb{u}",
    u"\uD567": br"\mathbb{v}",
    u"\uD568": br"\mathbb{w}",
    u"\uD569": br"\mathbb{x}",
    u"\uD56A": br"\mathbb{y}",
    u"\uD56B": br"\mathbb{z}",
    u"\uD56C": br"\mathslbb{A}",
    u"\uD56D": br"\mathslbb{B}",
    u"\uD56E": br"\mathslbb{C}",
    u"\uD56F": br"\mathslbb{D}",
    u"\uD570": br"\mathslbb{E}",
    u"\uD571": br"\mathslbb{F}",
    u"\uD572": br"\mathslbb{G}",
    u"\uD573": br"\mathslbb{H}",
    u"\uD574": br"\mathslbb{I}",
    u"\uD575": br"\mathslbb{J}",
    u"\uD576": br"\mathslbb{K}",
    u"\uD577": br"\mathslbb{L}",
    u"\uD578": br"\mathslbb{M}",
    u"\uD579": br"\mathslbb{N}",
    u"\uD57A": br"\mathslbb{O}",
    u"\uD57B": br"\mathslbb{P}",
    u"\uD57C": br"\mathslbb{Q}",
    u"\uD57D": br"\mathslbb{R}",
    u"\uD57E": br"\mathslbb{S}",
    u"\uD57F": br"\mathslbb{T}",
    u"\uD580": br"\mathslbb{U}",
    u"\uD581": br"\mathslbb{V}",
    u"\uD582": br"\mathslbb{W}",
    u"\uD583": br"\mathslbb{X}",
    u"\uD584": br"\mathslbb{Y}",
    u"\uD585": br"\mathslbb{Z}",
    u"\uD586": br"\mathslbb{a}",
    u"\uD587": br"\mathslbb{b}",
    u"\uD588": br"\mathslbb{c}",
    u"\uD589": br"\mathslbb{d}",
    u"\uD58A": br"\mathslbb{e}",
    u"\uD58B": br"\mathslbb{f}",
    u"\uD58C": br"\mathslbb{g}",
    u"\uD58D": br"\mathslbb{h}",
    u"\uD58E": br"\mathslbb{i}",
    u"\uD58F": br"\mathslbb{j}",
    u"\uD590": br"\mathslbb{k}",
    u"\uD591": br"\mathslbb{l}",
    u"\uD592": br"\mathslbb{m}",
    u"\uD593": br"\mathslbb{n}",
    u"\uD594": br"\mathslbb{o}",
    u"\uD595": br"\mathslbb{p}",
    u"\uD596": br"\mathslbb{q}",
    u"\uD597": br"\mathslbb{r}",
    u"\uD598": br"\mathslbb{s}",
    u"\uD599": br"\mathslbb{t}",
    u"\uD59A": br"\mathslbb{u}",
    u"\uD59B": br"\mathslbb{v}",
    u"\uD59C": br"\mathslbb{w}",
    u"\uD59D": br"\mathslbb{x}",
    u"\uD59E": br"\mathslbb{y}",
    u"\uD59F": br"\mathslbb{z}",
    u"\uD5A0": br"\mathsf{A}",
    u"\uD5A1": br"\mathsf{B}",
    u"\uD5A2": br"\mathsf{C}",
    u"\uD5A3": br"\mathsf{D}",
    u"\uD5A4": br"\mathsf{E}",
    u"\uD5A5": br"\mathsf{F}",
    u"\uD5A6": br"\mathsf{G}",
    u"\uD5A7": br"\mathsf{H}",
    u"\uD5A8": br"\mathsf{I}",
    u"\uD5A9": br"\mathsf{J}",
    u"\uD5AA": br"\mathsf{K}",
    u"\uD5AB": br"\mathsf{L}",
    u"\uD5AC": br"\mathsf{M}",
    u"\uD5AD": br"\mathsf{N}",
    u"\uD5AE": br"\mathsf{O}",
    u"\uD5AF": br"\mathsf{P}",
    u"\uD5B0": br"\mathsf{Q}",
    u"\uD5B1": br"\mathsf{R}",
    u"\uD5B2": br"\mathsf{S}",
    u"\uD5B3": br"\mathsf{T}",
    u"\uD5B4": br"\mathsf{U}",
    u"\uD5B5": br"\mathsf{V}",
    u"\uD5B6": br"\mathsf{W}",
    u"\uD5B7": br"\mathsf{X}",
    u"\uD5B8": br"\mathsf{Y}",
    u"\uD5B9": br"\mathsf{Z}",
    u"\uD5BA": br"\mathsf{a}",
    u"\uD5BB": br"\mathsf{b}",
    u"\uD5BC": br"\mathsf{c}",
    u"\uD5BD": br"\mathsf{d}",
    u"\uD5BE": br"\mathsf{e}",
    u"\uD5BF": br"\mathsf{f}",
    u"\uD5C0": br"\mathsf{g}",
    u"\uD5C1": br"\mathsf{h}",
    u"\uD5C2": br"\mathsf{i}",
    u"\uD5C3": br"\mathsf{j}",
    u"\uD5C4": br"\mathsf{k}",
    u"\uD5C5": br"\mathsf{l}",
    u"\uD5C6": br"\mathsf{m}",
    u"\uD5C7": br"\mathsf{n}",
    u"\uD5C8": br"\mathsf{o}",
    u"\uD5C9": br"\mathsf{p}",
    u"\uD5CA": br"\mathsf{q}",
    u"\uD5CB": br"\mathsf{r}",
    u"\uD5CC": br"\mathsf{s}",
    u"\uD5CD": br"\mathsf{t}",
    u"\uD5CE": br"\mathsf{u}",
    u"\uD5CF": br"\mathsf{v}",
    u"\uD5D0": br"\mathsf{w}",
    u"\uD5D1": br"\mathsf{x}",
    u"\uD5D2": br"\mathsf{y}",
    u"\uD5D3": br"\mathsf{z}",
    u"\uD5D4": br"\mathsfbf{A}",
    u"\uD5D5": br"\mathsfbf{B}",
    u"\uD5D6": br"\mathsfbf{C}",
    u"\uD5D7": br"\mathsfbf{D}",
    u"\uD5D8": br"\mathsfbf{E}",
    u"\uD5D9": br"\mathsfbf{F}",
    u"\uD5DA": br"\mathsfbf{G}",
    u"\uD5DB": br"\mathsfbf{H}",
    u"\uD5DC": br"\mathsfbf{I}",
    u"\uD5DD": br"\mathsfbf{J}",
    u"\uD5DE": br"\mathsfbf{K}",
    u"\uD5DF": br"\mathsfbf{L}",
    u"\uD5E0": br"\mathsfbf{M}",
    u"\uD5E1": br"\mathsfbf{N}",
    u"\uD5E2": br"\mathsfbf{O}",
    u"\uD5E3": br"\mathsfbf{P}",
    u"\uD5E4": br"\mathsfbf{Q}",
    u"\uD5E5": br"\mathsfbf{R}",
    u"\uD5E6": br"\mathsfbf{S}",
    u"\uD5E7": br"\mathsfbf{T}",
    u"\uD5E8": br"\mathsfbf{U}",
    u"\uD5E9": br"\mathsfbf{V}",
    u"\uD5EA": br"\mathsfbf{W}",
    u"\uD5EB": br"\mathsfbf{X}",
    u"\uD5EC": br"\mathsfbf{Y}",
    u"\uD5ED": br"\mathsfbf{Z}",
    u"\uD5EE": br"\mathsfbf{a}",
    u"\uD5EF": br"\mathsfbf{b}",
    u"\uD5F0": br"\mathsfbf{c}",
    u"\uD5F1": br"\mathsfbf{d}",
    u"\uD5F2": br"\mathsfbf{e}",
    u"\uD5F3": br"\mathsfbf{f}",
    u"\uD5F4": br"\mathsfbf{g}",
    u"\uD5F5": br"\mathsfbf{h}",
    u"\uD5F6": br"\mathsfbf{i}",
    u"\uD5F7": br"\mathsfbf{j}",
    u"\uD5F8": br"\mathsfbf{k}",
    u"\uD5F9": br"\mathsfbf{l}",
    u"\uD5FA": br"\mathsfbf{m}",
    u"\uD5FB": br"\mathsfbf{n}",
    u"\uD5FC": br"\mathsfbf{o}",
    u"\uD5FD": br"\mathsfbf{p}",
    u"\uD5FE": br"\mathsfbf{q}",
    u"\uD5FF": br"\mathsfbf{r}",
    u"\uD600": br"\mathsfbf{s}",
    u"\uD601": br"\mathsfbf{t}",
    u"\uD602": br"\mathsfbf{u}",
    u"\uD603": br"\mathsfbf{v}",
    u"\uD604": br"\mathsfbf{w}",
    u"\uD605": br"\mathsfbf{x}",
    u"\uD606": br"\mathsfbf{y}",
    u"\uD607": br"\mathsfbf{z}",
    u"\uD608": br"\mathsfsl{A}",
    u"\uD609": br"\mathsfsl{B}",
    u"\uD60A": br"\mathsfsl{C}",
    u"\uD60B": br"\mathsfsl{D}",
    u"\uD60C": br"\mathsfsl{E}",
    u"\uD60D": br"\mathsfsl{F}",
    u"\uD60E": br"\mathsfsl{G}",
    u"\uD60F": br"\mathsfsl{H}",
    u"\uD610": br"\mathsfsl{I}",
    u"\uD611": br"\mathsfsl{J}",
    u"\uD612": br"\mathsfsl{K}",
    u"\uD613": br"\mathsfsl{L}",
    u"\uD614": br"\mathsfsl{M}",
    u"\uD615": br"\mathsfsl{N}",
    u"\uD616": br"\mathsfsl{O}",
    u"\uD617": br"\mathsfsl{P}",
    u"\uD618": br"\mathsfsl{Q}",
    u"\uD619": br"\mathsfsl{R}",
    u"\uD61A": br"\mathsfsl{S}",
    u"\uD61B": br"\mathsfsl{T}",
    u"\uD61C": br"\mathsfsl{U}",
    u"\uD61D": br"\mathsfsl{V}",
    u"\uD61E": br"\mathsfsl{W}",
    u"\uD61F": br"\mathsfsl{X}",
    u"\uD620": br"\mathsfsl{Y}",
    u"\uD621": br"\mathsfsl{Z}",
    u"\uD622": br"\mathsfsl{a}",
    u"\uD623": br"\mathsfsl{b}",
    u"\uD624": br"\mathsfsl{c}",
    u"\uD625": br"\mathsfsl{d}",
    u"\uD626": br"\mathsfsl{e}",
    u"\uD627": br"\mathsfsl{f}",
    u"\uD628": br"\mathsfsl{g}",
    u"\uD629": br"\mathsfsl{h}",
    u"\uD62A": br"\mathsfsl{i}",
    u"\uD62B": br"\mathsfsl{j}",
    u"\uD62C": br"\mathsfsl{k}",
    u"\uD62D": br"\mathsfsl{l}",
    u"\uD62E": br"\mathsfsl{m}",
    u"\uD62F": br"\mathsfsl{n}",
    u"\uD630": br"\mathsfsl{o}",
    u"\uD631": br"\mathsfsl{p}",
    u"\uD632": br"\mathsfsl{q}",
    u"\uD633": br"\mathsfsl{r}",
    u"\uD634": br"\mathsfsl{s}",
    u"\uD635": br"\mathsfsl{t}",
    u"\uD636": br"\mathsfsl{u}",
    u"\uD637": br"\mathsfsl{v}",
    u"\uD638": br"\mathsfsl{w}",
    u"\uD639": br"\mathsfsl{x}",
    u"\uD63A": br"\mathsfsl{y}",
    u"\uD63B": br"\mathsfsl{z}",
    u"\uD63C": br"\mathsfbfsl{A}",
    u"\uD63D": br"\mathsfbfsl{B}",
    u"\uD63E": br"\mathsfbfsl{C}",
    u"\uD63F": br"\mathsfbfsl{D}",
    u"\uD640": br"\mathsfbfsl{E}",
    u"\uD641": br"\mathsfbfsl{F}",
    u"\uD642": br"\mathsfbfsl{G}",
    u"\uD643": br"\mathsfbfsl{H}",
    u"\uD644": br"\mathsfbfsl{I}",
    u"\uD645": br"\mathsfbfsl{J}",
    u"\uD646": br"\mathsfbfsl{K}",
    u"\uD647": br"\mathsfbfsl{L}",
    u"\uD648": br"\mathsfbfsl{M}",
    u"\uD649": br"\mathsfbfsl{N}",
    u"\uD64A": br"\mathsfbfsl{O}",
    u"\uD64B": br"\mathsfbfsl{P}",
    u"\uD64C": br"\mathsfbfsl{Q}",
    u"\uD64D": br"\mathsfbfsl{R}",
    u"\uD64E": br"\mathsfbfsl{S}",
    u"\uD64F": br"\mathsfbfsl{T}",
    u"\uD650": br"\mathsfbfsl{U}",
    u"\uD651": br"\mathsfbfsl{V}",
    u"\uD652": br"\mathsfbfsl{W}",
    u"\uD653": br"\mathsfbfsl{X}",
    u"\uD654": br"\mathsfbfsl{Y}",
    u"\uD655": br"\mathsfbfsl{Z}",
    u"\uD656": br"\mathsfbfsl{a}",
    u"\uD657": br"\mathsfbfsl{b}",
    u"\uD658": br"\mathsfbfsl{c}",
    u"\uD659": br"\mathsfbfsl{d}",
    u"\uD65A": br"\mathsfbfsl{e}",
    u"\uD65B": br"\mathsfbfsl{f}",
    u"\uD65C": br"\mathsfbfsl{g}",
    u"\uD65D": br"\mathsfbfsl{h}",
    u"\uD65E": br"\mathsfbfsl{i}",
    u"\uD65F": br"\mathsfbfsl{j}",
    u"\uD660": br"\mathsfbfsl{k}",
    u"\uD661": br"\mathsfbfsl{l}",
    u"\uD662": br"\mathsfbfsl{m}",
    u"\uD663": br"\mathsfbfsl{n}",
    u"\uD664": br"\mathsfbfsl{o}",
    u"\uD665": br"\mathsfbfsl{p}",
    u"\uD666": br"\mathsfbfsl{q}",
    u"\uD667": br"\mathsfbfsl{r}",
    u"\uD668": br"\mathsfbfsl{s}",
    u"\uD669": br"\mathsfbfsl{t}",
    u"\uD66A": br"\mathsfbfsl{u}",
    u"\uD66B": br"\mathsfbfsl{v}",
    u"\uD66C": br"\mathsfbfsl{w}",
    u"\uD66D": br"\mathsfbfsl{x}",
    u"\uD66E": br"\mathsfbfsl{y}",
    u"\uD66F": br"\mathsfbfsl{z}",
    u"\uD670": br"\mathtt{A}",
    u"\uD671": br"\mathtt{B}",
    u"\uD672": br"\mathtt{C}",
    u"\uD673": br"\mathtt{D}",
    u"\uD674": br"\mathtt{E}",
    u"\uD675": br"\mathtt{F}",
    u"\uD676": br"\mathtt{G}",
    u"\uD677": br"\mathtt{H}",
    u"\uD678": br"\mathtt{I}",
    u"\uD679": br"\mathtt{J}",
    u"\uD67A": br"\mathtt{K}",
    u"\uD67B": br"\mathtt{L}",
    u"\uD67C": br"\mathtt{M}",
    u"\uD67D": br"\mathtt{N}",
    u"\uD67E": br"\mathtt{O}",
    u"\uD67F": br"\mathtt{P}",
    u"\uD680": br"\mathtt{Q}",
    u"\uD681": br"\mathtt{R}",
    u"\uD682": br"\mathtt{S}",
    u"\uD683": br"\mathtt{T}",
    u"\uD684": br"\mathtt{U}",
    u"\uD685": br"\mathtt{V}",
    u"\uD686": br"\mathtt{W}",
    u"\uD687": br"\mathtt{X}",
    u"\uD688": br"\mathtt{Y}",
    u"\uD689": br"\mathtt{Z}",
    u"\uD68A": br"\mathtt{a}",
    u"\uD68B": br"\mathtt{b}",
    u"\uD68C": br"\mathtt{c}",
    u"\uD68D": br"\mathtt{d}",
    u"\uD68E": br"\mathtt{e}",
    u"\uD68F": br"\mathtt{f}",
    u"\uD690": br"\mathtt{g}",
    u"\uD691": br"\mathtt{h}",
    u"\uD692": br"\mathtt{i}",
    u"\uD693": br"\mathtt{j}",
    u"\uD694": br"\mathtt{k}",
    u"\uD695": br"\mathtt{l}",
    u"\uD696": br"\mathtt{m}",
    u"\uD697": br"\mathtt{n}",
    u"\uD698": br"\mathtt{o}",
    u"\uD699": br"\mathtt{p}",
    u"\uD69A": br"\mathtt{q}",
    u"\uD69B": br"\mathtt{r}",
    u"\uD69C": br"\mathtt{s}",
    u"\uD69D": br"\mathtt{t}",
    u"\uD69E": br"\mathtt{u}",
    u"\uD69F": br"\mathtt{v}",
    u"\uD6A0": br"\mathtt{w}",
    u"\uD6A1": br"\mathtt{x}",
    u"\uD6A2": br"\mathtt{y}",
    u"\uD6A3": br"\mathtt{z}",
    u"\uD6A8": br"\mathbf{\Alpha}",
    u"\uD6A9": br"\mathbf{\Beta}",
    u"\uD6AA": br"\mathbf{\Gamma}",
    u"\uD6AB": br"\mathbf{\Delta}",
    u"\uD6AC": br"\mathbf{\Epsilon}",
    u"\uD6AD": br"\mathbf{\Zeta}",
    u"\uD6AE": br"\mathbf{\Eta}",
    u"\uD6AF": br"\mathbf{\Theta}",
    u"\uD6B0": br"\mathbf{\Iota}",
    u"\uD6B1": br"\mathbf{\Kappa}",
    u"\uD6B2": br"\mathbf{\Lambda}",
    u"\uD6B5": br"\mathbf{\Xi}",
    u"\uD6B7": br"\mathbf{\Pi}",
    u"\uD6B8": br"\mathbf{\Rho}",
    u"\uD6B9": br"\mathbf{\vartheta}",
    u"\uD6BA": br"\mathbf{\Sigma}",
    u"\uD6BB": br"\mathbf{\Tau}",
    u"\uD6BC": br"\mathbf{\Upsilon}",
    u"\uD6BD": br"\mathbf{\Phi}",
    u"\uD6BE": br"\mathbf{\Chi}",
    u"\uD6BF": br"\mathbf{\Psi}",
    u"\uD6C0": br"\mathbf{\Omega}",
    u"\uD6C1": br"\mathbf{\nabla}",
    u"\uD6C2": br"\mathbf{\Alpha}",
    u"\uD6C3": br"\mathbf{\Beta}",
    u"\uD6C4": br"\mathbf{\Gamma}",
    u"\uD6C5": br"\mathbf{\Delta}",
    u"\uD6C6": br"\mathbf{\Epsilon}",
    u"\uD6C7": br"\mathbf{\Zeta}",
    u"\uD6C8": br"\mathbf{\Eta}",
    u"\uD6C9": br"\mathbf{\theta}",
    u"\uD6CA": br"\mathbf{\Iota}",
    u"\uD6CB": br"\mathbf{\Kappa}",
    u"\uD6CC": br"\mathbf{\Lambda}",
    u"\uD6CF": br"\mathbf{\Xi}",
    u"\uD6D1": br"\mathbf{\Pi}",
    u"\uD6D2": br"\mathbf{\Rho}",
    u"\uD6D3": br"\mathbf{\varsigma}",
    u"\uD6D4": br"\mathbf{\Sigma}",
    u"\uD6D5": br"\mathbf{\Tau}",
    u"\uD6D6": br"\mathbf{\Upsilon}",
    u"\uD6D7": br"\mathbf{\Phi}",
    u"\uD6D8": br"\mathbf{\Chi}",
    u"\uD6D9": br"\mathbf{\Psi}",
    u"\uD6DA": br"\mathbf{\Omega}",
    u"\uD6DB": br"\partial{}",
    u"\uD6DC": br"\in",
    u"\uD6DD": br"\mathbf{\vartheta}",
    u"\uD6DE": br"\mathbf{\varkappa}",
    u"\uD6DF": br"\mathbf{\phi}",
    u"\uD6E0": br"\mathbf{\varrho}",
    u"\uD6E1": br"\mathbf{\varpi}",
    u"\uD6E2": br"\mathsl{\Alpha}",
    u"\uD6E3": br"\mathsl{\Beta}",
    u"\uD6E4": br"\mathsl{\Gamma}",
    u"\uD6E5": br"\mathsl{\Delta}",
    u"\uD6E6": br"\mathsl{\Epsilon}",
    u"\uD6E7": br"\mathsl{\Zeta}",
    u"\uD6E8": br"\mathsl{\Eta}",
    u"\uD6E9": br"\mathsl{\Theta}",
    u"\uD6EA": br"\mathsl{\Iota}",
    u"\uD6EB": br"\mathsl{\Kappa}",
    u"\uD6EC": br"\mathsl{\Lambda}",
    u"\uD6EF": br"\mathsl{\Xi}",
    u"\uD6F1": br"\mathsl{\Pi}",
    u"\uD6F2": br"\mathsl{\Rho}",
    u"\uD6F3": br"\mathsl{\vartheta}",
    u"\uD6F4": br"\mathsl{\Sigma}",
    u"\uD6F5": br"\mathsl{\Tau}",
    u"\uD6F6": br"\mathsl{\Upsilon}",
    u"\uD6F7": br"\mathsl{\Phi}",
    u"\uD6F8": br"\mathsl{\Chi}",
    u"\uD6F9": br"\mathsl{\Psi}",
    u"\uD6FA": br"\mathsl{\Omega}",
    u"\uD6FB": br"\mathsl{\nabla}",
    u"\uD6FC": br"\mathsl{\Alpha}",
    u"\uD6FD": br"\mathsl{\Beta}",
    u"\uD6FE": br"\mathsl{\Gamma}",
    u"\uD6FF": br"\mathsl{\Delta}",
    u"\uD700": br"\mathsl{\Epsilon}",
    u"\uD701": br"\mathsl{\Zeta}",
    u"\uD702": br"\mathsl{\Eta}",
    u"\uD703": br"\mathsl{\Theta}",
    u"\uD704": br"\mathsl{\Iota}",
    u"\uD705": br"\mathsl{\Kappa}",
    u"\uD706": br"\mathsl{\Lambda}",
    u"\uD709": br"\mathsl{\Xi}",
    u"\uD70B": br"\mathsl{\Pi}",
    u"\uD70C": br"\mathsl{\Rho}",
    u"\uD70D": br"\mathsl{\varsigma}",
    u"\uD70E": br"\mathsl{\Sigma}",
    u"\uD70F": br"\mathsl{\Tau}",
    u"\uD710": br"\mathsl{\Upsilon}",
    u"\uD711": br"\mathsl{\Phi}",
    u"\uD712": br"\mathsl{\Chi}",
    u"\uD713": br"\mathsl{\Psi}",
    u"\uD714": br"\mathsl{\Omega}",
    u"\uD715": br"\partial{}",
    u"\uD716": br"\in",
    u"\uD717": br"\mathsl{\vartheta}",
    u"\uD718": br"\mathsl{\varkappa}",
    u"\uD719": br"\mathsl{\phi}",
    u"\uD71A": br"\mathsl{\varrho}",
    u"\uD71B": br"\mathsl{\varpi}",
    u"\uD71C": br"\mathbit{\Alpha}",
    u"\uD71D": br"\mathbit{\Beta}",
    u"\uD71E": br"\mathbit{\Gamma}",
    u"\uD71F": br"\mathbit{\Delta}",
    u"\uD720": br"\mathbit{\Epsilon}",
    u"\uD721": br"\mathbit{\Zeta}",
    u"\uD722": br"\mathbit{\Eta}",
    u"\uD723": br"\mathbit{\Theta}",
    u"\uD724": br"\mathbit{\Iota}",
    u"\uD725": br"\mathbit{\Kappa}",
    u"\uD726": br"\mathbit{\Lambda}",
    u"\uD729": br"\mathbit{\Xi}",
    u"\uD72B": br"\mathbit{\Pi}",
    u"\uD72C": br"\mathbit{\Rho}",
    u"\uD72D": br"\mathbit{O}",
    u"\uD72E": br"\mathbit{\Sigma}",
    u"\uD72F": br"\mathbit{\Tau}",
    u"\uD730": br"\mathbit{\Upsilon}",
    u"\uD731": br"\mathbit{\Phi}",
    u"\uD732": br"\mathbit{\Chi}",
    u"\uD733": br"\mathbit{\Psi}",
    u"\uD734": br"\mathbit{\Omega}",
    u"\uD735": br"\mathbit{\nabla}",
    u"\uD736": br"\mathbit{\Alpha}",
    u"\uD737": br"\mathbit{\Beta}",
    u"\uD738": br"\mathbit{\Gamma}",
    u"\uD739": br"\mathbit{\Delta}",
    u"\uD73A": br"\mathbit{\Epsilon}",
    u"\uD73B": br"\mathbit{\Zeta}",
    u"\uD73C": br"\mathbit{\Eta}",
    u"\uD73D": br"\mathbit{\Theta}",
    u"\uD73E": br"\mathbit{\Iota}",
    u"\uD73F": br"\mathbit{\Kappa}",
    u"\uD740": br"\mathbit{\Lambda}",
    u"\uD743": br"\mathbit{\Xi}",
    u"\uD745": br"\mathbit{\Pi}",
    u"\uD746": br"\mathbit{\Rho}",
    u"\uD747": br"\mathbit{\varsigma}",
    u"\uD748": br"\mathbit{\Sigma}",
    u"\uD749": br"\mathbit{\Tau}",
    u"\uD74A": br"\mathbit{\Upsilon}",
    u"\uD74B": br"\mathbit{\Phi}",
    u"\uD74C": br"\mathbit{\Chi}",
    u"\uD74D": br"\mathbit{\Psi}",
    u"\uD74E": br"\mathbit{\Omega}",
    u"\uD74F": br"\partial{}",
    u"\uD750": br"\in",
    u"\uD751": br"\mathbit{\vartheta}",
    u"\uD752": br"\mathbit{\varkappa}",
    u"\uD753": br"\mathbit{\phi}",
    u"\uD754": br"\mathbit{\varrho}",
    u"\uD755": br"\mathbit{\varpi}",
    u"\uD756": br"\mathsfbf{\Alpha}",
    u"\uD757": br"\mathsfbf{\Beta}",
    u"\uD758": br"\mathsfbf{\Gamma}",
    u"\uD759": br"\mathsfbf{\Delta}",
    u"\uD75A": br"\mathsfbf{\Epsilon}",
    u"\uD75B": br"\mathsfbf{\Zeta}",
    u"\uD75C": br"\mathsfbf{\Eta}",
    u"\uD75D": br"\mathsfbf{\Theta}",
    u"\uD75E": br"\mathsfbf{\Iota}",
    u"\uD75F": br"\mathsfbf{\Kappa}",
    u"\uD760": br"\mathsfbf{\Lambda}",
    u"\uD763": br"\mathsfbf{\Xi}",
    u"\uD765": br"\mathsfbf{\Pi}",
    u"\uD766": br"\mathsfbf{\Rho}",
    u"\uD767": br"\mathsfbf{\vartheta}",
    u"\uD768": br"\mathsfbf{\Sigma}",
    u"\uD769": br"\mathsfbf{\Tau}",
    u"\uD76A": br"\mathsfbf{\Upsilon}",
    u"\uD76B": br"\mathsfbf{\Phi}",
    u"\uD76C": br"\mathsfbf{\Chi}",
    u"\uD76D": br"\mathsfbf{\Psi}",
    u"\uD76E": br"\mathsfbf{\Omega}",
    u"\uD76F": br"\mathsfbf{\nabla}",
    u"\uD770": br"\mathsfbf{\Alpha}",
    u"\uD771": br"\mathsfbf{\Beta}",
    u"\uD772": br"\mathsfbf{\Gamma}",
    u"\uD773": br"\mathsfbf{\Delta}",
    u"\uD774": br"\mathsfbf{\Epsilon}",
    u"\uD775": br"\mathsfbf{\Zeta}",
    u"\uD776": br"\mathsfbf{\Eta}",
    u"\uD777": br"\mathsfbf{\Theta}",
    u"\uD778": br"\mathsfbf{\Iota}",
    u"\uD779": br"\mathsfbf{\Kappa}",
    u"\uD77A": br"\mathsfbf{\Lambda}",
    u"\uD77D": br"\mathsfbf{\Xi}",
    u"\uD77F": br"\mathsfbf{\Pi}",
    u"\uD780": br"\mathsfbf{\Rho}",
    u"\uD781": br"\mathsfbf{\varsigma}",
    u"\uD782": br"\mathsfbf{\Sigma}",
    u"\uD783": br"\mathsfbf{\Tau}",
    u"\uD784": br"\mathsfbf{\Upsilon}",
    u"\uD785": br"\mathsfbf{\Phi}",
    u"\uD786": br"\mathsfbf{\Chi}",
    u"\uD787": br"\mathsfbf{\Psi}",
    u"\uD788": br"\mathsfbf{\Omega}",
    u"\uD789": br"\partial{}",
    u"\uD78A": br"\in",
    u"\uD78B": br"\mathsfbf{\vartheta}",
    u"\uD78C": br"\mathsfbf{\varkappa}",
    u"\uD78D": br"\mathsfbf{\phi}",
    u"\uD78E": br"\mathsfbf{\varrho}",
    u"\uD78F": br"\mathsfbf{\varpi}",
    u"\uD790": br"\mathsfbfsl{\Alpha}",
    u"\uD791": br"\mathsfbfsl{\Beta}",
    u"\uD792": br"\mathsfbfsl{\Gamma}",
    u"\uD793": br"\mathsfbfsl{\Delta}",
    u"\uD794": br"\mathsfbfsl{\Epsilon}",
    u"\uD795": br"\mathsfbfsl{\Zeta}",
    u"\uD796": br"\mathsfbfsl{\Eta}",
    u"\uD797": br"\mathsfbfsl{\vartheta}",
    u"\uD798": br"\mathsfbfsl{\Iota}",
    u"\uD799": br"\mathsfbfsl{\Kappa}",
    u"\uD79A": br"\mathsfbfsl{\Lambda}",
    u"\uD79D": br"\mathsfbfsl{\Xi}",
    u"\uD79F": br"\mathsfbfsl{\Pi}",
    u"\uD7A0": br"\mathsfbfsl{\Rho}",
    u"\uD7A1": br"\mathsfbfsl{\vartheta}",
    u"\uD7A2": br"\mathsfbfsl{\Sigma}",
    u"\uD7A3": br"\mathsfbfsl{\Tau}",
    u"\uD7A4": br"\mathsfbfsl{\Upsilon}",
    u"\uD7A5": br"\mathsfbfsl{\Phi}",
    u"\uD7A6": br"\mathsfbfsl{\Chi}",
    u"\uD7A7": br"\mathsfbfsl{\Psi}",
    u"\uD7A8": br"\mathsfbfsl{\Omega}",
    u"\uD7A9": br"\mathsfbfsl{\nabla}",
    u"\uD7AA": br"\mathsfbfsl{\Alpha}",
    u"\uD7AB": br"\mathsfbfsl{\Beta}",
    u"\uD7AC": br"\mathsfbfsl{\Gamma}",
    u"\uD7AD": br"\mathsfbfsl{\Delta}",
    u"\uD7AE": br"\mathsfbfsl{\Epsilon}",
    u"\uD7AF": br"\mathsfbfsl{\Zeta}",
    u"\uD7B0": br"\mathsfbfsl{\Eta}",
    u"\uD7B1": br"\mathsfbfsl{\vartheta}",
    u"\uD7B2": br"\mathsfbfsl{\Iota}",
    u"\uD7B3": br"\mathsfbfsl{\Kappa}",
    u"\uD7B4": br"\mathsfbfsl{\Lambda}",
    u"\uD7B7": br"\mathsfbfsl{\Xi}",
    u"\uD7B9": br"\mathsfbfsl{\Pi}",
    u"\uD7BA": br"\mathsfbfsl{\Rho}",
    u"\uD7BB": br"\mathsfbfsl{\varsigma}",
    u"\uD7BC": br"\mathsfbfsl{\Sigma}",
    u"\uD7BD": br"\mathsfbfsl{\Tau}",
    u"\uD7BE": br"\mathsfbfsl{\Upsilon}",
    u"\uD7BF": br"\mathsfbfsl{\Phi}",
    u"\uD7C0": br"\mathsfbfsl{\Chi}",
    u"\uD7C1": br"\mathsfbfsl{\Psi}",
    u"\uD7C2": br"\mathsfbfsl{\Omega}",
    u"\uD7C3": br"\partial{}",
    u"\uD7C4": br"\in",
    u"\uD7C5": br"\mathsfbfsl{\vartheta}",
    u"\uD7C6": br"\mathsfbfsl{\varkappa}",
    u"\uD7C7": br"\mathsfbfsl{\phi}",
    u"\uD7C8": br"\mathsfbfsl{\varrho}",
    u"\uD7C9": br"\mathsfbfsl{\varpi}",
    u"\uD7CE": br"\mathbf{0}",
    u"\uD7CF": br"\mathbf{1}",
    u"\uD7D0": br"\mathbf{2}",
    u"\uD7D1": br"\mathbf{3}",
    u"\uD7D2": br"\mathbf{4}",
    u"\uD7D3": br"\mathbf{5}",
    u"\uD7D4": br"\mathbf{6}",
    u"\uD7D5": br"\mathbf{7}",
    u"\uD7D6": br"\mathbf{8}",
    u"\uD7D7": br"\mathbf{9}",
    u"\uD7D8": br"\mathbb{0}",
    u"\uD7D9": br"\mathbb{1}",
    u"\uD7DA": br"\mathbb{2}",
    u"\uD7DB": br"\mathbb{3}",
    u"\uD7DC": br"\mathbb{4}",
    u"\uD7DD": br"\mathbb{5}",
    u"\uD7DE": br"\mathbb{6}",
    u"\uD7DF": br"\mathbb{7}",
    u"\uD7E0": br"\mathbb{8}",
    u"\uD7E1": br"\mathbb{9}",
    u"\uD7E2": br"\mathsf{0}",
    u"\uD7E3": br"\mathsf{1}",
    u"\uD7E4": br"\mathsf{2}",
    u"\uD7E5": br"\mathsf{3}",
    u"\uD7E6": br"\mathsf{4}",
    u"\uD7E7": br"\mathsf{5}",
    u"\uD7E8": br"\mathsf{6}",
    u"\uD7E9": br"\mathsf{7}",
    u"\uD7EA": br"\mathsf{8}",
    u"\uD7EB": br"\mathsf{9}",
    u"\uD7EC": br"\mathsfbf{0}",
    u"\uD7ED": br"\mathsfbf{1}",
    u"\uD7EE": br"\mathsfbf{2}",
    u"\uD7EF": br"\mathsfbf{3}",
    u"\uD7F0": br"\mathsfbf{4}",
    u"\uD7F1": br"\mathsfbf{5}",
    u"\uD7F2": br"\mathsfbf{6}",
    u"\uD7F3": br"\mathsfbf{7}",
    u"\uD7F4": br"\mathsfbf{8}",
    u"\uD7F5": br"\mathsfbf{9}",
    u"\uD7F6": br"\mathtt{0}",
    u"\uD7F7": br"\mathtt{1}",
    u"\uD7F8": br"\mathtt{2}",
    u"\uD7F9": br"\mathtt{3}",
    u"\uD7FA": br"\mathtt{4}",
    u"\uD7FB": br"\mathtt{5}",
    u"\uD7FC": br"\mathtt{6}",
    u"\uD7FD": br"\mathtt{7}",
    u"\uD7FE": br"\mathtt{8}",
    u"\uD7FF": br"\mathtt{9}",
}

# Some multi-character items in the source dataset that we can't
# translate. This is probably not exhaustive.
#    u"\u2009\u200A\u200A": r"\;",
#    u"\u2242\u0338": r"\NotEqualTilde{}",
#    u"\u224B\u0338": r"\not\apid{}",
#    u"\u224E\u0338": r"\NotHumpDownHump{}",
#    u"\u224F\u0338": r"\NotHumpEqual{}",
#    u"\u2250\u0338": r"\not\doteq",
#    u"\u2268\uFE00": r"\lvertneqq{}",
#    u"\u2269\uFE00": r"\gvertneqq{}",
#    u"\u226A\u0338": r"\NotLessLess{}",
#    u"\u226B\u0338": r"\NotGreaterGreater{}",
#    u"\u227E\u0338": r"\NotPrecedesTilde{}",
#    u"\u227F\u0338": r"\NotSucceedsTilde{}",
#    u"\u228A\uFE00": r"\varsubsetneqq{}",
#    u"\u228B\uFE00": r"\varsupsetneq{}",
#    u"\u228F\u0338": r"\NotSquareSubset{}",
#    u"\u2290\u0338": r"\NotSquareSuperset{}",
#    u"\u2933\u0338": r"\ElsevierGlyph{E21D}",
#    u"\u29CF\u0338": r"\NotLeftTriangleBar{}",
#    u"\u29D0\u0338": r"\NotRightTriangleBar{}",
#    u"\u2A7D\u0338": r"\nleqslant{}",
#    u"\u2A7E\u0338": r"\ngeqslant{}",
#    u"\u2AA1\u0338": r"\NotNestedLessLess{}",
#    u"\u2AA2\u0338": r"\NotNestedGreaterGreater{}",
#    u"\u2AAF\u0338": r"\not\preceq{}",
#    u"\u2AB0\u0338": r"\not\succeq{}",
#    u"\u2AC5\u0338": r"\nsubseteqq{}",
#    u"\u2AC6\u0338": r"\nsupseteqq",
#    u"\u2AFD\u20E5": r"{\rlap{\textbackslash}{{/}\!\!{/}}}",

unicode_to_latex_table = dict ((ord (k), text_type (v))
                               for k, v in unicode_to_latex_table_base.iteritems ())
unicode_to_latex_string = lambda u: u.translate (unicode_to_latex_table)
unicode_to_latex = lambda u: u.translate (unicode_to_latex_table).encode ('ascii')
