import socket
import struct
import itertools
import select

SERVERDATA_AUTH = 3
SERVERDATA_AUTH_RESPONSE = 2
SERVERDATA_EXECCOMMAND = 2
SERVERDATA_RESPONSE_VALUE = 0
SERVERDATA_RESPONSE_MAGICK = 4
PACKET_END = '\x00'

VALID_TYPES = (SERVERDATA_RESPONSE_VALUE, SERVERDATA_EXECCOMMAND, SERVERDATA_AUTH, SERVERDATA_RESPONSE_MAGICK)

MIN_PACKET_LEN = 10


class RconException(Exception):
    pass


class Rcon(object):
    """
        Example:
        import rcon
        server_ip = '1.1.1.1'
        server_port = 1111
        rcon_password = 'some_passwd'
        server = rcon.Rcon(server_ip, server_port, rcon_password)
        server.connect()
        server.authorize()
        print server.rcon('status')
    """

    def __init__(self, host, port=27015, password='', timeout=1.0):
        self.host = host
        self.port = port
        self.password = password
        self.timeout = timeout
        self.socket = False
        self.cur_request_id = 1

    def disconnect(self):
        """Disconnect from the server."""

        if self.socket:
            self.socket.close()

    def connect(self):
        """Connect to the server. Should only be used internally."""

        self.socket = socket.socket(socket.AF_INET, socket.SOCK_STREAM)
        self.socket.settimeout(self.timeout)
        self.socket.connect((self.host, self.port))

    def authorize(self):
        self.send(SERVERDATA_AUTH, self.password)
        msg_type, msg_body = self.receive()
        if msg_type != SERVERDATA_AUTH_RESPONSE:
            msg_type, msg_body = self.receive()
            if msg_type != SERVERDATA_AUTH_RESPONSE:
                self.disconnect()
                raise RconException('Authentication failure: %s:%s' % (repr(msg_type), repr(msg_body)))

    def send(self, msg_type, msg_body):
        """
            Send command and message to the server.
            Should only be used internally.
        """

        msg_id = struct.pack('<l', self.cur_request_id)
        msg_type = struct.pack('<l', msg_type)
        packet = msg_id + msg_type + msg_body + PACKET_END + PACKET_END
        packet_len = struct.pack('<l', len(packet))
        self.socket.send(packet_len + packet)

    def receive(self):
        """Receive a reply from the server. Should only be used internally."""

        packetsize_raw = self.socket.recv(4)
        if len(packetsize_raw) < 4:
            raise RconException('Not valid packet len: %s' % packetsize_raw)
        packetsize = struct.unpack('<l', packetsize_raw)[0]
        if packetsize < MIN_PACKET_LEN:
            raise RconException('Very small packet: %d' % packetsize)

        data = self.socket.recv(packetsize)
        msg_id, msg_type, msg_body, msg_end = struct.unpack('<l', data[:4])[0], struct.unpack('<l', data[4:8])[0], data[8:-1], data[-1]

        if msg_id == -1:
            raise RconException('Not valid RCON password')

        if msg_id == 0 and msg_type == SERVERDATA_RESPONSE_MAGICK:
            return SERVERDATA_RESPONSE_MAGICK, None

        if msg_id != self.cur_request_id:
            raise RconException('Not valid packet id: %d' % msg_id)

        if msg_type not in VALID_TYPES:
            raise RconException('Not valid packet type: %d' % msg_type)

        if len(msg_body) == 0:
            raise RconException('Not valid packet body len: 0')

        if msg_end != PACKET_END:
            raise RconException('Not valid packet end: %s' % repr(msg_end))

        poll = select.select([self.socket], [], [], 0)

        return msg_type, msg_body

    def _rcon(self, command):
        self.send(SERVERDATA_EXECCOMMAND, command)
        return self.receive()

    def rcon(self, command):
        """Send RCON command"""

        msg_type, msg_body = self._rcon(command)
        if msg_type == SERVERDATA_RESPONSE_MAGICK:
            return self._rcon(command)

        return msg_body
