# -*- coding: utf-8 -*-
from __future__ import (division, print_function, unicode_literals)
#############################################################################
# Author  : Jerome ODIER, Jerome FULACHIER, Fabian LAMBERT, Solveig ALBRAND
#
# Email   : jerome.odier@lpsc.in2p3.fr
#           jerome.fulachier@lpsc.in2p3.fr
#           fabian.lambert@lpsc.in2p3.fr
#           solveig.albrand@lpsc.in2p3.fr
#
# Version : 5.X.X (2014)
#
#############################################################################

import sys, base64, getpass, pyAMI.utils, pyAMI.config, pyAMI.object, pyAMI.parser, pyAMI.exception, pyAMI.transform, pyAMI.httpclient

if sys.version_info[0] == 3:
	import urllib.parse as urllib_parse
else:
	import    urllib    as urllib_parse

#############################################################################

class Client(object):
	#####################################################################

	def __init__(self, endpoint, xslt_file = '', key_file = '', cert_file = '', ignore_proxy = False, verbose = False):
		#############################################################

		self.config = pyAMI.config.Config(
			endpoint,
			xslt_file = xslt_file,
			key_file = key_file,
			cert_file = cert_file,
			ignore_proxy = ignore_proxy
		)

		self.config.read()

		#############################################################

		self.httpclient = pyAMI.httpclient.HttpClient(self.config)

		#############################################################

		self.verbose = verbose

	#####################################################################

	def auth(self, AMIUser = None, AMIPass = None, AMIJSID = ''):
		#############################################################
		# READ INFO                                                 #
		#############################################################

		if not AMIUser or not AMIPass:
			AMIUser = std_input('AMIUser: ')
			AMIPass = xxx_input('AMIPass: ')

		self.config.set('AMI', 'user', AMIUser)
		self.config.set('AMI', 'pass', AMIPass)
		self.config.set('AMI', 'jsid', AMIJSID)

		#############################################################
		# LOG-IN                                                    #
		#############################################################

		for user in self.execute('GetUserInfo -amiLogin=%s' % AMIUser, format = 'dict_object').get_rows_i('User_Info'):

			pyAMI.utils.safeprint('  First name: %s' % user['firstName'])
			pyAMI.utils.safeprint('  Last name: %s' % user['lastName'])
			pyAMI.utils.safeprint('  Email: %s' % user['mail'])

	#####################################################################

	def reset(self):
		self.config.reset()
		self.config.write()

	#####################################################################

	def execute(self, command, format = 'text', AMIUser = None, AMIPass = None, raise_errors = True):
		#############################################################
		# GET CONVERTER                                             #
		#############################################################

		try:
			converter = pyAMI.config.formats[format]['converter']
			transform = pyAMI.config.formats[format]['transform']

		except KeyError:
			raise pyAMI.exception.Error('invalid format `%s`, not in [%s]' % (
				format,
				', '.join(['`%s`' % x for x in list(pyAMI.config.formats.keys())]),
			))

		#############################################################
		# BUILD COMMAND                                             #
		#############################################################

		command = pyAMI.parser.parse(command)

		#############################################################
		# GET AMIUser AND AMIPass                                   #
		#############################################################

		if not AMIUser\
		   or         \
		   not AMIPass:
			AMIUser = self.config.get('AMI', 'user')
			AMIPass = self.config.get('AMI', 'pass')

			AMIPass = decode(AMIPass)

		#############################################################
		# BUILD REQUEST                                             #
		#############################################################

		data = {'Converter': converter}

		#############################################################

		if command.startswith('UploadProxy'):

			data['Proxy'] = self.config.cert_file_content

		else:

			data['Command'] = '%s -AMIUser=%s -AMIPass=%s' % (command, AMIUser, AMIPass)    \
			                                                                                \
				if command                                                              \
				   and                                                                  \
				   AMIUser         	                                                \
				   and                                                                  \
				   AMIPass else command

		#############################################################

		DATA = urllib_parse.urlencode(data)

		#############################################################
		# DO REQUEST                                                #
		#############################################################

		self.httpclient.connect()

		#############################################################

		if self.verbose:

			if   'Command' in data:

				## SAFETY ##
				data['Command'] = command
				## SAFETY ##

				print('URL     : %s://%s:%s%s?%s' % (
					self.httpclient.endpoint['prot'],
					self.httpclient.endpoint['host'],
					self.httpclient.endpoint['port'],
					self.httpclient.endpoint['path'],
					urllib_parse.urlencode(data)
				))

				print('Details :')
				print('  Session -> %s' % self.config.jsid)
				print('  Key file -> %s' % self.config.key_file)
				print('  Cert file -> %s' % self.config.cert_file)
				print('  Conn. mode -> %s' % self.config.conn_mode_str)
				print('')
				print('  Command -> %s' % data['Command'])
				print('  Converter -> %s' % data['Converter'])
				print('')

			elif 'Proxy' in data:

				print('Details :')
				print('  Session -> %s' % self.config.jsid)
				print('  Key file -> %s' % self.config.key_file)
				print('  Cert file -> %s' % self.config.cert_file)
				print('  Conn. mode -> %s' % self.config.conn_mode_str)
				print('')
				print('  Proxy ->\n%s' % data['Proxy'])
				print('  Converter -> %s' % data['Converter'])
				print('')

		#############################################################

		try:
			result = self.httpclient.request(DATA).read().decode('utf-8')

		finally:
			self.httpclient.close()

		#############################################################
		# FORMAT RESULT                                             #
		#############################################################

		if   transform == 'custom':
			return pyAMI.transform.transform(self.config.xslt_file, result)

		elif transform == 'dom_object':
			return pyAMI.object.DOMObject(result, raise_errors = raise_errors)

		elif transform == 'dict_object':
			return pyAMI.object.DICTObject(result, raise_errors = raise_errors)

		#############################################################

		return result

#############################################################################
# SECURITY                                                                  #
#############################################################################

def safe_b64encode(s):

	if sys.version_info < (2, 7):
		s = buffer(s)

	return base64.b64encode(s)

#############################################################################

def safe_b64decode(s):

	if sys.version_info < (2, 7):
		s = buffer(s)

	return base64.b64decode(s)

#############################################################################

USER_STR = str(getpass.getuser())
USER_LEN = len(getpass.getuser())

#############################################################################

def ker(s):
	return bytearray([ord(USER_STR[i % USER_LEN]) ^ c for i, c in enumerate(bytearray(s))])

#############################################################################

def encode(s):
	return safe_b64encode(ker(s.encode('utf-8'))).decode('utf-8')

#############################################################################

def decode(s):
	return ker(safe_b64decode(s.encode('utf-8'))).decode('utf-8')

#############################################################################

def std_input(prompt = ''):

	if sys.version_info[0] == 3:
		return input(prompt)
	else:
		return raw_input(prompt.encode(sys.stdout.encoding)).decode('utf-8')

#############################################################################

def xxx_input(prompt):
	#####################################################################
	# READ PASSWORD                                                     #
	#####################################################################

	if not type(sys.stdout).__module__.startswith('IPython'):

		if sys.version_info[0] == 3:
			password = getpass.getpass(prompt)
		else:
			password = getpass.getpass(prompt.encode(sys.stdout.encoding)).decode('utf-8')

	else:
		sys.stdout.write(prompt)

		sys.stdout.write('\033[40;m')
		password = std_input()
		sys.stdout.write('\033[0;m')

	#####################################################################
	# ENCODE PASSWORD                                                   #
	#####################################################################

	return encode(password)

#############################################################################
