import os
import yaml
from pyff.utils import resource_string
import logging

"""
Pipes and plumbing

Plumbing instances are sequences of pipes. Each pipe is called in order to load, select, transform
and output SAML metadata.
"""

__author__ = 'leifj'

class PipeLoader(object):
    """
A utility class for dynamically loading the parts of a plumbing instance. Each part (aka pipe)
is a module with a single method: *run(md,t,name,args,id)*.

The *md* argument is an MDRepository instance that stores the current set of known entitydescriptors
and entitydescriptor sets.

The *t* argument holds the current ElementTree representing the EntityDescriptors listing the selected
EntityDescriptor(s). This variable is None until select is passed in the plumbing.

The *name* argument is the @Name attribute of the EntityDescriptor set

The *args* argument is the list or dict of arguments to the pipe from the plumbing config.

The *id* argument is an identifier of the plumbing instance. This is typically the filename of the
plumbing config with the extension stripped off but in the case of a *fork*ed plumbing this may be
None or an autogenerated id.

The pipe may return a transformed copy of or replacement for t. The return value must be an
instance of ElementTree.
    """

    def load_pipe(self,d):
        """
Return a triple module,name,args of the pipe specified by the object d. The following alternatives
for d are allowed:

 - d is a string (or unicode) in which case the pipe is named d called with None as args.
 - d is a dict of the form {name: args} (i.e one key) in which case the pipe named *name* is called with args
 - d is an iterable (eg tuple or list) in which case d[0] is treated as the pipe name and d[1:] becomes the args
        """
        name = None
        args = None
        if type(d) is str or type(d) is unicode:
            name = d
        elif hasattr(d,'__iter__') and not type(d) is dict:
            if not len(d):
                raise Exception,"This does not look like a length of pipe... \n%s" % repr(d)
            print d
            name = d[0]
            args = d[1:]
        elif type(d) is dict:
            name = d.keys()[0]
            args = d[name]
        else:
            raise Exception,"This does not look like a length of pipe... \n%s" % repr(d)

        if name is None:
            raise Exception,"Anonymous length of pipe... \n%s" % repr(d)

        return __import__("pyff.pipes.%s" % name, fromlist=["pyff.pipes"]),name,args

class Plumbing(object):
    """
A plumbing instance represents a basic processing chaing for SAML metadata. A basic example:

.. code-block:: yaml

    - local:
       - /var/metadata/registry

    - select:
       - #md:EntityDescriptor[md:IDPSSODescriptor]

    - xslt:
        stylesheet: tidy.xsl

    - xslt:
        stylesheet: pp.xsl

    - fork:

       - xslt:
           stylesheet: publish.xsl
           Name: http://example.com/metadata.xml
           cacheDuration: PT1H
           validUntil: 30d

        - sign:
           key: signer.key
           cert: signer.crt

       - publish:
           output: /var/metadata/public/metadata.xml

Running this plumbing would bake all metadata found in /var/metadata/registry
into an EntitiesDescriptor element with @Name http://example.com/metadata.xml,
cacheDuration 1hr, validUntil 1 day from now. The tree woud be transformed
using the "tidy" and "pp" (for pretty-print) stylesheets and would then be
signed (using signer.key) and finally published in /var/metadata/public/metadata.xml
    """
    def __init__(self,pipeline,id):
        self.id = id
        self.pipeline = pipeline

    def __iter__(self):
        return self.pipeline

    def process(self,md):
        logging.debug('Processing %s' % self)
        t = None
        for p in self.pipeline:
            pipe,name,args = loader.load_pipe(p)
            logging.debug("Found pipe %s,%s,%s" % (pipe,name,args))
            ot = pipe.run(md,t,name,args,self.id)
            if ot is not None:
                t = ot
        return t

def plumbing(fn):
    id = os.path.splitext(fn)[0]
    ystr = resource_string(fn)
    pipeline = yaml.safe_load(ystr)

    return Plumbing(pipeline=pipeline,id=id)

loader = PipeLoader()