"""
pyFlies code generators transforms pyFlies experiment models to
different run-time platforms.
"""

import os
from os.path import join, dirname, isdir, exists
import imp
import shutil

from pyflies.exceptions import PyFliesException

PYFLIES_GENERATED = '.pyflies.generated'
MANUAL_DIR = 'manual'

# A dict of generator modules keyed by generator names
_generators = {}


def copytree(src, dst, symlinks=False, ignore=None):
    """
    shutil.copytree replacement from
    http://stackoverflow.com/questions/1868714/how-do-i-copy-an-entire-directory-of-files-into-an-existing-directory-using-pyth
    """
    if not os.path.exists(dst):
        os.makedirs(dst)
    for item in os.listdir(src):
        s = os.path.join(src, item)
        d = os.path.join(dst, item)
        if os.path.isdir(s):
            shutil.copytree(s, d, symlinks, ignore)
        else:
            if not d.endswith('template'):
                shutil.copy2(s, d)


def load_generators():
    """
    Search for and loads all generators.
    """
    if not _generators:
        generators_dir = join(dirname(__file__))

        # Each dir in generators folder is considered to be a generator for
        # particular platform. The name of the generator is defined in
        # the generator __init__ module
        dir_names = [f for f in os.listdir(generators_dir)
                     if f != '__pycache__' and
                     isdir(join(dirname(__file__), f))]
        for gen in dir_names:
            gen_module = __import__('pyflies.generators.{}'.format(gen),
                                    fromlist=["*"])
            _generators[gen_module.name] = (gen, gen_module)


def generator_names():
    """
    Return a list of generator names.
    """
    load_generators()
    return _generators.keys()


def generate(model, target):
    """
    Do some sanity checks for target folder. Delete the content of
    target folder leaving manually written files, and calls generator.

    Args:
        model(pyFlies model):
        target(Target): An object that describe target platform.
    """

    generator_name = target.name
    target_folder = target.output

    # Sanity check. Target folder will be deleted only if it has
    # .pyflies.generated
    # file inside
    # Else an exception will be raised.
    pyflies_gen = join(target_folder, PYFLIES_GENERATED)
    if exists(target_folder):
        target_folder_list = os.listdir(target_folder)
        # If the target folder is empty or is generated by pyflies
        # it is OK to proceed.
        if target_folder_list == [] or exists(pyflies_gen):
            # Remove content of target folder. Leave src-gen folder and
            # pyflies.generated file
            for to_delete in os.listdir(target_folder):
                if to_delete not in [MANUAL_DIR, PYFLIES_GENERATED]:
                    full_name = join(target_folder, to_delete)
                    if isdir(full_name):
                        shutil.rmtree(full_name)
                    else:
                        os.remove(full_name)
        else:
            raise PyFliesException(
                "Target folder does not have a '.pyflies.generated' file " +
                "and is not empty!" +
                " It is not generated by pyflies.")
    else:
        os.makedirs(join(target_folder, MANUAL_DIR))

    open(join(target_folder, '.pyflies.generated'), 'w').close()

    # Copy non-template files
    copytree(join(dirname(__file__),
             _generators[generator_name][0], 'templates'),
             target_folder,
             ignore=shutil.ignore_patterns('*.template'))

    # Copy all files referenced from model
    for b in model.blocks:
        if b._typename == "TestType":
            for cs in b.stimuli.condStimuli:
                stimuli = cs.stimuli
                for stimulus in stimuli:
                    if stimulus._typename in ["Audio", "Image"]:
                        if stimulus.file:
                            dir_part = dirname(stimulus.file)
                            file_target_dir = join(target_folder, dir_part)
                            if not os.path.exists(file_target_dir):
                                os.makedirs(file_target_dir)

                            # Source files are relative to model file
                            model_dir = dirname(model._filename)
                            src_file = join(model_dir, stimulus.file)
                            dst_file = join(target_folder, stimulus.file)
                            shutil.copyfile(src_file, dst_file)

    # Run generator
    _generators[generator_name][1].generate(model, target)
