#!/usr/bin/env python
# -*- coding: utf-8  -*-
################################################################################
#
#  pyLOCSMS -- Python Interface to LOC Store Management Suite
#  Copyright © 2013 Lance Edgar
#
#  This file is part of pyLOCSMS.
#
#  pyLOCSMS is free software: you can redistribute it and/or modify it under
#  the terms of the GNU General Public License as published by the Free
#  Software Foundation, either version 3 of the License, or (at your option)
#  any later version.
#
#  pyLOCSMS is distributed in the hope that it will be useful, but WITHOUT ANY
#  WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
#  FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
#  details.
#
#  You should have received a copy of the GNU General Public License along with
#  pyLOCSMS.  If not, see <http://www.gnu.org/licenses/>.
#
################################################################################


from sqlalchemy.ext.declarative import declarative_base
from sqlalchemy import Column, String, Integer, Float, Numeric, DateTime
from sqlalchemy.orm import relationship
from sqlalchemy.orm.collections import attribute_mapped_collection


__all__ = ['Store', 'Department', 'Subdepartment',
           'Product', 'POS', 'PerpetualInventory']


Base = declarative_base()


class Store(Base):
    """
    Represents a store.
    """

    __tablename__ = 'STD_TAB'

    number = Column(
        'F1056', String(length=4), primary_key=True, nullable=False)

    long_number = Column(
        'F1530', String(length=20), nullable=True)

    name = Column(
        'F1531', String(length=50), nullable=True)

    address_1 = Column(
        'F1532', String(length=50), nullable=True)

    address_2 = Column(
        'F1533', String(length=50), nullable=True)

    city = Column(
        'F1534', String(length=50), nullable=True)

    zipcode = Column(
        'F1535', String(length=15), nullable=True)

    phone = Column(
        'F1536', String(length=20), nullable=True)

    fax = Column(
        'F1537', String(length=20), nullable=True)

    email = Column(
        'F1538', String(length=50), nullable=True)

    class_ = Column(
        'F1539', String(length=6), nullable=True)

    group = Column(
        'F1540', String(length=6), nullable=True)

    location_type = Column(
        'F1541', String(length=6), nullable=True)

    region = Column(
        'F1542', String(length=6), nullable=True)

    manager = Column(
        'F1543', Integer(), nullable=True)

    interim = Column(
        'F1544', Integer(), nullable=True)

    man_hours_week_1 = Column(
        'F1545', Float(), nullable=True)

    opening_date = Column(
        'F1546', DateTime(), nullable=True)

    closing_date = Column(
        'F1547', DateTime(), nullable=True)

    type = Column(
        'F1548', String(length=6), nullable=True)

    province = Column(
        'F1549', String(length=15), nullable=True)

    eligible = Column(
        'F1551', String(length=4), nullable=True)

    man_hours_week_2 = Column(
        'F1579', Float(), nullable=True)

    man_hours_week_3 = Column(
        'F1580', Float(), nullable=True)

    long_number_2 = Column(
        'F2688', String(length=20), nullable=True)

    accounting_prefix = Column(
        'F2689', String(length=20), nullable=True)

    accounting_prefix_2 = Column(
        'F2690', String(length=20), nullable=True)

    backstore_capacity = Column(
        'F2698', Float(), nullable=True)

    bank_account = Column(
        'F2774', String(length=20), nullable=True)

    bank_account_2 = Column(
        'F2775', String(length=20), nullable=True)

    camera_server_address = Column(
        'F2776', String(length=60), nullable=True)

    camera_terminal_link = Column(
        'F2777', String(length=240), nullable=True)

    def __repr__(self):
        return "Store(number={0})".format(repr(self.number))

    def __str__(self):
        return str(self.name or '')


class Department(Base):
    """
    Represents a department.
    """

    __tablename__ = 'DEPT_TAB'

    code = Column(
        'F03', Integer(), primary_key=True, nullable=False)

    price_margin = Column(
        'F49', Float(), nullable=True)

    require_validation = Column(
        'F78', String(length=1), nullable=True)

    food_stamp = Column(
        'F79', String(length=1), nullable=True)

    trading_stamp = Column(
        'F80', String(length=1), nullable=True)

    tax_flag_1 = Column(
        'F81', String(length=1), nullable=True)

    scalable_item = Column(
        'F82', String(length=1), nullable=True)

    store_coupon = Column(
        'F88', String(length=1), nullable=True)

    tax_flag_2 = Column(
        'F96', String(length=1), nullable=True)

    tax_flag_3 = Column(
        'F97', String(length=1), nullable=True)

    tax_flag_4 = Column(
        'F98', String(length=1), nullable=True)

    tax_flag_5 = Column(
        'F99', String(length=1), nullable=True)

    tax_flag_6 = Column(
        'F100', String(length=1), nullable=True)

    tax_flag_7 = Column(
        'F101', String(length=1), nullable=True)

    vendor_coupon = Column(
        'F104', String(length=1), nullable=True)

    coupon_restricted = Column(
        'F108', String(length=1), nullable=True)

    allow_price_override = Column(
        'F114', String(length=1), nullable=True)

    not_in_net_sale = Column(
        'F115', String(length=1), nullable=True)

    allow_manual_weight = Column(
        'F121', String(length=1), nullable=True)

    prohibit_returns = Column(
        'F124', String(length=1), nullable=True)

    prohibit_discount = Column(
        'F150', String(length=1), nullable=True)

    restriction_code = Column(
        'F170', Integer(), nullable=True)

    minimum_age_customer = Column(
        'F171', Integer(), nullable=True)

    prohibit_refund = Column(
        'F172', String(length=1), nullable=True)

    not_in_admissible_spending = Column(
        'F177', String(length=1), nullable=True)

    wic_eligible = Column(
        'F178', String(length=1), nullable=True)

    description = Column(
        'F238', String(length=30), nullable=True)

    maximum_amount = Column(
        'F239', Numeric(precision=8, scale=4), nullable=True)

    minimum_amount = Column(
        'F240', Numeric(precision=8, scale=4), nullable=True)

    maximum_void = Column(
        'F241', Numeric(precision=8, scale=4), nullable=True)

    maximum_refund = Column(
        'F242', Numeric(precision=8, scale=4), nullable=True)

    pos_specific_flags = Column(
        'F1120', String(length=12), nullable=True)

    known_shrink_factor = Column(
        'F1123', Float(), nullable=True)

    commission_rate = Column(
        'F1124', Float(), nullable=True)

    group = Column(
        'F1132', String(length=20), nullable=True)

    minimum_age_operator = Column(
        'F1139', Integer(), nullable=True)

    sequence_number = Column(
        'F1147', Integer(), nullable=True)

    operator_responsible = Column(
        'F1168', Integer(), nullable=True)

    discount = Column(
        'F1256', Float(), nullable=True)

    behavior = Column(
        'F1785', String(length=12), nullable=True)

    alternate_description = Column(
        'F1894', String(length=30), nullable=True)

    show_priority = Column(
        'F1965', Integer(), nullable=True)

    show_filter = Column(
        'F1966', String(length=20), nullable=True)

    url = Column(
        'F2660', String(length=250), nullable=True)

    def __repr__(self):
        return "Department(code={0})".format(repr(self.code))

    def __str__(self):
        return str(self.description or '')


class Subdepartment(Base):
    """
    Represents a subdepartment.
    """

    __tablename__ = 'SDP_TAB'

    code = Column(
        'F04', Integer(), primary_key=True, nullable=False)

    department_code = Column(
        'F03', Integer(), nullable=True)

    price_margin = Column(
        'F49', Float(), nullable=True)

    require_validation = Column(
        'F78', String(length=1), nullable=True)

    food_stamp = Column(
        'F79', String(length=1), nullable=True)

    trading_stamp = Column(
        'F80', String(length=1), nullable=True)

    tax_flag_1 = Column(
        'F81', String(length=1), nullable=True)

    scalable_item = Column(
        'F82', String(length=1), nullable=True)

    store_coupon = Column(
        'F88', String(length=1), nullable=True)

    tax_flag_2 = Column(
        'F96', String(length=1), nullable=True)

    tax_flag_3 = Column(
        'F97', String(length=1), nullable=True)

    tax_flag_4 = Column(
        'F98', String(length=1), nullable=True)

    tax_flag_5 = Column(
        'F99', String(length=1), nullable=True)

    tax_flag_6 = Column(
        'F100', String(length=1), nullable=True)

    tax_flag_7 = Column(
        'F101', String(length=1), nullable=True)

    vendor_coupon = Column(
        'F104', String(length=1), nullable=True)

    coupon_restricted = Column(
        'F108', String(length=1), nullable=True)

    allow_price_override = Column(
        'F114', String(length=1), nullable=True)

    not_in_net_sale = Column(
        'F115', String(length=1), nullable=True)

    allow_manual_weight = Column(
        'F121', String(length=1), nullable=True)

    prohibit_returns = Column(
        'F124', String(length=1), nullable=True)

    prohibit_discount = Column(
        'F150', String(length=1), nullable=True)

    restriction_code = Column(
        'F170', Integer(), nullable=True)

    minimum_age_customer = Column(
        'F171', Integer(), nullable=True)

    prohibit_refund = Column(
        'F172', String(length=1), nullable=True)

    not_in_admissible_spending = Column(
        'F177', String(length=1), nullable=True)

    wic_eligible = Column(
        'F178', String(length=1), nullable=True)

    maximum_amount = Column(
        'F239', Numeric(precision=8, scale=4), nullable=True)

    minimum_amount = Column(
        'F240', Numeric(precision=8, scale=4), nullable=True)

    maximum_void = Column(
        'F241', Numeric(precision=8, scale=4), nullable=True)

    maximum_refund = Column(
        'F242', Numeric(precision=8, scale=4), nullable=True)

    description = Column(
        'F1022', String(length=30), nullable=True)

    pos_specific_flags = Column(
        'F1120', String(length=12), nullable=True)

    known_shrink_factor = Column(
        'F1123', Float(), nullable=True)

    commission_rate = Column(
        'F1124', Float(), nullable=True)

    minimum_age_operator = Column(
        'F1139', Integer(), nullable=True)

    sequence_number = Column(
        'F1147', Integer(), nullable=True)

    operator_responsible = Column(
        'F1168', Integer(), nullable=True)

    discount = Column(
        'F1256', Float(), nullable=True)

    behavior = Column(
        'F1785', String(length=12), nullable=True)

    alternate_description = Column(
        'F1893', String(length=30), nullable=True)

    cost_plus_percent = Column(
        'F1938', Float(), nullable=True)

    show_priority = Column(
        'F1965', Integer(), nullable=True)

    show_filter = Column(
        'F1966', String(length=20), nullable=True)

    url = Column(
        'F2660', String(length=250), nullable=True)

    def __repr__(self):
        return "Subdepartment(code={0})".format(repr(self.code))


Department.subdepartments = relationship(
    Subdepartment,
    primaryjoin=Subdepartment.department_code == Department.code,
    foreign_keys=[Subdepartment.department_code],
    uselist=True,
    backref='department')


class Product(Base):
    """
    Represents a product (main item record).
    """

    __tablename__ = 'OBJ_TAB'

    upc = Column(
        'F01', String(length=13), primary_key=True, nullable=False)

    family_code = Column(
        'F16', Integer(), nullable=True)

    category_code = Column(
        'F17', Integer(), nullable=True)

    report_code = Column(
        'F18', Integer(), nullable=True)

    account_code = Column(
        'F93', String(length=16), nullable=True)

    reporting_department = Column(
        'F193', String(length=16), nullable=True)

    batch_id = Column(
        'F902', String(length=8), nullable=True)

    record_status = Column(
        'F1001', Integer(), nullable=True)

    upc_code_format = Column(
        'F07', Integer(), nullable=True)

    measurement_system = Column(
        'F11', Integer(), nullable=True)

    size_height = Column(
        'F12', Float(), nullable=True)

    size_width = Column(
        'F13', Float(), nullable=True)

    size_depth = Column(
        'F14', Float(), nullable=True)

    measure_sell_pack = Column(
        'F21', Float(), nullable=True)

    size_description = Column(
        'F22', String(length=30), nullable=True)

    measure_description = Column(
        'F23', String(length=10), nullable=True)

    expanded_description = Column(
        'F29', String(length=60), nullable=True)

    brand_description = Column(
        'F155', String(length=30), nullable=True)

    manufacturer_code = Column(
        'F180', String(length=20), nullable=True)

    duns_number_plus_suffix = Column(
        'F213', String(length=13), nullable=True)

    alias_code = Column(
        'F214', String(length=13), nullable=True)

    alias_code_format = Column(
        'F215', Integer(), nullable=True)

    comparable_size_unit_of_measure_description = Column(
        'F218', String(length=4), nullable=True)

    date_start = Column(
        'F253', DateTime(), nullable=True)

    long_description = Column(
        'F255', String(length=120), nullable=True)

    weight_net = Column(
        'F270', Float(), nullable=True)

    target_id = Column(
        'F1000', String(length=5), nullable=True)

    size_cubic = Column(
        'F1002', Float(), nullable=True)

    container_type = Column(
        'F1004', Integer(), nullable=True)

    manufacturer_id = Column(
        'F1118', String(length=9), nullable=True)

    graphic_file = Column(
        'F1119', String(length=30), nullable=True)

    operator_responsible = Column(
        'F1168', Integer(), nullable=True)

    shipping_piece_count = Column(
        'F1699', Float(), nullable=True)

    best_before_days = Column(
        'F1736', Integer(), nullable=True)

    medical_product_code = Column(
        'F1737', String(length=13), nullable=True)

    ndc_din_number = Column(
        'F1738', String(length=13), nullable=True)

    measure_weight_or_volume = Column(
        'F1744', Float(), nullable=True)

    maintenance_operator_level = Column(
        'F1759', Integer(), nullable=True)

    alternate_brand_description = Column(
        'F1939', String(length=30), nullable=True)

    alternate_expanded_description = Column(
        'F1940', String(length=60), nullable=True)

    alternate_size_description = Column(
        'F1941', String(length=30), nullable=True)

    alternate_long_description = Column(
        'F1942', String(length=120), nullable=True)

    classification = Column(
        'F1957', String(length=100), nullable=True)

    target_customer_type = Column(
        'F1958', String(length=20), nullable=True)

    target_store_type = Column(
        'F1959', String(length=20), nullable=True)

    handling_type = Column(
        'F1960', String(length=10), nullable=True)

    marketing_justification = Column(
        'F1962', String(length=250), nullable=True)

    store_responsible = Column(
        'F1964', String(length=4), nullable=True)

    item_substitution_policy = Column(
        'F2600', String(length=2), nullable=True)

    competitive_code = Column(
        'F2693', String(length=13), nullable=True)

    def __repr__(self):
        return "Product(upc={0})".format(repr(self.upc))

    @property
    def subdepartment(self):
        if 'PAL' in self.pos:
            return self.pos['PAL'].subdepartment
        return None

    @property
    def department(self):
        sub = self.subdepartment
        if sub:
            return sub.department
        return None


class POS(Base):
    """
    Represents various POS flags for a product.
    """

    __tablename__ = 'POS_TAB'

    upc = Column(
        'F01', String(length=13), primary_key=True, nullable=False)

    target_id = Column(
        'F1000', String(length=5), primary_key=True, nullable=False)

    subdepartment_code = Column(
        'F04', Integer(), nullable=True)

    promotion_code = Column(
        'F383', String(length=20), nullable=True)

    batch_id = Column(
        'F902', String(length=8), nullable=True)

    record_status = Column(
        'F1001', Integer(), nullable=True)

    description = Column(
        'F02', String(length=40), nullable=True)

    department_code = Column(
        'F03', Integer(), nullable=True)

    bottle_deposit_link = Column(
        'F05', Integer(), nullable=True)

    tare_weight_link = Column(
        'F06', Integer(), nullable=True)

    upc_code_format = Column(
        'F07', Integer(), nullable=True)

    status_code = Column(
        'F08', String(length=4), nullable=True)

    status_indicator_date = Column(
        'F09', DateTime(), nullable=True)

    weight_divisor = Column(
        'F24', Float(), nullable=True)

    item_pricing_required = Column(
        'F40', String(length=1), nullable=True)

    bottle_deposit_value = Column(
        'F50', Numeric(precision=8, scale=4), nullable=True)

    excise_tax_amount = Column(
        'F60', Numeric(precision=8, scale=4), nullable=True)

    tax_exempt_amount = Column(
        'F61', Numeric(precision=8, scale=4), nullable=True)

    sales_activity_level = Column(
        'F66', Integer(), nullable=True)

    coupon_family_code = Column(
        'F77', Integer(), nullable=True)

    require_validation = Column(
        'F78', String(length=1), nullable=True)

    food_stamp = Column(
        'F79', String(length=1), nullable=True)

    trading_stamp = Column(
        'F80', String(length=1), nullable=True)

    tax_flag_1 = Column(
        'F81', String(length=1), nullable=True)

    scalable_item = Column(
        'F82', String(length=1), nullable=True)

    require_price_entry = Column(
        'F83', String(length=1), nullable=True)

    require_visual_verify = Column(
        'F84', String(length=1), nullable=True)

    require_quantity = Column(
        'F85', String(length=1), nullable=True)

    not_for_sale_in_store = Column(
        'F86', String(length=1), nullable=True)

    restricted_sale = Column(
        'F87', String(length=1), nullable=True)

    store_coupon = Column(
        'F88', String(length=1), nullable=True)

    deposit_container_code = Column(
        'F92', String(length=1), nullable=True)

    tax_flag_2 = Column(
        'F96', String(length=1), nullable=True)

    tax_flag_3 = Column(
        'F97', String(length=1), nullable=True)

    tax_flag_4 = Column(
        'F98', String(length=1), nullable=True)

    tax_flag_5 = Column(
        'F99', String(length=1), nullable=True)

    tax_flag_6 = Column(
        'F100', String(length=1), nullable=True)

    tax_flag_7 = Column(
        'F101', String(length=1), nullable=True)

    prohibit_quantity = Column(
        'F102', String(length=1), nullable=True)

    vendor_coupon = Column(
        'F104', String(length=1), nullable=True)

    follow_subdepartment_status = Column(
        'F106', String(length=1), nullable=True)

    record_item_sale = Column(
        'F107', String(length=1), nullable=True)

    coupon_restricted = Column(
        'F108', String(length=1), nullable=True)

    electronic_coupon = Column(
        'F110', String(length=1), nullable=True)

    allow_price_override = Column(
        'F114', String(length=1), nullable=True)

    not_in_net_sale = Column(
        'F115', String(length=1), nullable=True)

    allow_manual_weight = Column(
        'F121', String(length=1), nullable=True)

    plu_code = Column(
        'F123', String(length=13), nullable=True)

    prohibit_returns = Column(
        'F124', String(length=1), nullable=True)

    prohibit_repeat_key = Column(
        'F125', String(length=1), nullable=True)

    link_quantity_limit = Column(
        'F141', Float(), nullable=True)

    coupon_multiplication = Column(
        'F149', String(length=1), nullable=True)

    prohibit_discount = Column(
        'F150', String(length=1), nullable=True)

    link_reason_code = Column(
        'F153', Integer(), nullable=True)

    deposit_item = Column(
        'F158', String(length=1), nullable=True)

    refund_item = Column(
        'F159', String(length=1), nullable=True)

    bottle_return = Column(
        'F160', String(length=1), nullable=True)

    misc_receipt = Column(
        'F161', String(length=1), nullable=True)

    misc_payout = Column(
        'F162', String(length=1), nullable=True)

    previous_coupon_family_code = Column(
        'F163', Integer(), nullable=True)

    restriction_code = Column(
        'F170', Integer(), nullable=True)

    minimum_age_customer = Column(
        'F171', Integer(), nullable=True)

    prohibit_refund = Column(
        'F172', String(length=1), nullable=True)

    prohibit_multiple_coupon = Column(
        'F173', String(length=1), nullable=True)

    tax_included = Column(
        'F174', String(length=1), nullable=True)

    keyed_department_override = Column(
        'F176', String(length=1), nullable=True)

    wic_eligible = Column(
        'F178', String(length=1), nullable=True)

    valid_item = Column(
        'F188', String(length=1), nullable=True)

    send_to_scale = Column(
        'F189', String(length=1), nullable=True)

    competitive_price_quantity = Column(
        'F209', Float(), nullable=True)

    competitive_price = Column(
        'F210', Numeric(precision=8, scale=4), nullable=True)

    competitive_price_start_date = Column(
        'F211', DateTime(), nullable=True)

    comparable_size = Column(
        'F217', Float(), nullable=True)

    date_start = Column(
        'F253', DateTime(), nullable=True)

    coupon_offer_code = Column(
        'F302', String(length=8), nullable=True)

    coupon_expiration_date = Column(
        'F303', DateTime(), nullable=True)

    coupon_household_id = Column(
        'F304', String(length=8), nullable=True)

    coupon_redemption_multiple = Column(
        'F306', Float(), nullable=True)

    price_multiple_as_quantity = Column(
        'F388', String(length=1), nullable=True)

    prompt_for_sku = Column(
        'F397', String(length=1), nullable=True)

    rentable_quantity = Column(
        'F1099', Integer(), nullable=True)

    specific_flags = Column(
        'F1120', String(length=12), nullable=True)

    commission_rate = Column(
        'F1124', Float(), nullable=True)

    ticket_template = Column(
        'F1136', String(length=8), nullable=True)

    tare_weight_proportional = Column(
        'F1138', Float(), nullable=True)

    minimum_age_operator = Column(
        'F1139', Integer(), nullable=True)

    allow_raincheck_ticket = Column(
        'F1236', String(length=1), nullable=True)

    scale_divisor = Column(
        'F1237', Float(), nullable=True)

    controlled_product_indicator = Column(
        'F1735', String(length=4), nullable=True)

    maintenance_operator_level = Column(
        'F1759', Integer(), nullable=True)

    behavior = Column(
        'F1785', String(length=12), nullable=True)

    replace_adding_function = Column(
        'F1787', Integer(), nullable=True)

    receipt_copy_count = Column(
        'F1788', Integer(), nullable=True)

    store_coupon_count = Column(
        'F1789', Float(), nullable=True)

    alternate_description = Column(
        'F1892', String(length=40), nullable=True)

    store_responsible = Column(
        'F1964', String(length=4), nullable=True)

    url = Column(
        'F2660', String(length=250), nullable=True)

    def __repr__(self):
        return "POS(upc={0}, target_id={1})".format(
            repr(self.upc), repr(self.target_id))


POS.subdepartment = relationship(
    Subdepartment,
    primaryjoin=Subdepartment.code == POS.subdepartment_code,
    foreign_keys=[Subdepartment.code],
    uselist=False)


Product.pos = relationship(
    POS,
    primaryjoin=POS.upc == Product.upc,
    foreign_keys=[POS.upc],
    uselist=True,
    collection_class=attribute_mapped_collection('target_id'),
    backref='product')


class PerpetualInventory(Base):
    """
    Represents the perpetual inventory (running total) for a product.
    """

    __tablename__ = 'RPT_ITM_N'

    upc = Column(
        'F01', String(length=13), primary_key=True, nullable=False)

    totalizer_number = Column(
        'F1034', Integer(), primary_key=True, nullable=False, default=8501)

    store_number = Column(
        'F1056', String(length=4), primary_key=True, nullable=False)

    total_units = Column(
        'F64', Float(), nullable=False, default=0)

    total_dollars = Column(
        'F65', Numeric(precision=8, scale=4), nullable=False, default=0)

    total_weight = Column(
        'F67', Float(), nullable=False, default=0)

    updated = Column(
        'F253', DateTime(), nullable=False)

    def __repr__(self):
        return "PerpetualInventory(upc={0}, store_number={1})".format(
            repr(self.upc), repr(self.store_number))


PerpetualInventory.product = relationship(
    Product,
    primaryjoin=Product.upc == PerpetualInventory.upc,
    foreign_keys=[Product.upc],
    uselist=False)

PerpetualInventory.store = relationship(
    Store,
    primaryjoin=Store.number == PerpetualInventory.store_number,
    foreign_keys=[Store.number],
    uselist=False)
