#!/usr/bin/env python
# -*- coding: utf-8  -*-
################################################################################
#
#  pyLOCSMS -- Python Interface to LOC Store Management Suite
#  Copyright © 2013 Lance Edgar
#
#  This file is part of pyLOCSMS.
#
#  pyLOCSMS is free software: you can redistribute it and/or modify it under
#  the terms of the GNU General Public License as published by the Free
#  Software Foundation, either version 3 of the License, or (at your option)
#  any later version.
#
#  pyLOCSMS is distributed in the hope that it will be useful, but WITHOUT ANY
#  WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
#  FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
#  details.
#
#  You should have received a copy of the GNU General Public License along with
#  pyLOCSMS.  If not, see <http://www.gnu.org/licenses/>.
#
################################################################################


from sqlalchemy.ext.declarative import declarative_base
from sqlalchemy import Column, String, Integer, Float, Numeric, DateTime
from sqlalchemy.orm import relationship
from sqlalchemy.orm.collections import attribute_mapped_collection


__all__ = ['Store', 'Department', 'Subdepartment',
           'Product', 'POS', 'PerpetualInventory',
           'Operator', 'Vendor', 'ReceivingDocument']


Base = declarative_base()


class Store(Base):
    """
    Represents a store.
    """

    __tablename__ = 'STD_TAB'

    number = Column(
        'F1056', String(length=4), primary_key=True, nullable=False)

    long_number = Column(
        'F1530', String(length=20), nullable=True)

    name = Column(
        'F1531', String(length=50), nullable=True)

    address_1 = Column(
        'F1532', String(length=50), nullable=True)

    address_2 = Column(
        'F1533', String(length=50), nullable=True)

    city = Column(
        'F1534', String(length=50), nullable=True)

    zipcode = Column(
        'F1535', String(length=15), nullable=True)

    phone = Column(
        'F1536', String(length=20), nullable=True)

    fax = Column(
        'F1537', String(length=20), nullable=True)

    email = Column(
        'F1538', String(length=50), nullable=True)

    class_ = Column(
        'F1539', String(length=6), nullable=True)

    group = Column(
        'F1540', String(length=6), nullable=True)

    location_type = Column(
        'F1541', String(length=6), nullable=True)

    region = Column(
        'F1542', String(length=6), nullable=True)

    manager = Column(
        'F1543', Integer(), nullable=True)

    interim = Column(
        'F1544', Integer(), nullable=True)

    man_hours_week_1 = Column(
        'F1545', Float(), nullable=True)

    opening_date = Column(
        'F1546', DateTime(), nullable=True)

    closing_date = Column(
        'F1547', DateTime(), nullable=True)

    type = Column(
        'F1548', String(length=6), nullable=True)

    province = Column(
        'F1549', String(length=15), nullable=True)

    eligible = Column(
        'F1551', String(length=4), nullable=True)

    man_hours_week_2 = Column(
        'F1579', Float(), nullable=True)

    man_hours_week_3 = Column(
        'F1580', Float(), nullable=True)

    long_number_2 = Column(
        'F2688', String(length=20), nullable=True)

    accounting_prefix = Column(
        'F2689', String(length=20), nullable=True)

    accounting_prefix_2 = Column(
        'F2690', String(length=20), nullable=True)

    backstore_capacity = Column(
        'F2698', Float(), nullable=True)

    bank_account = Column(
        'F2774', String(length=20), nullable=True)

    bank_account_2 = Column(
        'F2775', String(length=20), nullable=True)

    camera_server_address = Column(
        'F2776', String(length=60), nullable=True)

    camera_terminal_link = Column(
        'F2777', String(length=240), nullable=True)

    def __repr__(self):
        return "Store(number={0})".format(repr(self.number))

    def __str__(self):
        return str(self.name or '')


class Department(Base):
    """
    Represents a department.
    """

    __tablename__ = 'DEPT_TAB'

    code = Column(
        'F03', Integer(), primary_key=True, nullable=False)

    price_margin = Column(
        'F49', Float(), nullable=True)

    require_validation = Column(
        'F78', String(length=1), nullable=True)

    food_stamp = Column(
        'F79', String(length=1), nullable=True)

    trading_stamp = Column(
        'F80', String(length=1), nullable=True)

    tax_flag_1 = Column(
        'F81', String(length=1), nullable=True)

    scalable_item = Column(
        'F82', String(length=1), nullable=True)

    store_coupon = Column(
        'F88', String(length=1), nullable=True)

    tax_flag_2 = Column(
        'F96', String(length=1), nullable=True)

    tax_flag_3 = Column(
        'F97', String(length=1), nullable=True)

    tax_flag_4 = Column(
        'F98', String(length=1), nullable=True)

    tax_flag_5 = Column(
        'F99', String(length=1), nullable=True)

    tax_flag_6 = Column(
        'F100', String(length=1), nullable=True)

    tax_flag_7 = Column(
        'F101', String(length=1), nullable=True)

    vendor_coupon = Column(
        'F104', String(length=1), nullable=True)

    coupon_restricted = Column(
        'F108', String(length=1), nullable=True)

    allow_price_override = Column(
        'F114', String(length=1), nullable=True)

    not_in_net_sale = Column(
        'F115', String(length=1), nullable=True)

    allow_manual_weight = Column(
        'F121', String(length=1), nullable=True)

    prohibit_returns = Column(
        'F124', String(length=1), nullable=True)

    prohibit_discount = Column(
        'F150', String(length=1), nullable=True)

    restriction_code = Column(
        'F170', Integer(), nullable=True)

    minimum_age_customer = Column(
        'F171', Integer(), nullable=True)

    prohibit_refund = Column(
        'F172', String(length=1), nullable=True)

    not_in_admissible_spending = Column(
        'F177', String(length=1), nullable=True)

    wic_eligible = Column(
        'F178', String(length=1), nullable=True)

    description = Column(
        'F238', String(length=30), nullable=True)

    maximum_amount = Column(
        'F239', Numeric(precision=8, scale=4), nullable=True)

    minimum_amount = Column(
        'F240', Numeric(precision=8, scale=4), nullable=True)

    maximum_void = Column(
        'F241', Numeric(precision=8, scale=4), nullable=True)

    maximum_refund = Column(
        'F242', Numeric(precision=8, scale=4), nullable=True)

    pos_specific_flags = Column(
        'F1120', String(length=12), nullable=True)

    known_shrink_factor = Column(
        'F1123', Float(), nullable=True)

    commission_rate = Column(
        'F1124', Float(), nullable=True)

    group = Column(
        'F1132', String(length=20), nullable=True)

    minimum_age_operator = Column(
        'F1139', Integer(), nullable=True)

    sequence_number = Column(
        'F1147', Integer(), nullable=True)

    operator_responsible = Column(
        'F1168', Integer(), nullable=True)

    discount = Column(
        'F1256', Float(), nullable=True)

    behavior = Column(
        'F1785', String(length=12), nullable=True)

    alternate_description = Column(
        'F1894', String(length=30), nullable=True)

    show_priority = Column(
        'F1965', Integer(), nullable=True)

    show_filter = Column(
        'F1966', String(length=20), nullable=True)

    url = Column(
        'F2660', String(length=250), nullable=True)

    def __repr__(self):
        return "Department(code={0})".format(repr(self.code))

    def __str__(self):
        return str(self.description or '')


class Subdepartment(Base):
    """
    Represents a subdepartment.
    """

    __tablename__ = 'SDP_TAB'

    code = Column(
        'F04', Integer(), primary_key=True, nullable=False)

    department_code = Column(
        'F03', Integer(), nullable=True)

    price_margin = Column(
        'F49', Float(), nullable=True)

    require_validation = Column(
        'F78', String(length=1), nullable=True)

    food_stamp = Column(
        'F79', String(length=1), nullable=True)

    trading_stamp = Column(
        'F80', String(length=1), nullable=True)

    tax_flag_1 = Column(
        'F81', String(length=1), nullable=True)

    scalable_item = Column(
        'F82', String(length=1), nullable=True)

    store_coupon = Column(
        'F88', String(length=1), nullable=True)

    tax_flag_2 = Column(
        'F96', String(length=1), nullable=True)

    tax_flag_3 = Column(
        'F97', String(length=1), nullable=True)

    tax_flag_4 = Column(
        'F98', String(length=1), nullable=True)

    tax_flag_5 = Column(
        'F99', String(length=1), nullable=True)

    tax_flag_6 = Column(
        'F100', String(length=1), nullable=True)

    tax_flag_7 = Column(
        'F101', String(length=1), nullable=True)

    vendor_coupon = Column(
        'F104', String(length=1), nullable=True)

    coupon_restricted = Column(
        'F108', String(length=1), nullable=True)

    allow_price_override = Column(
        'F114', String(length=1), nullable=True)

    not_in_net_sale = Column(
        'F115', String(length=1), nullable=True)

    allow_manual_weight = Column(
        'F121', String(length=1), nullable=True)

    prohibit_returns = Column(
        'F124', String(length=1), nullable=True)

    prohibit_discount = Column(
        'F150', String(length=1), nullable=True)

    restriction_code = Column(
        'F170', Integer(), nullable=True)

    minimum_age_customer = Column(
        'F171', Integer(), nullable=True)

    prohibit_refund = Column(
        'F172', String(length=1), nullable=True)

    not_in_admissible_spending = Column(
        'F177', String(length=1), nullable=True)

    wic_eligible = Column(
        'F178', String(length=1), nullable=True)

    maximum_amount = Column(
        'F239', Numeric(precision=8, scale=4), nullable=True)

    minimum_amount = Column(
        'F240', Numeric(precision=8, scale=4), nullable=True)

    maximum_void = Column(
        'F241', Numeric(precision=8, scale=4), nullable=True)

    maximum_refund = Column(
        'F242', Numeric(precision=8, scale=4), nullable=True)

    description = Column(
        'F1022', String(length=30), nullable=True)

    pos_specific_flags = Column(
        'F1120', String(length=12), nullable=True)

    known_shrink_factor = Column(
        'F1123', Float(), nullable=True)

    commission_rate = Column(
        'F1124', Float(), nullable=True)

    minimum_age_operator = Column(
        'F1139', Integer(), nullable=True)

    sequence_number = Column(
        'F1147', Integer(), nullable=True)

    operator_responsible = Column(
        'F1168', Integer(), nullable=True)

    discount = Column(
        'F1256', Float(), nullable=True)

    behavior = Column(
        'F1785', String(length=12), nullable=True)

    alternate_description = Column(
        'F1893', String(length=30), nullable=True)

    cost_plus_percent = Column(
        'F1938', Float(), nullable=True)

    show_priority = Column(
        'F1965', Integer(), nullable=True)

    show_filter = Column(
        'F1966', String(length=20), nullable=True)

    url = Column(
        'F2660', String(length=250), nullable=True)

    def __repr__(self):
        return "Subdepartment(code={0})".format(repr(self.code))


Department.subdepartments = relationship(
    Subdepartment,
    primaryjoin=Subdepartment.department_code == Department.code,
    foreign_keys=[Subdepartment.department_code],
    uselist=True,
    backref='department')


class Product(Base):
    """
    Represents a product (main item record).
    """

    __tablename__ = 'OBJ_TAB'

    upc = Column(
        'F01', String(length=13), primary_key=True, nullable=False)

    family_code = Column(
        'F16', Integer(), nullable=True)

    category_code = Column(
        'F17', Integer(), nullable=True)

    report_code = Column(
        'F18', Integer(), nullable=True)

    account_code = Column(
        'F93', String(length=16), nullable=True)

    reporting_department = Column(
        'F193', String(length=16), nullable=True)

    batch_id = Column(
        'F902', String(length=8), nullable=True)

    record_status = Column(
        'F1001', Integer(), nullable=True)

    upc_code_format = Column(
        'F07', Integer(), nullable=True)

    measurement_system = Column(
        'F11', Integer(), nullable=True)

    size_height = Column(
        'F12', Float(), nullable=True)

    size_width = Column(
        'F13', Float(), nullable=True)

    size_depth = Column(
        'F14', Float(), nullable=True)

    measure_sell_pack = Column(
        'F21', Float(), nullable=True)

    size_description = Column(
        'F22', String(length=30), nullable=True)

    measure_description = Column(
        'F23', String(length=10), nullable=True)

    expanded_description = Column(
        'F29', String(length=60), nullable=True)

    brand_description = Column(
        'F155', String(length=30), nullable=True)

    manufacturer_code = Column(
        'F180', String(length=20), nullable=True)

    duns_number_plus_suffix = Column(
        'F213', String(length=13), nullable=True)

    alias_code = Column(
        'F214', String(length=13), nullable=True)

    alias_code_format = Column(
        'F215', Integer(), nullable=True)

    comparable_size_unit_of_measure_description = Column(
        'F218', String(length=4), nullable=True)

    date_start = Column(
        'F253', DateTime(), nullable=True)

    long_description = Column(
        'F255', String(length=120), nullable=True)

    weight_net = Column(
        'F270', Float(), nullable=True)

    target_id = Column(
        'F1000', String(length=5), nullable=True)

    size_cubic = Column(
        'F1002', Float(), nullable=True)

    container_type = Column(
        'F1004', Integer(), nullable=True)

    manufacturer_id = Column(
        'F1118', String(length=9), nullable=True)

    graphic_file = Column(
        'F1119', String(length=30), nullable=True)

    operator_responsible = Column(
        'F1168', Integer(), nullable=True)

    shipping_piece_count = Column(
        'F1699', Float(), nullable=True)

    best_before_days = Column(
        'F1736', Integer(), nullable=True)

    medical_product_code = Column(
        'F1737', String(length=13), nullable=True)

    ndc_din_number = Column(
        'F1738', String(length=13), nullable=True)

    measure_weight_or_volume = Column(
        'F1744', Float(), nullable=True)

    maintenance_operator_level = Column(
        'F1759', Integer(), nullable=True)

    alternate_brand_description = Column(
        'F1939', String(length=30), nullable=True)

    alternate_expanded_description = Column(
        'F1940', String(length=60), nullable=True)

    alternate_size_description = Column(
        'F1941', String(length=30), nullable=True)

    alternate_long_description = Column(
        'F1942', String(length=120), nullable=True)

    classification = Column(
        'F1957', String(length=100), nullable=True)

    target_customer_type = Column(
        'F1958', String(length=20), nullable=True)

    target_store_type = Column(
        'F1959', String(length=20), nullable=True)

    handling_type = Column(
        'F1960', String(length=10), nullable=True)

    marketing_justification = Column(
        'F1962', String(length=250), nullable=True)

    store_responsible = Column(
        'F1964', String(length=4), nullable=True)

    item_substitution_policy = Column(
        'F2600', String(length=2), nullable=True)

    competitive_code = Column(
        'F2693', String(length=13), nullable=True)

    def __repr__(self):
        return "Product(upc={0})".format(repr(self.upc))

    @property
    def subdepartment(self):
        if 'PAL' in self.pos:
            return self.pos['PAL'].subdepartment
        return None

    @property
    def department(self):
        sub = self.subdepartment
        if sub:
            return sub.department
        return None


class POS(Base):
    """
    Represents various POS flags for a product.
    """

    __tablename__ = 'POS_TAB'

    upc = Column(
        'F01', String(length=13), primary_key=True, nullable=False)

    target_id = Column(
        'F1000', String(length=5), primary_key=True, nullable=False)

    subdepartment_code = Column(
        'F04', Integer(), nullable=True)

    promotion_code = Column(
        'F383', String(length=20), nullable=True)

    batch_id = Column(
        'F902', String(length=8), nullable=True)

    record_status = Column(
        'F1001', Integer(), nullable=True)

    description = Column(
        'F02', String(length=40), nullable=True)

    department_code = Column(
        'F03', Integer(), nullable=True)

    bottle_deposit_link = Column(
        'F05', Integer(), nullable=True)

    tare_weight_link = Column(
        'F06', Integer(), nullable=True)

    upc_code_format = Column(
        'F07', Integer(), nullable=True)

    status_code = Column(
        'F08', String(length=4), nullable=True)

    status_indicator_date = Column(
        'F09', DateTime(), nullable=True)

    weight_divisor = Column(
        'F24', Float(), nullable=True)

    item_pricing_required = Column(
        'F40', String(length=1), nullable=True)

    bottle_deposit_value = Column(
        'F50', Numeric(precision=8, scale=4), nullable=True)

    excise_tax_amount = Column(
        'F60', Numeric(precision=8, scale=4), nullable=True)

    tax_exempt_amount = Column(
        'F61', Numeric(precision=8, scale=4), nullable=True)

    sales_activity_level = Column(
        'F66', Integer(), nullable=True)

    coupon_family_code = Column(
        'F77', Integer(), nullable=True)

    require_validation = Column(
        'F78', String(length=1), nullable=True)

    food_stamp = Column(
        'F79', String(length=1), nullable=True)

    trading_stamp = Column(
        'F80', String(length=1), nullable=True)

    tax_flag_1 = Column(
        'F81', String(length=1), nullable=True)

    scalable_item = Column(
        'F82', String(length=1), nullable=True)

    require_price_entry = Column(
        'F83', String(length=1), nullable=True)

    require_visual_verify = Column(
        'F84', String(length=1), nullable=True)

    require_quantity = Column(
        'F85', String(length=1), nullable=True)

    not_for_sale_in_store = Column(
        'F86', String(length=1), nullable=True)

    restricted_sale = Column(
        'F87', String(length=1), nullable=True)

    store_coupon = Column(
        'F88', String(length=1), nullable=True)

    deposit_container_code = Column(
        'F92', String(length=1), nullable=True)

    tax_flag_2 = Column(
        'F96', String(length=1), nullable=True)

    tax_flag_3 = Column(
        'F97', String(length=1), nullable=True)

    tax_flag_4 = Column(
        'F98', String(length=1), nullable=True)

    tax_flag_5 = Column(
        'F99', String(length=1), nullable=True)

    tax_flag_6 = Column(
        'F100', String(length=1), nullable=True)

    tax_flag_7 = Column(
        'F101', String(length=1), nullable=True)

    prohibit_quantity = Column(
        'F102', String(length=1), nullable=True)

    vendor_coupon = Column(
        'F104', String(length=1), nullable=True)

    follow_subdepartment_status = Column(
        'F106', String(length=1), nullable=True)

    record_item_sale = Column(
        'F107', String(length=1), nullable=True)

    coupon_restricted = Column(
        'F108', String(length=1), nullable=True)

    electronic_coupon = Column(
        'F110', String(length=1), nullable=True)

    allow_price_override = Column(
        'F114', String(length=1), nullable=True)

    not_in_net_sale = Column(
        'F115', String(length=1), nullable=True)

    allow_manual_weight = Column(
        'F121', String(length=1), nullable=True)

    plu_code = Column(
        'F123', String(length=13), nullable=True)

    prohibit_returns = Column(
        'F124', String(length=1), nullable=True)

    prohibit_repeat_key = Column(
        'F125', String(length=1), nullable=True)

    link_quantity_limit = Column(
        'F141', Float(), nullable=True)

    coupon_multiplication = Column(
        'F149', String(length=1), nullable=True)

    prohibit_discount = Column(
        'F150', String(length=1), nullable=True)

    link_reason_code = Column(
        'F153', Integer(), nullable=True)

    deposit_item = Column(
        'F158', String(length=1), nullable=True)

    refund_item = Column(
        'F159', String(length=1), nullable=True)

    bottle_return = Column(
        'F160', String(length=1), nullable=True)

    misc_receipt = Column(
        'F161', String(length=1), nullable=True)

    misc_payout = Column(
        'F162', String(length=1), nullable=True)

    previous_coupon_family_code = Column(
        'F163', Integer(), nullable=True)

    restriction_code = Column(
        'F170', Integer(), nullable=True)

    minimum_age_customer = Column(
        'F171', Integer(), nullable=True)

    prohibit_refund = Column(
        'F172', String(length=1), nullable=True)

    prohibit_multiple_coupon = Column(
        'F173', String(length=1), nullable=True)

    tax_included = Column(
        'F174', String(length=1), nullable=True)

    keyed_department_override = Column(
        'F176', String(length=1), nullable=True)

    wic_eligible = Column(
        'F178', String(length=1), nullable=True)

    valid_item = Column(
        'F188', String(length=1), nullable=True)

    send_to_scale = Column(
        'F189', String(length=1), nullable=True)

    competitive_price_quantity = Column(
        'F209', Float(), nullable=True)

    competitive_price = Column(
        'F210', Numeric(precision=8, scale=4), nullable=True)

    competitive_price_start_date = Column(
        'F211', DateTime(), nullable=True)

    comparable_size = Column(
        'F217', Float(), nullable=True)

    date_start = Column(
        'F253', DateTime(), nullable=True)

    coupon_offer_code = Column(
        'F302', String(length=8), nullable=True)

    coupon_expiration_date = Column(
        'F303', DateTime(), nullable=True)

    coupon_household_id = Column(
        'F304', String(length=8), nullable=True)

    coupon_redemption_multiple = Column(
        'F306', Float(), nullable=True)

    price_multiple_as_quantity = Column(
        'F388', String(length=1), nullable=True)

    prompt_for_sku = Column(
        'F397', String(length=1), nullable=True)

    rentable_quantity = Column(
        'F1099', Integer(), nullable=True)

    specific_flags = Column(
        'F1120', String(length=12), nullable=True)

    commission_rate = Column(
        'F1124', Float(), nullable=True)

    ticket_template = Column(
        'F1136', String(length=8), nullable=True)

    tare_weight_proportional = Column(
        'F1138', Float(), nullable=True)

    minimum_age_operator = Column(
        'F1139', Integer(), nullable=True)

    allow_raincheck_ticket = Column(
        'F1236', String(length=1), nullable=True)

    scale_divisor = Column(
        'F1237', Float(), nullable=True)

    controlled_product_indicator = Column(
        'F1735', String(length=4), nullable=True)

    maintenance_operator_level = Column(
        'F1759', Integer(), nullable=True)

    behavior = Column(
        'F1785', String(length=12), nullable=True)

    replace_adding_function = Column(
        'F1787', Integer(), nullable=True)

    receipt_copy_count = Column(
        'F1788', Integer(), nullable=True)

    store_coupon_count = Column(
        'F1789', Float(), nullable=True)

    alternate_description = Column(
        'F1892', String(length=40), nullable=True)

    store_responsible = Column(
        'F1964', String(length=4), nullable=True)

    url = Column(
        'F2660', String(length=250), nullable=True)

    def __repr__(self):
        return "POS(upc={0}, target_id={1})".format(
            repr(self.upc), repr(self.target_id))


POS.subdepartment = relationship(
    Subdepartment,
    primaryjoin=Subdepartment.code == POS.subdepartment_code,
    foreign_keys=[Subdepartment.code],
    uselist=False)


Product.pos = relationship(
    POS,
    primaryjoin=POS.upc == Product.upc,
    foreign_keys=[POS.upc],
    uselist=True,
    collection_class=attribute_mapped_collection('target_id'),
    backref='product')


class PerpetualInventory(Base):
    """
    Represents the perpetual inventory (running total) for a product.
    """

    __tablename__ = 'RPT_ITM_N'

    upc = Column(
        'F01', String(length=13), primary_key=True, nullable=False)

    totalizer_number = Column(
        'F1034', Integer(), primary_key=True, nullable=False, default=8501)

    store_number = Column(
        'F1056', String(length=4), primary_key=True, nullable=False)

    total_units = Column(
        'F64', Float(), nullable=False, default=0)

    total_dollars = Column(
        'F65', Numeric(precision=8, scale=4), nullable=False, default=0)

    total_weight = Column(
        'F67', Float(), nullable=False, default=0)

    updated = Column(
        'F253', DateTime(), nullable=False)

    def __repr__(self):
        return "PerpetualInventory(upc={0}, store_number={1})".format(
            repr(self.upc), repr(self.store_number))


PerpetualInventory.product = relationship(
    Product,
    primaryjoin=Product.upc == PerpetualInventory.upc,
    foreign_keys=[Product.upc],
    uselist=False)

PerpetualInventory.store = relationship(
    Store,
    primaryjoin=Store.number == PerpetualInventory.store_number,
    foreign_keys=[Store.number],
    uselist=False)


class Operator(Base):
    """
    Represents an user / employee.
    """

    __tablename__ = 'CLK_TAB'

    order_number = Column(
        'F1185', Integer(), primary_key=True, nullable=False)

    record_status = Column(
        'F1001', Integer(), nullable=True)

    number = Column(
        'F1126', Integer(), nullable=True)

    vendor_id = Column(
        'F27', String(length=14), nullable=True)

    restriction_code = Column(
        'F170', Integer(), nullable=True)

    date_start = Column(
        'F253', DateTime(), nullable=True)

    batch_id = Column(
        'F902', String(length=8), nullable=True)

    target_id = Column(
        'F1000', String(length=5), nullable=True)

    store_number = Column(
        'F1056', String(length=4), nullable=True)

    short_name = Column(
        'F1127', String(length=30), nullable=True)

    secret_number = Column(
        'F1141', Integer(), nullable=True)

    restriction_level = Column(
        'F1142', Integer(), nullable=True)

    first_name = Column(
        'F1143', String(length=30), nullable=True)

    last_name = Column(
        'F1144', String(length=30), nullable=True)

    birth_date = Column(
        'F1145', DateTime(), nullable=True)

    language = Column(
        'F1146', String(length=10), nullable=True)

    customer_id = Column(
        'F1148', String(length=14), nullable=True)

    active_on_node = Column(
        'F1176', String(length=3), nullable=True)

    status = Column(
        'F1552', String(length=4), nullable=True)

    hiring_date = Column(
        'F1553', DateTime(), nullable=True)

    break_date = Column(
        'F1554', DateTime(), nullable=True)

    last_promotion_date = Column(
        'F1555', DateTime(), nullable=True)

    superior = Column(
        'F1556', Integer(), nullable=True)

    address_1 = Column(
        'F1557', String(length=40), nullable=True)

    address_2 = Column(
        'F1558', String(length=40), nullable=True)

    city = Column(
        'F1559', String(length=40), nullable=True)

    state = Column(
        'F1560', String(length=20), nullable=True)

    country = Column(
        'F1561', String(length=20), nullable=True)

    zipcode = Column(
        'F1562', String(length=15), nullable=True)

    phone_1 = Column(
        'F1563', String(length=20), nullable=True)

    phone_2 = Column(
        'F1564', String(length=20), nullable=True)

    insurance_number = Column(
        'F1565', String(length=20), nullable=True)

    holidays = Column(
        'F1566', Float(), nullable=True)

    payroll_table_link = Column(
        'F1567', Integer(), nullable=True)

    assigned_drawer_number = Column(
        'F1568', Integer(), nullable=True)

    department_number = Column(
        'F1569', Integer(), nullable=True)

    added_experience = Column(
        'F1570', Float(), nullable=True)

    email = Column(
        'F1571', String(length=60), nullable=True)

    job_description = Column(
        'F1585', String(length=20), nullable=True)

    gender = Column(
        'F1586', String(length=4), nullable=True)

    break_reason = Column(
        'F1587', String(length=20), nullable=True)

    civil_state = Column(
        'F1588', String(length=4), nullable=True)

    employee_number = Column(
        'F1589', String(length=15), nullable=True)

    note = Column(
        'F1590', String(length=80), nullable=True)

    store_responsible = Column(
        'F1964', String(length=4), nullable=True)

    clock_in_out = Column(
        'F2587', DateTime(), nullable=True)

    profit = Column(
        'F2597', String(length=20), nullable=True)

    show_filter = Column(
        'F2692', String(length=1), nullable=True)

    def __repr__(self):
        return "Operator(order_number={0}, number={1})".format(
            repr(self.order_number), repr(self.number))

    def __str__(self):
        return str(self.short_name or '')


class Vendor(Base):
    """
    Represents a vendor.
    """

    __tablename__ = 'VENDOR_TAB'

    id = Column(
        'F27', String(length=14), primary_key=True, nullable=False)

    batch_id = Column(
        'F902', String(length=8), nullable=True)

    record_status = Column(
        'F1001', Integer(), nullable=True)

    date_start = Column(
        'F253', DateTime(), nullable=True)

    name = Column(
        'F334', String(length=40), nullable=True)

    contact_name = Column(
        'F335', String(length=40), nullable=True)

    address_line_1 = Column(
        'F336', String(length=40), nullable=True)

    address_line_2 = Column(
        'F337', String(length=40), nullable=True)

    address_city = Column(
        'F338', String(length=40), nullable=True)

    address_state = Column(
        'F339', String(length=15), nullable=True)

    address_zipcode = Column(
        'F340', String(length=15), nullable=True)

    phone_voice = Column(
        'F341', String(length=20), nullable=True)

    phone_fax = Column(
        'F342', String(length=15), nullable=True)

    product_categories = Column(
        'F343', String(length=30), nullable=True)

    freight_broker_company = Column(
        'F344', String(length=20), nullable=True)

    fob_point = Column(
        'F345', String(length=20), nullable=True)

    limit = Column(
        'F346', Float(), nullable=True)

    special_discount_percent = Column(
        'F347', Float(), nullable=True)

    terms_days = Column(
        'F348', Integer(), nullable=True)

    terms_discount_percent = Column(
        'F349', Float(), nullable=True)

    duns_number = Column(
        'F350', String(length=9), nullable=True)

    supplier_location_number = Column(
        'F351', String(length=6), nullable=True)

    account = Column(
        'F352', String(length=10), nullable=True)

    deliver_monday = Column(
        'F353', String(length=1), nullable=True)

    deliver_tuesday = Column(
        'F354', String(length=1), nullable=True)

    deliver_wednesday = Column(
        'F355', String(length=1), nullable=True)

    deliver_thursday = Column(
        'F356', String(length=1), nullable=True)

    deliver_friday = Column(
        'F357', String(length=1), nullable=True)

    deliver_saturday = Column(
        'F358', String(length=1), nullable=True)

    deliver_sunday = Column(
        'F359', String(length=1), nullable=True)

    target_id = Column(
        'F1000', String(length=5), nullable=True)

    operator_responsible = Column(
        'F1168', Integer(), nullable=True)

    ship_id_number = Column(
        'F1642', String(length=14), nullable=True)

    freight_rate_weight = Column(
        'F1654', Float(), nullable=True)

    freight_rate_volume = Column(
        'F1655', Float(), nullable=True)

    term_description = Column(
        'F1656', String(length=30), nullable=True)

    back_order_days = Column(
        'F1685', Integer(), nullable=True)

    maintenance_operator_level = Column(
        'F1759', Integer(), nullable=True)

    minimum_cases = Column(
        'F1760', Float(), nullable=True)

    minimum_cubic = Column(
        'F1761', Float(), nullable=True)

    internal_comment = Column(
        'F1779', String(length=120), nullable=True)

    delivery_days = Column(
        'F1793', Integer(), nullable=True)

    delivery_cycle_days = Column(
        'F1794', Integer(), nullable=True)

    comment = Column(
        'F1882', String(length=120), nullable=True)

    minimum_weight = Column(
        'F1883', Float(), nullable=True)

    minimum_dollars = Column(
        'F1884', Numeric(precision=8, scale=4), nullable=True)

    buying_format = Column(
        'F1887', String(length=2), nullable=True)

    contact_2 = Column(
        'F1889', String(length=40), nullable=True)

    contact_2_phone = Column(
        'F1890', String(length=20), nullable=True)

    contact_2_fax = Column(
        'F1891', String(length=15), nullable=True)

    country = Column(
        'F1948', String(length=30), nullable=True)

    status = Column(
        'F1949', String(length=6), nullable=True)

    store_responsible = Column(
        'F1964', String(length=4), nullable=True)

    show_priority = Column(
        'F1965', Integer(), nullable=True)

    show_filter = Column(
        'F1966', String(length=20), nullable=True)

    profit = Column(
        'F2597', String(length=20), nullable=True)

    currency_code = Column(
        'F2602', String(length=3), nullable=True)

    email = Column(
        'F2603', String(length=60), nullable=True)

    invisible_on_pda = Column(
        'F2658', String(length=1), nullable=True)

    url = Column(
        'F2660', String(length=250), nullable=True)

    default_quantity = Column(
        'F2666', Float(), nullable=True)

    maximum_cases = Column(
        'F2700', Float(), nullable=True)

    maximum_cubic = Column(
        'F2701', Float(), nullable=True)

    maximum_weight = Column(
        'F2702', Float(), nullable=True)

    associated_tax_rates = Column(
        'F2710', String(length=10), nullable=True)

    def __repr__(self):
        return "Vendor(id={0})".format(repr(self.id))

    def __str__(self):
        return str(self.name or '')
    

class ReceivingDocument(Base):
    """
    Represents a receiving document in any of its phases.

    Although this model is named ``ReceivingDocument``, it can represent any of
    the following:

    * purchasing document
    * receiving document
    * billing document
    * direct delivery document
    """

    __tablename__ = 'REC_HDR'

    transaction_number = Column(
        'F1032', Integer(), primary_key=True, nullable=False)

    vendor_id = Column(
        'F27', String(length=14), nullable=True)

    order_date = Column(
        'F76', DateTime(), nullable=True)

    purchase_order_number = Column(
        'F91', String(length=8), nullable=True)

    date_start = Column(
        'F253', DateTime(), nullable=True)

    date_end = Column(
        'F254', DateTime(), nullable=True)

    vendor_name = Column(
        'F334', String(length=40), nullable=True)

    vendor_contact_name = Column(
        'F335', String(length=40), nullable=True)

    vendor_address_line_1 = Column(
        'F336', String(length=40), nullable=True)

    vendor_address_line_2 = Column(
        'F337', String(length=40), nullable=True)

    vendor_address_city = Column(
        'F338', String(length=40), nullable=True)

    vendor_address_state = Column(
        'F339', String(length=15), nullable=True)

    vendor_address_zipcode = Column(
        'F340', String(length=15), nullable=True)

    vendor_phone_voice = Column(
        'F341', String(length=20), nullable=True)

    vendor_phone_fax = Column(
        'F342', String(length=15), nullable=True)

    vendor_product_categories = Column(
        'F343', String(length=30), nullable=True)

    freight_broker_company = Column(
        'F344', String(length=20), nullable=True)

    fob_point = Column(
        'F345', String(length=20), nullable=True)

    vendor_limit = Column(
        'F346', Float(), nullable=True)

    special_discount_percent = Column(
        'F347', Float(), nullable=True)

    terms_days = Column(
        'F348', Integer(), nullable=True)

    terms_discount_percent = Column(
        'F349', Float(), nullable=True)

    vendor_account = Column(
        'F352', String(length=10), nullable=True)

    batch_id = Column(
        'F902', String(length=8), nullable=True)

    movement_start_time = Column(
        'F1035', String(length=6), nullable=True)

    movement_end_time = Column(
        'F1036', String(length=6), nullable=True)

    store_number = Column(
        'F1056', String(length=4), nullable=True)

    terminal_number = Column(
        'F1057', String(length=4), nullable=True)

    registration_mode = Column(
        'F1067', String(length=7), nullable=True)

    transaction_mode = Column(
        'F1068', String(length=6), nullable=True)

    line_number = Column(
        'F1101', Integer(), nullable=True)

    operator_number = Column(
        'F1126', Integer(), nullable=True)

    operator_short_name = Column(
        'F1127', String(length=30), nullable=True)

    customer_id = Column(
        'F1148', String(length=14), nullable=True)

    operator_responsible = Column(
        'F1168', Integer(), nullable=True)

    reference_number = Column(
        'F1245', String(length=20), nullable=True)

    delivery_date = Column(
        'F1246', DateTime(), nullable=True)

    transaction_note = Column(
        'F1254', String(length=500), nullable=True)

    batch_file_name = Column(
        'F1255', String(length=40), nullable=True)

    creation_date = Column(
        'F1264', DateTime(), nullable=True)

    ship_id_number = Column(
        'F1642', String(length=14), nullable=True)

    ship_name = Column(
        'F1643', String(length=40), nullable=True)

    ship_contact_name = Column(
        'F1644', String(length=40), nullable=True)

    ship_address_line_1 = Column(
        'F1645', String(length=40), nullable=True)

    ship_address_line_2 = Column(
        'F1646', String(length=40), nullable=True)

    ship_address_city = Column(
        'F1647', String(length=40), nullable=True)

    ship_address_state = Column(
        'F1648', String(length=15), nullable=True)

    ship_address_zipcode = Column(
        'F1649', String(length=15), nullable=True)

    ship_phone_voice = Column(
        'F1650', String(length=15), nullable=True)

    ship_phone_fax = Column(
        'F1651', String(length=15), nullable=True)

    ship_product_categories = Column(
        'F1652', String(length=30), nullable=True)

    shipping_date = Column(
        'F1653', DateTime(), nullable=True)

    freight_rate_weight = Column(
        'F1654', Float(), nullable=True)

    freight_rate_volume = Column(
        'F1655', Float(), nullable=True)

    vendor_term_description = Column(
        'F1656', String(length=30), nullable=True)

    back_order_days = Column(
        'F1685', Integer(), nullable=True)

    operator_order_printed = Column(
        'F1686', Integer(), nullable=True)

    operator_receipt_printed = Column(
        'F1687', Integer(), nullable=True)

    operator_invoice_printed = Column(
        'F1688', Integer(), nullable=True)

    operator_export_made = Column(
        'F1689', Integer(), nullable=True)

    shipping_note = Column(
        'F1692', String(length=80), nullable=True)

    operator_order_entered_id = Column(
        'F1693', Integer(), nullable=True)

    operator_receipt_confirm = Column(
        'F1694', Integer(), nullable=True)

    operator_invoice_entered = Column(
        'F1695', Integer(), nullable=True)

    operator_trucker = Column(
        'F1696', Integer(), nullable=True)

    minimum_cases = Column(
        'F1760', Float(), nullable=True)

    minimum_cubic = Column(
        'F1761', Float(), nullable=True)

    delivery_days = Column(
        'F1793', Integer(), nullable=True)

    minimum_weight = Column(
        'F1883', Float(), nullable=True)

    minimum_dollars = Column(
        'F1884', Numeric(precision=8, scale=4), nullable=True)

    buying_format = Column(
        'F1887', String(length=2), nullable=True)

    vendor_contact_2 = Column(
        'F1889', String(length=40), nullable=True)

    vendor_contact_2_phone = Column(
        'F1890', String(length=20), nullable=True)

    date_payable = Column(
        'F2596', DateTime(), nullable=True)

    currency_exchange_now = Column(
        'F2598', Float(), nullable=True)

    currency_exchange_order = Column(
        'F2599', Float(), nullable=True)

    operator_pickup_printed = Column(
        'F2632', Integer(), nullable=True)

    order_time = Column(
        'F2655', String(length=6), nullable=True)

    delivery_time = Column(
        'F2656', String(length=6), nullable=True)

    event_date = Column(
        'F2657', DateTime(), nullable=True)

    invisible_on_pda = Column(
        'F2658', String(length=1), nullable=True)

    document_subtype = Column(
        'F2659', String(length=6), nullable=True)

    associated_tax_rates = Column(
        'F2710', String(length=10), nullable=True)

    def __repr__(self):
        return "ReceivingDocument(transaction_number={0})".format(
            repr(self.transaction_number))


ReceivingDocument.vendor = relationship(
    Vendor,
    primaryjoin=Vendor.id == ReceivingDocument.vendor_id,
    foreign_keys=[Vendor.id],
    uselist=False)

ReceivingDocument.operator_order_entered = relationship(
    Operator,
    primaryjoin=Operator.number == ReceivingDocument.operator_order_entered_id,
    foreign_keys=[Operator.number],
    uselist=False)
