#!/usr/bin/env python
# -*- coding: utf-8  -*-
################################################################################
#
#  pyLOCSMS -- Python Interface to LOC Store Management Suite
#  Copyright © 2013-2014 Lance Edgar
#
#  This file is part of pyLOCSMS.
#
#  pyLOCSMS is free software: you can redistribute it and/or modify it under
#  the terms of the GNU General Public License as published by the Free
#  Software Foundation, either version 3 of the License, or (at your option)
#  any later version.
#
#  pyLOCSMS is distributed in the hope that it will be useful, but WITHOUT ANY
#  WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
#  FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
#  details.
#
#  You should have received a copy of the GNU General Public License along with
#  pyLOCSMS.  If not, see <http://www.gnu.org/licenses/>.
#
################################################################################

"""
Database Utilities
"""

import sys
import argparse
import csv

from sqlalchemy import create_engine, MetaData, Table, Integer

from ..fields import get_field_map


def generate_model(engine, fields_csv_path, table_name, output):
    """
    Generate a SQLAlchemy data model for a table.

    This function will connect to the SMS database, inspect the specified table
    to obtain its schema, borrow some extra information from a fields CSV file,
    and finally write some Python code to the specified output buffer.

    Note that the Python code generated by this function is not meant to be a
    "final draft" per se, but a reasonable starting point which requires
    further manual refinement.  This was the approach used to generate the bulk
    of the classes found in the :mod:`locsms.db.model` module.
    """
    field_map = get_field_map(fields_csv_path)

    def column_type(column):
        field = field_map.get(column.name)
        if field:
            if field[u'TYPE'] == u'C':
                return u'String(length={0})'.format(field[u'SIZE'])
            if field[u'TYPE'] == u'D':
                return u'Date()'
            if field[u'TYPE'] == u'F':
                return u'String(length=1)'
            if field[u'TYPE'] == u'I':
                return u'Integer()'
            if field[u'TYPE'] == u'M':
                return u'Numeric(precision=8, scale=4)'
            if field[u'TYPE'] == u'N':
                return u'Float()'
            if field[u'TYPE'] == u'T':
                return u'String(length=6)'
        # Use reflected type as fallback.
        return unicode(repr(column.type))

    def column_primary_key(column):
        if column.primary_key:
            if isinstance(column.type, Integer):
                return u', primary_key=True, autoincrement=False'
            return u', primary_key=True'
        return u''

    def column_nullable(column):
        return u', nullable={0}'.format(column.nullable)

    def column_specification(column):
        return column_type(column) + column_primary_key(column) + column_nullable(column)

    def column_doc(column):
        field = field_map.get(column.name)
        if field:
            return field[u'DESCRIPTOR'].strip()
        return u''

    metadata = MetaData(bind=engine)
    table = Table(table_name, metadata, autoload=True)
    output.write(u'class {0}(Base):\n'.format(table_name))
    output.write(u'\n    __tablename__ = u\'{0}\'\n'.format(table_name))
    for column in table.columns:
        output.write(u'\n    {0} = Column(\n'.format(column.name))
        output.write(u'        {0},\n'.format(column_specification(column)))
        output.write(u'        doc=u"{0}")\n'.format(column_doc(column)))


def generate_model_command(*args):
    """
    Command to generate a SQLAlchemy data model for a table.

    This provides a command-line interface to the :func:`generate_model()`
    function.  It is invoked as ``locsms-generate-model``:

    .. program-output:: locsms-generate-model --help
    """
    parser = argparse.ArgumentParser(prog=u'locsms-generate-model', description=u"""
Generate a module containing a SQLAlchemy data model class for a given table.
""")
    parser.add_argument(u'engine_url',
                        help=u"SQLAlchemy database engine URL.")
    parser.add_argument(u'fields_csv_path',
                        help=u"Path to the SMS 'Fields.csv' file.")
    parser.add_argument(u'table_name',
                        help=u"Internal name of the table (e.g. 'OBJ_TAB').")
    parser.add_argument(u'--output', metavar=u'PATH', type=argparse.FileType(u'w'),
                        help=u"File to which output should be written (defaults to STDOUT).")
    args = parser.parse_args(args or sys.argv[1:])

    engine = create_engine(args.engine_url)
    generate_model(engine, args.fields_csv_path, args.table_name, args.output or sys.stdout)
