#!/usr/bin/env python
# -*- coding: utf-8 -*-
################################################################################
#
#  pyLOCSMS -- Python Interface to LOC Store Management Suite
#  Copyright © 2013-2014 Lance Edgar
#
#  This file is part of pyLOCSMS.
#
#  pyLOCSMS is free software: you can redistribute it and/or modify it under
#  the terms of the GNU General Public License as published by the Free
#  Software Foundation, either version 3 of the License, or (at your option)
#  any later version.
#
#  pyLOCSMS is distributed in the hope that it will be useful, but WITHOUT ANY
#  WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
#  FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
#  details.
#
#  You should have received a copy of the GNU General Public License along with
#  pyLOCSMS.  If not, see <http://www.gnu.org/licenses/>.
#
################################################################################

"""
Product Data API
"""

from sqlalchemy import and_

from locsms.db import model


__all__ = [u'last_purchase_base_cost']


def last_purchase_base_cost(session, F01, F27=None):
    """
    Fetch the last-paid base cost for a product, if available.

    Returns the most recent :attr:`locsms.db.model.PurchaseDetail.F38` value
    found in the product's purchase history.  If ``F27`` is specified, the
    search will be restricted to purchases for that vendor only.

    :param session: Reference to a ``Session`` instance.

    :param F01: UPC for the product.

    :param F27: Optional vendor code, used to filter the query against
       :attr:`locsms.db.model.Purchase.F27`.

    :returns: The last-paid base cost as a ``Decimal``, or ``None``.
    """
    query = session.query(model.PurchaseDetail.F38)\
        .join((model.Purchase, model.Purchase.F1032 == model.PurchaseDetail.F1032))\
        .filter(and_(
            model.Purchase.F1068.in_((u'ORDER', u'RECV', u'BILL', u'DSD')),
            model.PurchaseDetail.F01 == F01,
            model.PurchaseDetail.F38 != None))
    if F27:
        query = query.filter(model.Purchase.F27 == F27)
    query = query.order_by(model.PurchaseDetail.F76.desc())
    detail = query.first()
    if detail:
        return detail.F38
    return None
